import axios from 'axios';
import io from 'socket.io-client';

// Use environment variable for production, fallback to localhost for development
const API_BASE_URL = process.env.NEXT_PUBLIC_API_URL
  ? `${process.env.NEXT_PUBLIC_API_URL}/api`
  : 'http://localhost:3002/api';
const SOCKET_URL = process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002';

// Axios instance
const api = axios.create({
  baseURL: API_BASE_URL,
  timeout: 30000,
  headers: {
    'Content-Type': 'application/json',
  },
});

// Socket.io instance
export const socket = io(SOCKET_URL, {
  autoConnect: false,
});

// API Types
export interface CheckinRequest {
  pnr: string;
  lastName: string;
  firstName?: string;
  phone?: string;
}

export interface CheckinResponse {
  success: boolean;
  message: string;
  jobId: string;
  status: 'processing' | 'completed' | 'failed';
}

export interface CheckinResult {
  jobId: string;
  success: boolean;
  message: string;
  flightInfo?: {
    flight: string;
    from: string;
    to: string;
    date: string;
    seat?: string;
  };
  boardingPassUrl?: string;
  screenshots?: Array<{
    filename: string;
    url: string;
  }>;
}

export interface Customer {
  id: string;
  firstName: string;
  lastName: string;
  email?: string;
  phone: string;
  notes?: string;
  isVip: boolean;
  createdAt: Date;
  updatedAt: Date;
}

// Check-in API
export const checkinApi = {
  // Get all check-in tickets
  async getCheckinTickets(params?: { status?: string; airline?: string; search?: string }) {
    const response = await api.get('/checkin-tickets', { params });
    return response.data;
  },

  // Get single check-in ticket
  async getCheckinTicket(id: string) {
    const response = await api.get(`/checkin-tickets/${id}`);
    return response.data;
  },

  // Create check-in ticket
  async createCheckinTicket(data: unknown) {
    const response = await api.post('/checkin-tickets', data);
    return response.data;
  },

  // Update check-in ticket
  async updateCheckinTicket(id: string, data: unknown) {
    const response = await api.put(`/checkin-tickets/${id}`, data);
    return response.data;
  },

  // Delete check-in ticket
  async deleteCheckinTicket(id: string) {
    const response = await api.delete(`/checkin-tickets/${id}`);
    return response.data;
  },

  // Get upcoming check-ins
  async getUpcomingCheckins() {
    const response = await api.get('/checkin-tickets/upcoming');
    return response.data;
  },

  // Tek check-in işlemi başlat
  async startCheckin(data: CheckinRequest): Promise<CheckinResponse> {
    const response = await api.post('/checkin', data);
    return response.data;
  },

  // Toplu check-in işlemi başlat
  async startBulkCheckin(checkinList: CheckinRequest[]) {
    const response = await api.post('/checkin/bulk', { checkinList });
    return response.data;
  },

  // Job durumunu sorgula
  async getJobStatus(jobId: string) {
    const response = await api.get(`/checkin/status/${jobId}`);
    return response.data;
  },

  // Job'ı iptal et
  async cancelJob(jobId: string) {
    const response = await api.delete(`/checkin/${jobId}`);
    return response.data;
  },
};

// Customer API
export const customerApi = {
  // Tüm müşterileri listele
  async getCustomers(params?: { search?: string; vip?: boolean }) {
    const response = await api.get('/customers', { params });
    return response.data;
  },

  // Müşteri detayını getir
  async getCustomer(id: string): Promise<Customer> {
    const response = await api.get(`/customers/${id}`);
    return response.data;
  },

  // Yeni müşteri ekle
  async createCustomer(data: Omit<Customer, 'id' | 'createdAt' | 'updatedAt'>): Promise<Customer> {
    const response = await api.post('/customers', data);
    return response.data;
  },

  // Müşteri güncelle
  async updateCustomer(id: string, data: Partial<Customer>): Promise<Customer> {
    const response = await api.put(`/customers/${id}`, data);
    return response.data;
  },

  // Müşteri sil
  async deleteCustomer(id: string) {
    const response = await api.delete(`/customers/${id}`);
    return response.data;
  },

  // Müşteri check-in geçmişi
  async getCustomerHistory(id: string) {
    const response = await api.get(`/customers/${id}/history`);
    return response.data;
  },
};

// System API
export const systemApi = {
  // Health check
  async healthCheck() {
    const response = await api.get('/health');
    return response.data;
  },

  // Get dashboard statistics
  async getDashboardStats() {
    const response = await api.get('/reports/dashboard');
    return response.data;
  },
};

// Socket Event Types
export interface SocketEvents {
  checkin_status: {
    jobId: string;
    status: 'processing' | 'completed' | 'failed';
    message: string;
    progress: number;
  };

  checkin_result: CheckinResult;

  bulk_checkin_progress: {
    batchId: string;
    completedCount: number;
    totalCount: number;
    progress: number;
    lastResult: unknown;
  };

  bulk_checkin_complete: {
    batchId: string;
    results: unknown[];
    summary: {
      total: number;
      successful: number;
      failed: number;
    };
  };
}

// Socket connection helpers
export const connectSocket = () => {
  if (!socket.connected) {
    socket.connect();
  }
};

export const disconnectSocket = () => {
  if (socket.connected) {
    socket.disconnect();
  }
};

// Error interceptors
api.interceptors.response.use(
  (response) => response,
  (error) => {
    console.error('API Error:', error);
    
    if (error.response?.status === 404) {
      throw new Error('API endpoint bulunamadı');
    } else if (error.response?.status >= 500) {
      throw new Error('Sunucu hatası, lütfen daha sonra tekrar deneyin');
    } else if (error.code === 'ECONNREFUSED') {
      throw new Error('API sunucusuna bağlanılamadı');
    }
    
    throw error;
  }
);

// Ticket API
export const ticketApi = {
  // Get all tickets with enriched flight info
  async getTickets(params?: { enrichWithFlightInfo?: boolean; search?: string; status?: string }) {
    const response = await api.get('/tickets', { params });
    return response.data;
  },

  // Get single ticket
  async getTicket(id: string) {
    const response = await api.get(`/tickets/${id}`);
    return response.data;
  },

  // Create new ticket
  async createTicket(data: unknown) {
    const response = await api.post('/tickets', data);
    return response.data;
  },

  // Update ticket
  async updateTicket(id: string, data: unknown) {
    const response = await api.put(`/tickets/${id}`, data);
    return response.data;
  },

  // Delete ticket
  async deleteTicket(id: string) {
    const response = await api.delete(`/tickets/${id}`);
    return response.data;
  },

  // Search airports from Amadeus
  async searchAirports(keyword: string) {
    const response = await api.get(`/tickets/amadeus/airports/${encodeURIComponent(keyword)}`);
    return response.data;
  },

  // Search flights from Amadeus
  async searchFlights(searchParams: unknown) {
    const response = await api.post('/tickets/search-flights', searchParams);
    return response.data;
  },

  // Create ticket from Amadeus flight offer
  async createFromAmadeus(flightData: unknown) {
    const response = await api.post('/tickets/create-from-amadeus', flightData);
    return response.data;
  },

  // Enrich existing tickets with airport info
  async enrichAirports(params?: { ticketIds?: string[] }) {
    const response = await api.post('/tickets/enrich-airports', params);
    return response.data;
  }
};

// Department API
export const departmentApi = {
  // Get all departments
  async getDepartments() {
    const response = await api.get('/departments');
    return response.data;
  },

  // Get single department
  async getDepartment(id: string) {
    const response = await api.get(`/departments/${id}`);
    return response.data;
  },

  // Create new department
  async createDepartment(data: unknown) {
    const response = await api.post('/departments', data);
    return response.data;
  },

  // Update department
  async updateDepartment(id: string, data: unknown) {
    const response = await api.put(`/departments/${id}`, data);
    return response.data;
  },

  // Delete department
  async deleteDepartment(id: string) {
    const response = await api.delete(`/departments/${id}`);
    return response.data;
  },

  // Get department statistics
  async getDepartmentStats() {
    const response = await api.get('/departments/stats/overview');
    return response.data;
  }
};

// Agency User API
export const agencyUserApi = {
  // Get all agency users
  async getAgencyUsers(params?: { department?: string; role?: string; search?: string }) {
    const response = await api.get('/agency-users', { params });
    return response.data;
  },

  // Get single agency user
  async getAgencyUser(id: string) {
    const response = await api.get(`/agency-users/${id}`);
    return response.data;
  },

  // Create new agency user
  async createAgencyUser(data: unknown) {
    const response = await api.post('/agency-users', data);
    return response.data;
  },

  // Update agency user
  async updateAgencyUser(id: string, data: unknown) {
    const response = await api.put(`/agency-users/${id}`, data);
    return response.data;
  },

  // Deactivate agency user (soft delete)
  async deactivateAgencyUser(id: string) {
    const response = await api.patch(`/agency-users/${id}/deactivate`);
    return response.data;
  },

  // Permanently delete agency user (hard delete)
  async deleteAgencyUser(id: string) {
    const response = await api.delete(`/agency-users/${id}/permanent`);
    return response.data;
  },

  // Get agency user statistics
  async getAgencyUserStats() {
    const response = await api.get('/agency-users/stats/overview');
    return response.data;
  }
};

export default api;