'use client';

import React, { useState } from 'react';
import {
  Car,
  Check,
  Clock,
  Globe,
  Hotel,
  MapPin,
  Plane,
  Plus,
  Star
} from 'lucide-react';

interface FlightSuggestion {
  id: string;
  airline: string;
  flightNumber: string;
  departure: string;
  arrival: string;
  departureTime: string;
  arrivalTime: string;
  price: number;
  currency: string;
  duration: string;
  stops: number;
}

interface HotelSuggestion {
  id: string;
  name: string;
  location: string;
  rating: number;
  price: number;
  currency: string;
  priceType: 'per_night' | 'total';
  amenities: string[];
  distance: string;
  imageUrl?: string;
}

interface TransferSuggestion {
  id: string;
  type: 'economy' | 'comfort' | 'luxury' | 'minibus';
  provider: string;
  price: number;
  currency: string;
  duration: string;
  features: string[];
  vehicleInfo: string;
}

interface VisaSuggestion {
  id: string;
  type: string;
  processingTime: string;
  price: number;
  currency: string;
  requirements: string[];
  success_rate: number;
}

interface SuggestionsPanelProps {
  type: 'flight' | 'hotel' | 'transfer' | 'visa';
  suggestions: (FlightSuggestion | HotelSuggestion | TransferSuggestion | VisaSuggestion)[];
  onSelectSuggestion: (suggestion: FlightSuggestion | HotelSuggestion | TransferSuggestion | VisaSuggestion) => void;
  onAddCustom?: () => void;
  loading?: boolean;
}

export default function SuggestionsPanel({ 
  type, 
  suggestions, 
  onSelectSuggestion, 
  onAddCustom,
  loading = false 
}: SuggestionsPanelProps) {
  const [selectedSuggestion, setSelectedSuggestion] = useState<string | null>(null);

  const getIcon = () => {
    switch (type) {
      case 'flight': return <Plane className="h-5 w-5" />;
      case 'hotel': return <Hotel className="h-5 w-5" />;
      case 'transfer': return <Car className="h-5 w-5" />;
      case 'visa': return <Globe className="h-5 w-5" />;
    }
  };

  const getTitle = () => {
    switch (type) {
      case 'flight': return 'Uçuş Önerileri';
      case 'hotel': return 'Konaklama Önerileri';
      case 'transfer': return 'Transfer Önerileri';
      case 'visa': return 'Vize Seçenekleri';
    }
  };

  const handleSelect = (suggestion: FlightSuggestion | HotelSuggestion | TransferSuggestion | VisaSuggestion) => {
    setSelectedSuggestion(suggestion.id);
    onSelectSuggestion(suggestion);
  };

  const renderFlightSuggestion = (flight: FlightSuggestion) => (
    <div key={flight.id} className="border border-gray-200 rounded-lg p-4 hover:border-blue-300 transition-colors">
      <div className="flex items-center justify-between mb-3">
        <div className="flex items-center space-x-2">
          <Plane className="h-4 w-4 text-blue-600" />
          <span className="font-medium">{flight.airline} {flight.flightNumber}</span>
        </div>
        <div className="text-right">
          <p className="font-bold text-lg text-green-600">{flight.price.toLocaleString()} {flight.currency}</p>
        </div>
      </div>
      
      <div className="grid grid-cols-2 gap-4 mb-3">
        <div>
          <p className="text-sm text-gray-500">Kalkış</p>
          <p className="font-medium">{flight.departure}</p>
          <p className="text-sm text-gray-600">{flight.departureTime}</p>
        </div>
        <div>
          <p className="text-sm text-gray-500">Varış</p>
          <p className="font-medium">{flight.arrival}</p>
          <p className="text-sm text-gray-600">{flight.arrivalTime}</p>
        </div>
      </div>
      
      <div className="flex items-center justify-between text-sm text-gray-600">
        <span>Süre: {flight.duration}</span>
        <span>{flight.stops === 0 ? 'Direkt' : `${flight.stops} aktarma`}</span>
      </div>
      
      <button
        onClick={() => handleSelect(flight)}
        className="w-full mt-3 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
      >
        Bu Uçuşu Seç
      </button>
    </div>
  );

  const renderHotelSuggestion = (hotel: HotelSuggestion) => (
    <div key={hotel.id} className="border border-gray-200 rounded-lg p-4 hover:border-blue-300 transition-colors">
      <div className="flex items-center justify-between mb-3">
        <div>
          <h4 className="font-medium text-lg">{hotel.name}</h4>
          <div className="flex items-center space-x-2">
            <MapPin className="h-4 w-4 text-gray-400" />
            <span className="text-sm text-gray-600">{hotel.location}</span>
          </div>
        </div>
        <div className="text-right">
          <p className="font-bold text-lg text-green-600">
            {hotel.price.toLocaleString()} {hotel.currency}
          </p>
          <p className="text-xs text-gray-500">
            {hotel.priceType === 'per_night' ? 'gecelik' : 'toplam'}
          </p>
        </div>
      </div>
      
      <div className="flex items-center space-x-4 mb-3">
        <div className="flex items-center">
          {[...Array(5)].map((_, i) => (
            <Star
              key={i}
              className={`h-4 w-4 ${i < hotel.rating ? 'text-yellow-400 fill-current' : 'text-gray-300'}`}
            />
          ))}
        </div>
        <span className="text-sm text-gray-600">{hotel.distance}</span>
      </div>
      
      {hotel.amenities.length > 0 && (
        <div className="mb-3">
          <div className="flex flex-wrap gap-1">
            {hotel.amenities.slice(0, 3).map((amenity, index) => (
              <span key={index} className="px-2 py-1 text-xs bg-gray-100 text-gray-600 rounded">
                {amenity}
              </span>
            ))}
            {hotel.amenities.length > 3 && (
              <span className="px-2 py-1 text-xs bg-gray-100 text-gray-600 rounded">
                +{hotel.amenities.length - 3} daha
              </span>
            )}
          </div>
        </div>
      )}
      
      <button
        onClick={() => handleSelect(hotel)}
        className="w-full mt-3 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
      >
        Bu Oteli Seç
      </button>
    </div>
  );

  const renderTransferSuggestion = (transfer: TransferSuggestion) => (
    <div key={transfer.id} className="border border-gray-200 rounded-lg p-4 hover:border-blue-300 transition-colors">
      <div className="flex items-center justify-between mb-3">
        <div>
          <h4 className="font-medium">{transfer.provider}</h4>
          <p className="text-sm text-gray-600">{transfer.vehicleInfo}</p>
        </div>
        <div className="text-right">
          <p className="font-bold text-lg text-green-600">
            {transfer.price.toLocaleString()} {transfer.currency}
          </p>
        </div>
      </div>
      
      <div className="flex items-center space-x-4 mb-3">
        <div className="flex items-center space-x-1">
          <Clock className="h-4 w-4 text-gray-400" />
          <span className="text-sm text-gray-600">{transfer.duration}</span>
        </div>
        <span className="px-2 py-1 text-xs bg-blue-100 text-blue-800 rounded">
          {transfer.type}
        </span>
      </div>
      
      {transfer.features.length > 0 && (
        <div className="mb-3">
          <div className="flex flex-wrap gap-1">
            {transfer.features.map((feature, index) => (
              <span key={index} className="px-2 py-1 text-xs bg-gray-100 text-gray-600 rounded">
                {feature}
              </span>
            ))}
          </div>
        </div>
      )}
      
      <button
        onClick={() => handleSelect(transfer)}
        className="w-full mt-3 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
      >
        Bu Transferi Seç
      </button>
    </div>
  );

  const renderVisaSuggestion = (visa: VisaSuggestion) => (
    <div key={visa.id} className="border border-gray-200 rounded-lg p-4 hover:border-blue-300 transition-colors">
      <div className="flex items-center justify-between mb-3">
        <div>
          <h4 className="font-medium">{visa.type}</h4>
          <p className="text-sm text-gray-600">İşlem Süresi: {visa.processingTime}</p>
        </div>
        <div className="text-right">
          <p className="font-bold text-lg text-green-600">
            {visa.price.toLocaleString()} {visa.currency}
          </p>
          <p className="text-xs text-green-600">
            %{visa.success_rate} başarı oranı
          </p>
        </div>
      </div>
      
      <div className="mb-3">
        <p className="text-sm text-gray-600 mb-2">Gerekli Belgeler:</p>
        <div className="space-y-1">
          {visa.requirements.slice(0, 3).map((req, index) => (
            <div key={index} className="flex items-center space-x-2">
              <Check className="h-3 w-3 text-green-600" />
              <span className="text-xs text-gray-600">{req}</span>
            </div>
          ))}
          {visa.requirements.length > 3 && (
            <p className="text-xs text-gray-500">+{visa.requirements.length - 3} belge daha</p>
          )}
        </div>
      </div>
      
      <button
        onClick={() => handleSelect(visa)}
        className="w-full mt-3 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
      >
        Bu Vize Türünü Seç
      </button>
    </div>
  );

  const renderSuggestion = (suggestion: FlightSuggestion | HotelSuggestion | TransferSuggestion | VisaSuggestion) => {
    switch (type) {
      case 'flight': return renderFlightSuggestion(suggestion as FlightSuggestion);
      case 'hotel': return renderHotelSuggestion(suggestion as HotelSuggestion);
      case 'transfer': return renderTransferSuggestion(suggestion as TransferSuggestion);
      case 'visa': return renderVisaSuggestion(suggestion as VisaSuggestion);
    }
  };

  if (loading) {
    return (
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <div className="animate-pulse space-y-4">
          <div className="h-6 bg-gray-200 rounded w-1/3"></div>
          <div className="space-y-3">
            {[1, 2, 3].map(i => (
              <div key={i} className="h-24 bg-gray-200 rounded"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
      <div className="flex items-center justify-between mb-4">
        <h3 className="text-lg font-semibold text-gray-900 flex items-center">
          {getIcon()}
          <span className="ml-2">{getTitle()}</span>
        </h3>
        {onAddCustom && (
          <button
            onClick={onAddCustom}
            className="flex items-center px-3 py-2 text-sm text-blue-600 border border-blue-300 rounded-lg hover:bg-blue-50"
          >
            <Plus className="h-4 w-4 mr-1" />
            Özel Ekle
          </button>
        )}
      </div>

      {suggestions.length === 0 ? (
        <div className="text-center py-8">
          <div className="text-gray-400 mb-2">
            {getIcon()}
          </div>
          <p className="text-sm text-gray-500">Henüz öneri bulunamadı</p>
          {onAddCustom && (
            <button
              onClick={onAddCustom}
              className="mt-3 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
            >
              Manuel Ekle
            </button>
          )}
        </div>
      ) : (
        <div className="space-y-4 max-h-96 overflow-y-auto">
          {suggestions.map(renderSuggestion)}
        </div>
      )}
    </div>
  );
}