'use client';

import React from 'react';
import { ArrowLeft } from 'lucide-react';
import { useRouter } from 'next/navigation';
import ChatPanel, { type ChatMessage } from './ChatPanel';

// Legacy Comment interface - ChatMessage kullanılacak
interface Comment {
  id: string;
  author: string;
  content: string;
  timestamp: Date;
  type: 'comment' | 'system' | 'status_change';
  isInternal?: boolean;
}

interface DetailLayoutProps {
  title: string;
  subtitle?: React.ReactNode;
  backUrl: string;
  children: React.ReactNode;
  recordId: string;
  recordType: 'ticket' | 'accommodation' | 'visa' | 'transfer';
  
  // Chat props
  messages?: ChatMessage[];
  onSendMessage?: (content: string, isInternal: boolean) => void;
  onFileUpload?: (file: File) => void;
  currentUser: {
    id: string;
    name: string;
    type: 'agent';
  };
  customer?: {
    id: string;
    name: string;
    type: 'individual' | 'corporate';
    isOnline?: boolean;
  };
  
  // Status change
  onStatusChange?: (newStatus: string) => void;
  statusOptions?: { value: string; label: string }[];
  currentStatus?: string;
  
  // Legacy support
  comments?: Comment[];
  onAddComment?: (content: string, isInternal: boolean) => void;
}

export default function DetailLayout({
  title,
  subtitle,
  backUrl,
  children,
  recordId,
  recordType,
  messages = [],
  onSendMessage,
  onFileUpload,
  currentUser,
  customer,
  onStatusChange,
  statusOptions = [],
  currentStatus,
  // Legacy support
  comments = [],
  onAddComment
}: DetailLayoutProps) {
  const router = useRouter();

  // Legacy comments'i ChatMessage formatına çevir
  const convertedMessages: ChatMessage[] = comments.map(comment => ({
    id: comment.id,
    content: comment.content,
    timestamp: comment.timestamp,
    sender: {
      id: comment.author === currentUser?.name ? currentUser.id : 'customer',
      name: comment.author,
      type: comment.author === currentUser?.name ? 'agent' : 'customer'
    },
    type: comment.type === 'comment' ? 'text' : (comment.type as 'system' | 'status_change'),
    isInternal: comment.isInternal,
    status: 'read'
  }));

  // Yeni messages varsa onu kullan, yoksa converted messages
  const finalMessages = messages.length > 0 ? messages : convertedMessages;

  // Legacy onAddComment'i onSendMessage'a çevir
  const handleSendMessage = (content: string, isInternal: boolean) => {
    if (onSendMessage) {
      onSendMessage(content, isInternal);
    } else if (onAddComment) {
      onAddComment(content, isInternal);
    }
  };



  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-white border-b border-gray-200 px-6 py-4">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            <button
              onClick={() => router.push(backUrl)}
              className="flex items-center text-gray-600 hover:text-gray-900"
            >
              <ArrowLeft className="h-5 w-5 mr-2" />
              Geri
            </button>
            <div>
              <h1 className="text-2xl font-bold text-gray-900">{title}</h1>
              {subtitle && (
                <div className="text-sm text-gray-500">{subtitle}</div>
              )}
            </div>
          </div>
          
          {/* Status Change */}
          {statusOptions.length > 0 && onStatusChange && (
            <div className="flex items-center space-x-3">
              <span className="text-sm text-gray-600">Durum:</span>
              <select
                value={currentStatus}
                onChange={(e) => onStatusChange(e.target.value)}
                className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              >
                {statusOptions.map(option => (
                  <option key={option.value} value={option.value}>
                    {option.label}
                  </option>
                ))}
              </select>
            </div>
          )}
        </div>
      </div>

      <div className="flex h-[calc(100vh-80px)]">
        {/* Left Panel - Details */}
        <div className="flex-1 overflow-y-auto">
          <div className="p-6">
            {children}
          </div>
        </div>

        {/* Right Panel - Chat */}
        <ChatPanel
          messages={finalMessages}
          onSendMessage={handleSendMessage}
          onFileUpload={onFileUpload}
          currentUser={currentUser || { id: 'agent1', name: 'Acente', type: 'agent' }}
          customer={customer}
          loading={false}
        />
      </div>
    </div>
  );
}