'use client';

import React from 'react';
import { Search } from 'lucide-react';

export interface TableColumn<T = Record<string, unknown>> {
  key: string;
  title: string;
  width?: string;
  render?: (value: unknown, record: T, index: number) => React.ReactNode;
  sortable?: boolean;
  filterable?: boolean;
}

export interface TableAction<T = Record<string, unknown>> {
  key: string;
  label: string;
  icon?: React.ComponentType<{ className?: string }>;
  onClick: (record: T) => void;
  variant?: 'primary' | 'secondary' | 'success' | 'warning' | 'danger';
  condition?: (record: T) => boolean;
}

export interface DataTableProps<T = Record<string, unknown>> {
  // Data
  data: T[];
  columns: TableColumn<T>[];
  loading?: boolean;
  
  // Search & Filter
  searchable?: boolean;
  searchPlaceholder?: string;
  searchValue?: string;
  onSearchChange?: (value: string) => void;
  
  // Filters
  filters?: Array<{
    key: string;
    label: string;
    options: Array<{ value: string; label: string }>;
    value: string;
    onChange: (value: string) => void;
  }>;
  
  // Actions
  actions?: TableAction<T>[];
  bulkActions?: Array<{
    key: string;
    label: string;
    icon?: React.ComponentType<{ className?: string }>;
    onClick: (selectedIds: string[]) => void;
    variant?: 'primary' | 'secondary' | 'success' | 'warning' | 'danger';
  }>;
  
  // Header
  title?: string;
  description?: string;
  headerActions?: React.ReactNode;
  
  // Selection
  selectable?: boolean;
  selectedIds?: string[];
  onSelectionChange?: (selectedIds: string[]) => void;
  getRowId?: (record: T) => string;
  
  // Empty state
  emptyTitle?: string;
  emptyDescription?: string;
  emptyIcon?: React.ComponentType<{ className?: string }>;
  
  // Pagination
  pagination?: {
    current: number;
    pageSize: number;
    total: number;
    onChange: (page: number, pageSize: number) => void;
  };
}

export default function DataTable<T = Record<string, unknown>>({
  data,
  columns,
  loading = false,
  searchable = true,
  searchPlaceholder = "Ara...",
  searchValue = "",
  onSearchChange,
  filters = [],
  actions = [],
  bulkActions = [],
  title,
  description,
  headerActions,
  selectable = false,
  selectedIds = [],
  onSelectionChange,
  getRowId = (record: T) => (record as Record<string, unknown>).id as string,
  emptyTitle = "Veri bulunamadı",
  emptyDescription = "Henüz kayıt bulunmuyor.",
  emptyIcon: EmptyIcon = Search,
  pagination
}: DataTableProps<T>) {

  const handleSelectAll = (checked: boolean) => {
    if (!onSelectionChange) return;
    
    if (checked) {
      const allIds = data.map(getRowId);
      onSelectionChange(allIds);
    } else {
      onSelectionChange([]);
    }
  };

  const handleSelectRow = (recordId: string, checked: boolean) => {
    if (!onSelectionChange) return;
    
    if (checked) {
      onSelectionChange([...selectedIds, recordId]);
    } else {
      onSelectionChange(selectedIds.filter(id => id !== recordId));
    }
  };

  const getActionVariantClass = (variant?: string) => {
    switch (variant) {
      case 'primary': return 'text-blue-600 hover:text-blue-900';
      case 'success': return 'text-green-600 hover:text-green-900';
      case 'warning': return 'text-yellow-600 hover:text-yellow-900';
      case 'danger': return 'text-red-600 hover:text-red-900';
      default: return 'text-gray-600 hover:text-gray-900';
    }
  };

  const getBulkActionVariantClass = (variant?: string) => {
    switch (variant) {
      case 'primary': return 'bg-blue-600 hover:bg-blue-700 text-white';
      case 'success': return 'bg-green-600 hover:bg-green-700 text-white';
      case 'warning': return 'bg-yellow-600 hover:bg-yellow-700 text-white';
      case 'danger': return 'bg-red-600 hover:bg-red-700 text-white';
      default: return 'bg-gray-600 hover:bg-gray-700 text-white';
    }
  };

  if (loading) {
    return (
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="animate-pulse p-6">
          <div className="h-6 bg-gray-200 rounded w-1/4 mb-4"></div>
          <div className="space-y-3">
            {[1, 2, 3, 4, 5].map(i => (
              <div key={i} className="h-12 bg-gray-200 rounded"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      {(title || description || headerActions) && (
        <div className="flex justify-between items-start">
          <div>
            {title && (
              <h1 className="text-2xl font-bold text-gray-900">{title}</h1>
            )}
            {description && (
              <p className="text-gray-600 mt-1">{description}</p>
            )}
          </div>
          {headerActions && (
            <div className="flex items-center space-x-3">
              {headerActions}
            </div>
          )}
        </div>
      )}

      {/* Bulk Actions */}
      {selectable && selectedIds.length > 0 && bulkActions.length > 0 && (
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
          <div className="flex items-center justify-between">
            <span className="text-sm text-blue-800">
              {selectedIds.length} öğe seçildi
            </span>
            <div className="flex items-center space-x-2">
              {bulkActions.map((action) => (
                <button
                  key={action.key}
                  onClick={() => action.onClick(selectedIds)}
                  className={`px-3 py-1 text-sm rounded-lg flex items-center space-x-1 ${getBulkActionVariantClass(action.variant)}`}
                >
                  {action.icon && <action.icon className="h-4 w-4" />}
                  <span>{action.label}</span>
                </button>
              ))}
            </div>
          </div>
        </div>
      )}

      {/* Filters */}
      {(searchable || filters.length > 0) && (
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex flex-col sm:flex-row gap-4">
            {searchable && (
              <div className="flex-1">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                  <input
                    type="text"
                    placeholder={searchPlaceholder}
                    value={searchValue}
                    onChange={(e) => onSearchChange?.(e.target.value)}
                    className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  />
                </div>
              </div>
            )}
            
            {filters.map((filter) => (
              <div key={filter.key} className="sm:w-48">
                <select
                  value={filter.value}
                  onChange={(e) => filter.onChange(e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  {filter.options.map((option) => (
                    <option key={option.value} value={option.value}>
                      {option.label}
                    </option>
                  ))}
                </select>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Table */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
        <div className="overflow-x-auto">
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gray-50">
              <tr>
                {selectable && (
                  <th className="px-6 py-3 text-left">
                    <input
                      type="checkbox"
                      checked={selectedIds.length === data.length && data.length > 0}
                      onChange={(e) => handleSelectAll(e.target.checked)}
                      className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                    />
                  </th>
                )}
                {columns.map((column) => (
                  <th
                    key={column.key}
                    className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider"
                    style={{ width: column.width }}
                  >
                    {column.title}
                  </th>
                ))}
                {actions.length > 0 && (
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    İşlemler
                  </th>
                )}
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {data.map((record, index) => {
                const recordId = getRowId(record);
                const isSelected = selectedIds.includes(recordId);
                
                return (
                  <tr key={recordId} className="hover:bg-gray-50">
                    {selectable && (
                      <td className="px-6 py-4 whitespace-nowrap">
                        <input
                          type="checkbox"
                          checked={isSelected}
                          onChange={(e) => handleSelectRow(recordId, e.target.checked)}
                          className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                        />
                      </td>
                    )}
                    {columns.map((column) => (
                      <td key={column.key} className="px-6 py-4 whitespace-nowrap">
                        {column.render
                          ? column.render((record as Record<string, unknown>)[column.key], record, index)
                          : <span className="text-sm text-gray-900">{String((record as Record<string, unknown>)[column.key] ?? '')}</span>
                        }
                      </td>
                    ))}
                    {actions.length > 0 && (
                      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                        <div className="flex items-center space-x-2">
                          {actions
                            .filter(action => !action.condition || action.condition(record))
                            .map((action) => (
                              <button
                                key={action.key}
                                onClick={() => action.onClick(record)}
                                className={`${getActionVariantClass(action.variant)} transition-colors`}
                                title={action.label}
                              >
                                {action.icon ? (
                                  <action.icon className="h-4 w-4" />
                                ) : (
                                  <span className="text-xs px-2 py-1 border rounded">
                                    {action.label}
                                  </span>
                                )}
                              </button>
                            ))}
                        </div>
                      </td>
                    )}
                  </tr>
                );
              })}
            </tbody>
          </table>
        </div>

        {/* Empty State */}
        {data.length === 0 && (
          <div className="text-center py-12">
            <EmptyIcon className="mx-auto h-12 w-12 text-gray-400" />
            <h3 className="mt-2 text-sm font-medium text-gray-900">{emptyTitle}</h3>
            <p className="mt-1 text-sm text-gray-500">{emptyDescription}</p>
          </div>
        )}
      </div>

      {/* Pagination */}
      {pagination && data.length > 0 && (
        <div className="flex items-center justify-between">
          <div className="text-sm text-gray-700">
            Toplam {pagination.total} kayıttan {((pagination.current - 1) * pagination.pageSize) + 1}-{Math.min(pagination.current * pagination.pageSize, pagination.total)} arası gösteriliyor
          </div>
          <div className="flex items-center space-x-2">
            <button
              onClick={() => pagination.onChange(pagination.current - 1, pagination.pageSize)}
              disabled={pagination.current <= 1}
              className="px-3 py-1 border border-gray-300 rounded-lg text-sm disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-50"
            >
              Önceki
            </button>
            <span className="px-3 py-1 text-sm">
              {pagination.current} / {Math.ceil(pagination.total / pagination.pageSize)}
            </span>
            <button
              onClick={() => pagination.onChange(pagination.current + 1, pagination.pageSize)}
              disabled={pagination.current >= Math.ceil(pagination.total / pagination.pageSize)}
              className="px-3 py-1 border border-gray-300 rounded-lg text-sm disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-50"
            >
              Sonraki
            </button>
          </div>
        </div>
      )}
    </div>
  );
}