'use client';

import React, { useState, useEffect, useRef } from 'react';
import Link from 'next/link';
import { usePathname, useRouter } from 'next/navigation';
import {
  LayoutDashboard,
  Users,
  Plane,
  Settings,
  Building2,
  Car,
  ChevronDown,
  ChevronRight,
  UserCheck,
  List,
  Hotel,
  FileText,
  CheckCircle,
  MapPin,
  Globe,
  Calculator,
  User,
  LogOut,
  Shield,
  Bell
} from 'lucide-react';
import { cn } from '@/lib/utils';

const navigation = [
  {
    name: 'Dashboard',
    href: '/',
    icon: LayoutDashboard,
    permissions: [] // Dashboard herkese açık
  },
  {
    name: 'Biletleme',
    icon: Plane,
    permissions: ['tickets_view', 'tickets_view_own', 'tickets_view_all'],
    subItems: [
      {
        name: 'Bilet Listesi',
        href: '/tickets',
        icon: List,
        permissions: ['tickets_view', 'tickets_view_own', 'tickets_view_all']
      },
      {
        name: 'Check-in Listesi',
        href: '/checkin',
        icon: UserCheck,
        permissions: ['checkin_view', 'checkin_view_own', 'checkin_view_all']
      },
    ]
  },
  {
    name: 'Konaklama',
    icon: Hotel,
    permissions: ['accommodation_view'],
    subItems: [
      {
        name: 'Talep Listesi',
        href: '/accommodation/requests',
        icon: FileText,
        permissions: ['accommodation_view']
      },
    ]
  },
  {
    name: 'Vize',
    icon: Globe,
    permissions: ['visa_view'],
    subItems: [
      {
        name: 'Talep Listesi',
        href: '/visa/requests',
        icon: FileText,
        permissions: ['visa_view']
      },
      {
        name: 'Kontrol Listesi',
        href: '/visa/control',
        icon: CheckCircle,
        permissions: ['visa_view', 'visa_approve']
      },
    ]
  },
  {
    name: 'Transfer',
    icon: Car,
    permissions: ['transfer_view'],
    subItems: [
      {
        name: 'Talep Listesi',
        href: '/transfer/requests',
        icon: MapPin,
        permissions: ['transfer_view']
      },
    ]
  },
  {
    name: 'Muhasebe',
    href: '/accounting',
    icon: Calculator,
    permissions: ['accounting_view']
  },
  {
    name: 'Müşteriler',
    href: '/customers',
    icon: Users,
    permissions: ['customers_view', 'customers_view_own']
  },
  {
    name: 'Kullanıcı Yönetimi',
    icon: Users,
    permissions: ['users_view', 'departments_view'],
    subItems: [
      {
        name: 'Kullanıcılar',
        href: '/users',
        icon: Users,
        permissions: ['users_view']
      },
      {
        name: 'Departmanlar',
        href: '/departments',
        icon: Building2,
        permissions: ['departments_view']
      },
    ]
  },
  {
    name: 'Ayarlar',
    href: '/settings',
    icon: Settings,
    permissions: ['settings_view']
  },
];

export default function Sidebar() {
  const pathname = usePathname();
  const router = useRouter();
  const [expandedItem, setExpandedItem] = useState<string | null>(null);
  const [userMenuOpen, setUserMenuOpen] = useState(false);
  const [currentUser, setCurrentUser] = useState<{
    id: string;
    firstName: string;
    lastName: string;
    email: string;
    role: string;
    avatar: string | null;
    department: {
      name: string;
      color: string;
    };
    permissions?: string[];
  } | null>(null);
  const [mounted, setMounted] = useState(false);
  const menuRef = useRef<HTMLDivElement>(null);

  // Load user from localStorage
  useEffect(() => {
    setMounted(true);
    const userStr = localStorage.getItem('user');
    if (userStr) {
      try {
        const user = JSON.parse(userStr);
        setCurrentUser(user);
      } catch (error) {
        console.error('Error parsing user data:', error);
        // Fallback to mock user
        setCurrentUser({
          id: '1',
          firstName: 'Demo',
          lastName: 'User',
          email: 'demo@vamos.ist',
          role: 'specialist',
          avatar: null,
          department: {
            name: 'Genel',
            color: 'bg-gray-500'
          }
        });
      }
    }
  }, []);

  // Click outside to close menu
  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (menuRef.current && !menuRef.current.contains(event.target as Node)) {
        setUserMenuOpen(false);
      }
    }

    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  const toggleExpanded = (itemName: string) => {
    setExpandedItem(expandedItem === itemName ? null : itemName);
  };

  const handleLogout = async () => {
    if (!confirm('Çıkış yapmak istediğinize emin misiniz?')) {
      return;
    }

    try {
      // Call logout API if user exists
      if (currentUser?.id) {
        const apiUrl = process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002';
        await fetch(`${apiUrl}/api/auth/logout`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({ userId: currentUser.id }),
        });
      }
    } catch (error) {
      console.error('Logout error:', error);
    } finally {
      // Clear local storage
      localStorage.removeItem('token');
      localStorage.removeItem('user');
      // Clear auth cookie
      document.cookie = 'token=; path=/; expires=Thu, 01 Jan 1970 00:00:00 GMT';
      // Redirect to login
      router.push('/login');
    }
  };

  // Yetki kontrolü fonksiyonu
  const hasPermission = (requiredPermissions: string[]) => {
    // Eğer izin listesi boşsa (dashboard gibi), herkese açık
    if (!requiredPermissions || requiredPermissions.length === 0) {
      return true;
    }

    // Kullanıcı yoksa veya permissions yoksa erişim yok
    if (!currentUser || !currentUser.permissions) {
      return false;
    }

    const userPermissions = currentUser.permissions || [];

    // Wildcard (*) iznine sahipse her şeye erişebilir
    if (userPermissions.includes('*')) {
      return true;
    }

    // system_admin veya agency_manager rolleri çoğu şeye erişebilir
    const adminRoles = ['system_admin', 'agency_manager'];
    if (adminRoles.includes(currentUser.role)) {
      return true;
    }

    // En az bir gerekli izne sahipse true döndür
    return requiredPermissions.some(permission =>
      userPermissions.includes(permission)
    );
  };

  const getRoleLabel = (role: string) => {
    const roleLabels: Record<string, string> = {
      system_admin: 'Sistem Yöneticisi',
      agency_manager: 'Acente Yöneticisi',
      department_manager: 'Departman Yöneticisi',
      senior_specialist: 'Kıdemli Uzman',
      specialist: 'Uzman',
      assistant: 'Uzman Yardımcısı',
      intern: 'Stajyer'
    };
    return roleLabels[role] || role;
  };

  return (
    <div className="flex flex-col w-64 bg-white border-r border-gray-200">
      {/* Logo */}
      <div className="flex items-center justify-center h-16 px-4 border-b border-gray-200">
        <img 
          src="https://milaturizm.com/wp-content/uploads/2022/05/mila-weblogo.png" 
          alt="Mila Turizm" 
          className="h-8 object-contain"
        />
      </div>

      {/* Navigation */}
      <nav className="flex-1 px-4 py-6 space-y-1">
        {mounted && navigation
          .filter(item => hasPermission(item.permissions))
          .map((item) => {
          const hasSubItems = 'subItems' in item && item.subItems;
          const isExpanded = expandedItem === item.name;
          const isActive = !hasSubItems && pathname === item.href;
          const hasActiveSubItem = hasSubItems && item.subItems?.some(subItem => pathname === subItem.href);

          // Alt menülerini filtrele
          const filteredSubItems = hasSubItems
            ? item.subItems?.filter(subItem => hasPermission(subItem.permissions))
            : [];

          // Hiç erişilebilir alt menü yoksa bu menüyü gösterme
          if (hasSubItems && filteredSubItems.length === 0) {
            return null;
          }

          return (
            <div key={item.name} className="space-y-1">
              {hasSubItems ? (
                <>
                  <button
                    onClick={() => toggleExpanded(item.name)}
                    className={cn(
                      'group flex items-center justify-between w-full px-3 py-2 text-sm font-medium rounded-lg transition-colors',
                      hasActiveSubItem
                        ? 'bg-blue-50 text-blue-700'
                        : 'text-gray-600 hover:text-gray-900 hover:bg-gray-50'
                    )}
                  >
                    <div className="flex items-center">
                      <item.icon
                        className={cn(
                          'mr-3 h-5 w-5',
                          hasActiveSubItem ? 'text-blue-700' : 'text-gray-400'
                        )}
                      />
                      {item.name}
                    </div>
                    {isExpanded ? (
                      <ChevronDown className="h-4 w-4" />
                    ) : (
                      <ChevronRight className="h-4 w-4" />
                    )}
                  </button>

                  {isExpanded && (
                    <div className="ml-4 space-y-1">
                      {filteredSubItems.map((subItem) => {
                        const isSubItemActive = pathname === subItem.href;
                        return (
                          <Link
                            key={subItem.name}
                            href={subItem.href}
                            className={cn(
                              'group flex items-center px-3 py-2 text-sm font-medium rounded-lg transition-colors',
                              isSubItemActive
                                ? 'bg-blue-100 text-blue-700 border-blue-200'
                                : 'text-gray-500 hover:text-gray-900 hover:bg-gray-50'
                            )}
                          >
                            <subItem.icon
                              className={cn(
                                'mr-3 h-4 w-4',
                                isSubItemActive ? 'text-blue-700' : 'text-gray-400'
                              )}
                            />
                            {subItem.name}
                          </Link>
                        );
                      })}
                    </div>
                  )}
                </>
              ) : (
                <Link
                  href={item.href!}
                  className={cn(
                    'group flex items-center px-3 py-2 text-sm font-medium rounded-lg transition-colors',
                    isActive
                      ? 'bg-blue-50 text-blue-700 border-blue-200'
                      : 'text-gray-600 hover:text-gray-900 hover:bg-gray-50'
                  )}
                >
                  <item.icon
                    className={cn(
                      'mr-3 h-5 w-5',
                      isActive ? 'text-blue-700' : 'text-gray-400'
                    )}
                  />
                  {item.name}
                </Link>
              )}
            </div>
          );
        })}
      </nav>

      {/* User Profile Footer */}
      {currentUser && (
        <div className="relative border-t border-gray-200" ref={menuRef}>
          <button
            onClick={() => setUserMenuOpen(!userMenuOpen)}
            className="w-full p-4 flex items-center space-x-3 hover:bg-gray-50 transition-colors"
          >
            <div className="flex-shrink-0 w-10 h-10 bg-gradient-to-r from-blue-600 to-purple-600 rounded-full flex items-center justify-center">
              {currentUser.avatar ? (
                <img
                  src={currentUser.avatar}
                  alt={currentUser.firstName}
                  className="w-10 h-10 rounded-full"
                />
              ) : (
                <span className="text-sm font-medium text-white">
                  {currentUser.firstName.charAt(0)}{currentUser.lastName.charAt(0)}
                </span>
              )}
            </div>
            <div className="flex-1 text-left">
              <p className="text-sm font-medium text-gray-900">
                {currentUser.firstName} {currentUser.lastName}
              </p>
              <p className="text-xs text-gray-500">{getRoleLabel(currentUser.role)}</p>
            </div>
            <ChevronDown
              className={cn(
                'w-4 h-4 text-gray-400 transition-transform',
                userMenuOpen && 'transform rotate-180'
              )}
            />
          </button>

          {/* Dropdown Menu */}
          {userMenuOpen && (
            <div className="absolute bottom-full left-0 right-0 mb-2 mx-4 bg-white rounded-lg shadow-lg border border-gray-200 py-2">
              <div className="px-4 py-3 border-b border-gray-100">
                <p className="text-sm font-medium text-gray-900">
                  {currentUser.firstName} {currentUser.lastName}
                </p>
                <p className="text-xs text-gray-500 mt-0.5">{currentUser.email}</p>
                <div className="flex items-center mt-2">
                  <div className={`w-2 h-2 rounded-full ${currentUser.department.color} mr-2`}></div>
                  <span className="text-xs text-gray-600">{currentUser.department.name}</span>
                </div>
              </div>

              <div className="py-1">
                <Link
                  href={`/users/${currentUser.id}`}
                  className="flex items-center px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 transition-colors"
                  onClick={() => setUserMenuOpen(false)}
                >
                  <User className="w-4 h-4 mr-3 text-gray-400" />
                  Profilim
                </Link>

                <Link
                  href={`/users/${currentUser.id}/permissions`}
                  className="flex items-center px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 transition-colors"
                  onClick={() => setUserMenuOpen(false)}
                >
                  <Shield className="w-4 h-4 mr-3 text-gray-400" />
                  Yetkilerim
                </Link>

                <Link
                  href="/settings"
                  className="flex items-center px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 transition-colors"
                  onClick={() => setUserMenuOpen(false)}
                >
                  <Settings className="w-4 h-4 mr-3 text-gray-400" />
                  Ayarlar
                </Link>

                <button
                  className="flex items-center w-full px-4 py-2 text-sm text-gray-700 hover:bg-gray-50 transition-colors"
                  onClick={() => setUserMenuOpen(false)}
                >
                  <Bell className="w-4 h-4 mr-3 text-gray-400" />
                  Bildirimler
                  <span className="ml-auto bg-red-500 text-white text-xs rounded-full px-2 py-0.5">
                    3
                  </span>
                </button>
              </div>

              <div className="border-t border-gray-100 pt-1">
                <button
                  onClick={handleLogout}
                  className="flex items-center w-full px-4 py-2 text-sm text-red-600 hover:bg-red-50 transition-colors"
                >
                  <LogOut className="w-4 h-4 mr-3" />
                  Çıkış Yap
                </button>
              </div>
            </div>
          )}
        </div>
      )}
    </div>
  );
}