'use client';

import { useEffect, useState } from 'react';
import { useRouter } from 'next/navigation';
import { AlertCircle } from 'lucide-react';

interface UserData {
  role: string;
  permissions?: string[];
  [key: string]: unknown;
}

interface PermissionGuardProps {
  children: React.ReactNode;
  requiredPermissions?: string[];
  requiredRoles?: string[];
  fallbackUrl?: string;
  showError?: boolean;
}

export default function PermissionGuard({
  children,
  requiredPermissions = [],
  requiredRoles = [],
  fallbackUrl = '/',
  showError = true,
}: PermissionGuardProps) {
  const router = useRouter();
  const [hasAccess, setHasAccess] = useState<boolean | null>(null);
  const [user, setUser] = useState<UserData | null>(null);

  useEffect(() => {
    // Kullanıcı bilgisini localStorage'dan al
    const userStr = localStorage.getItem('user');

    if (!userStr) {
      // Kullanıcı giriş yapmamış, login'e yönlendir
      router.push('/login');
      return;
    }

    try {
      const userData = JSON.parse(userStr);
      setUser(userData);

      // Yetki kontrolü yap
      const access = checkPermissions(userData, requiredPermissions, requiredRoles);
      setHasAccess(access);

      // Erişim yoksa yönlendir
      if (!access) {
        setTimeout(() => {
          router.push(fallbackUrl);
        }, showError ? 3000 : 0); // Hata mesajı gösteriliyorsa 3 saniye bekle
      }
    } catch (error) {
      console.error('User data parse error:', error);
      router.push('/login');
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [fallbackUrl, router, showError]); // requiredPermissions ve requiredRoles'u kaldırdık

  // Yetki kontrolü fonksiyonu
  const checkPermissions = (
    userData: UserData,
    permissions: string[],
    roles: string[]
  ): boolean => {
    // Eğer hiç gereksinim yoksa herkese açık
    if (permissions.length === 0 && roles.length === 0) {
      return true;
    }

    const userPermissions = Array.isArray(userData.permissions)
      ? userData.permissions
      : [];

    // Wildcard (*) iznine sahipse her şeye erişebilir
    if (userPermissions.includes('*')) {
      return true;
    }

    // system_admin veya agency_manager rolleri çoğu şeye erişebilir
    const adminRoles = ['system_admin', 'agency_manager'];
    if (adminRoles.includes(userData.role)) {
      return true;
    }

    // Rol kontrolü
    if (roles.length > 0) {
      if (!roles.includes(userData.role)) {
        return false;
      }
    }

    // İzin kontrolü
    if (permissions.length > 0) {
      const hasRequiredPermission = permissions.some(permission =>
        userPermissions.includes(permission)
      );

      if (!hasRequiredPermission) {
        return false;
      }
    }

    return true;
  };

  // Yükleniyor durumu
  if (hasAccess === null) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  // Erişim yok
  if (!hasAccess) {
    if (!showError) {
      return null;
    }

    return (
      <div className="flex items-center justify-center min-h-screen bg-gray-50">
        <div className="max-w-md w-full bg-white shadow-lg rounded-lg p-6">
          <div className="flex items-center justify-center w-12 h-12 mx-auto bg-red-100 rounded-full">
            <AlertCircle className="h-6 w-6 text-red-600" />
          </div>
          <h3 className="mt-4 text-lg font-medium text-gray-900 text-center">
            Erişim Reddedildi
          </h3>
          <p className="mt-2 text-sm text-gray-500 text-center">
            Bu sayfaya erişim yetkiniz bulunmamaktadır.
          </p>
          {requiredPermissions.length > 0 && (
            <div className="mt-4 p-3 bg-gray-50 rounded-md">
              <p className="text-xs font-medium text-gray-700 mb-2">
                Gerekli İzinler:
              </p>
              <ul className="text-xs text-gray-600 space-y-1">
                {requiredPermissions.map((perm) => (
                  <li key={perm} className="flex items-center">
                    <span className="w-1.5 h-1.5 bg-gray-400 rounded-full mr-2"></span>
                    {perm}
                  </li>
                ))}
              </ul>
            </div>
          )}
          {user && (
            <div className="mt-4 p-3 bg-blue-50 rounded-md">
              <p className="text-xs font-medium text-blue-700 mb-1">
                Mevcut Rolünüz:
              </p>
              <p className="text-xs text-blue-600">{getRoleLabel(user.role)}</p>
            </div>
          )}
          <p className="mt-4 text-xs text-gray-500 text-center">
            Ana sayfaya yönlendiriliyorsunuz...
          </p>
        </div>
      </div>
    );
  }

  // Erişim var, children'ı göster
  return <>{children}</>;
}

function getRoleLabel(role: string): string {
  const roleLabels: Record<string, string> = {
    system_admin: 'Sistem Yöneticisi',
    agency_manager: 'Acente Yöneticisi',
    department_manager: 'Departman Yöneticisi',
    senior_specialist: 'Kıdemli Uzman',
    specialist: 'Uzman',
    assistant: 'Asistan',
    intern: 'Stajyer',
  };

  return roleLabels[role] || role;
}
