'use client';

import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle, Button, Input } from '@/components/ui';
import { Plane, User, Phone } from 'lucide-react';

interface CheckinFormData {
  pnr: string;
  lastName: string;
  firstName?: string;
  phone?: string;
}

interface CheckinFormProps {
  onSubmit: (data: CheckinFormData) => void;
  isLoading?: boolean;
}

export default function CheckinForm({ onSubmit, isLoading = false }: CheckinFormProps) {
  const [formData, setFormData] = useState<CheckinFormData>({
    pnr: '',
    lastName: '',
    firstName: '',
    phone: '',
  });

  const [errors, setErrors] = useState<Partial<CheckinFormData>>({});

  const validateForm = (): boolean => {
    const newErrors: Partial<CheckinFormData> = {};

    if (!formData.pnr.trim()) {
      newErrors.pnr = 'PNR zorunludur';
    } else if (formData.pnr.length < 5) {
      newErrors.pnr = 'PNR en az 5 karakter olmalıdır';
    }

    if (!formData.lastName.trim()) {
      newErrors.lastName = 'Soyisim zorunludur';
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (validateForm()) {
      onSubmit(formData);
    }
  };

  const handleInputChange = (field: keyof CheckinFormData, value: string) => {
    setFormData(prev => ({ ...prev, [field]: value }));
    // Clear error when user starts typing
    if (errors[field]) {
      setErrors(prev => ({ ...prev, [field]: undefined }));
    }
  };

  return (
    <Card className="w-full max-w-2xl">
      <CardHeader>
        <div className="flex items-center space-x-3">
          <div className="p-2 bg-blue-100 rounded-lg">
            <Plane className="h-5 w-5 text-blue-600" />
          </div>
          <div>
            <CardTitle>Hızlı Check-in</CardTitle>
            <p className="text-sm text-gray-600 mt-1">
              PNR ve soyisim bilgileri ile check-in yapın
            </p>
          </div>
        </div>
      </CardHeader>
      <CardContent>
        <form onSubmit={handleSubmit} className="space-y-6">
          {/* PNR Field */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              PNR Kodu *
            </label>
            <div className="relative">
              <Input
                type="text"
                value={formData.pnr}
                onChange={(e) => handleInputChange('pnr', e.target.value.toUpperCase())}
                placeholder="Örn: ABC123"
                className={`pl-10 ${errors.pnr ? 'border-red-500' : ''}`}
                maxLength={10}
              />
              <Plane className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
            </div>
            {errors.pnr && (
              <p className="mt-1 text-sm text-red-600">{errors.pnr}</p>
            )}
          </div>

          {/* Last Name Field */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Soyisim *
            </label>
            <div className="relative">
              <Input
                type="text"
                value={formData.lastName}
                onChange={(e) => handleInputChange('lastName', e.target.value.toUpperCase())}
                placeholder="YILMAZ"
                className={`pl-10 ${errors.lastName ? 'border-red-500' : ''}`}
              />
              <User className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
            </div>
            {errors.lastName && (
              <p className="mt-1 text-sm text-red-600">{errors.lastName}</p>
            )}
          </div>

          {/* Optional Fields */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                İsim <span className="text-gray-400">(İsteğe bağlı)</span>
              </label>
              <Input
                type="text"
                value={formData.firstName}
                onChange={(e) => handleInputChange('firstName', e.target.value)}
                placeholder="Ahmet"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Telefon <span className="text-gray-400">(SMS için)</span>
              </label>
              <div className="relative">
                <Input
                  type="tel"
                  value={formData.phone}
                  onChange={(e) => handleInputChange('phone', e.target.value)}
                  placeholder="05321234567"
                  className="pl-10"
                />
                <Phone className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
              </div>
            </div>
          </div>

          {/* Submit Button */}
          <div className="flex justify-end space-x-3">
            <Button
              type="button"
              variant="outline"
              onClick={() => {
                setFormData({ pnr: '', lastName: '', firstName: '', phone: '' });
                setErrors({});
              }}
            >
              Temizle
            </Button>
            <Button
              type="submit"
              disabled={isLoading}
              className="min-w-[120px]"
            >
              {isLoading ? 'İşleniyor...' : 'Check-in Başlat'}
            </Button>
          </div>
        </form>
      </CardContent>
    </Card>
  );
}