import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui';
import { 
  CheckCircle, 
  XCircle, 
  Clock, 
  Plane,
  AlertCircle,
  Calendar
} from 'lucide-react';
import { formatTime, formatDate } from '@/lib/utils';
import type { FlightTicket } from '@/types';

interface RecentActivityProps {
  tickets: FlightTicket[];
}

export default function RecentActivity({ tickets }: RecentActivityProps) {
  const getStatusIcon = (status: FlightTicket['checkinStatus']) => {
    switch (status) {
      case 'completed':
        return <CheckCircle className="h-4 w-4 text-green-500" />;
      case 'failed':
        return <XCircle className="h-4 w-4 text-red-500" />;
      case 'processing':
        return <Clock className="h-4 w-4 text-blue-500 animate-spin" />;
      case 'ready':
        return <AlertCircle className="h-4 w-4 text-orange-500" />;
      case 'waiting':
        return <Calendar className="h-4 w-4 text-gray-400" />;
      default:
        return <Clock className="h-4 w-4 text-gray-400" />;
    }
  };

  const getStatusText = (status: FlightTicket['checkinStatus']) => {
    switch (status) {
      case 'completed':
        return 'Tamamlandı';
      case 'failed':
        return 'Başarısız';
      case 'processing':
        return 'İşleniyor';
      case 'ready':
        return 'Hazır';
      case 'waiting':
        return 'Bekliyor';
      case 'expired':
        return 'Süresi geçti';
      default:
        return 'Bilinmiyor';
    }
  };

  const getStatusColor = (status: FlightTicket['checkinStatus']) => {
    switch (status) {
      case 'completed':
        return 'text-green-600 bg-green-50';
      case 'failed':
        return 'text-red-600 bg-red-50';
      case 'processing':
        return 'text-blue-600 bg-blue-50';
      case 'ready':
        return 'text-orange-600 bg-orange-50';
      case 'waiting':
        return 'text-gray-600 bg-gray-50';
      case 'expired':
        return 'text-gray-500 bg-gray-100';
      default:
        return 'text-gray-600 bg-gray-50';
    }
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle>Son İşlemler</CardTitle>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {tickets.length === 0 ? (
            <div className="text-center py-6">
              <Plane className="mx-auto h-12 w-12 text-gray-400" />
              <h3 className="mt-2 text-sm font-medium text-gray-900">
                Henüz bilet eklenmemiş
              </h3>
              <p className="mt-1 text-sm text-gray-500">
                İlk biletinizi ekleyin
              </p>
            </div>
          ) : (
            tickets.map((ticket) => (
              <div key={ticket.id} className="flex items-center space-x-3">
                {getStatusIcon(ticket.checkinStatus)}
                <div className="flex-1 min-w-0">
                  <p className="text-sm font-medium text-gray-900">
                    PNR: {ticket.pnr}
                  </p>
                  <p className="text-sm text-gray-500">
                    {ticket.customer.firstName} {ticket.customer.lastName}
                  </p>
                </div>
                <div className="flex items-center space-x-2">
                  <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getStatusColor(ticket.checkinStatus)}`}>
                    {getStatusText(ticket.checkinStatus)}
                  </span>
                  <span className="text-sm text-gray-500">
                    {formatTime(ticket.createdAt)}
                  </span>
                </div>
              </div>
            ))
          )}
        </div>
      </CardContent>
    </Card>
  );
}