'use client';

import React, { useState, useEffect } from 'react';
import {
  Building2,
  Plus,
  Eye,
  Edit,
  Trash2,
  Users,
  UserCheck,
  Settings,
  TrendingUp,
  Activity
} from 'lucide-react';
import Link from 'next/link';
import DataTable, { type TableColumn, type TableAction } from '@/components/ui/DataTable';
import type { Department, AgencyUser } from '@/types';
import { departmentApi, agencyUserApi } from '@/services/api';
import PermissionGuard from '@/components/guards/PermissionGuard';

export default function DepartmentsPage() {
  return (
    <PermissionGuard requiredPermissions={['departments_view']}>
      <DepartmentsPageContent />
    </PermissionGuard>
  );
}

function DepartmentsPageContent() {
  const [departments, setDepartments] = useState<Department[]>([]);
  const [users, setUsers] = useState<AgencyUser[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState<string>('all');

  useEffect(() => {
    fetchDepartments();
    fetchUsers();
  }, []);

  const fetchDepartments = async () => {
    try {
      setLoading(true);

      // Fetch departments from API
      const data = await departmentApi.getDepartments();

      // Convert API response to frontend format
      const departmentsWithDates = data.map((dept: {
        id: string;
        name: string;
        description?: string;
        color: string;
        parent_department_id?: string;
        manager_id?: string;
        is_active?: boolean;
        user_count?: number;
        created_at: string;
        updated_at: string;
      }) => ({
        id: dept.id,
        name: dept.name,
        description: dept.description || '',
        color: dept.color,
        parentDepartmentId: dept.parent_department_id,
        managerId: dept.manager_id,
        isActive: Boolean(dept.is_active),
        userCount: dept.user_count || 0,
        createdAt: new Date(dept.created_at),
        updatedAt: new Date(dept.updated_at)
      }));

      setDepartments(departmentsWithDates);
    } catch (error) {
      console.error('Error fetching departments:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchUsers = async () => {
    try {
      // Fetch users from API
      const data = await agencyUserApi.getAgencyUsers();
      setUsers(data);
    } catch (error) {
      console.error('Error fetching users:', error);
    }
  };

  const handleDepartmentStatusChange = async (departmentId: string, isActive: boolean) => {
    try {
      // Update department status via API
      await departmentApi.updateDepartment(departmentId, { is_active: isActive });

      // Update local state
      setDepartments(prev => prev.map(dept =>
        dept.id === departmentId
          ? { ...dept, isActive, updatedAt: new Date() }
          : dept
      ));

      alert(isActive ? 'Departman aktifleştirildi' : 'Departman pasifleştirildi');
    } catch (error) {
      console.error('Error updating department status:', error);
      alert('Departman durumu güncellenirken bir hata oluştu');
    }
  };

  const handleDepartmentDelete = async (departmentId: string) => {
    if (!confirm('Bu departmanı silmek istediğinize emin misiniz? Bu işlem geri alınamaz.')) {
      return;
    }

    try {
      // Delete department via API (soft delete)
      await departmentApi.deleteDepartment(departmentId);

      // Remove from local state
      setDepartments(prev => prev.filter(dept => dept.id !== departmentId));

      alert('Departman başarıyla silindi');
    } catch (error) {
      console.error('Error deleting department:', error);
      alert('Departman silinirken bir hata oluştu. Departmanda aktif kullanıcı olmadığından emin olun.');
    }
  };

  // Render departman bilgileri
  const renderDepartment = (department: Department) => (
    <div className="flex items-center">
      <div className={`w-4 h-4 rounded-full ${department.color} mr-3`}></div>
      <div>
        <div className="text-sm font-medium text-gray-900">{department.name}</div>
        <div className="text-sm text-gray-500">{department.description}</div>
      </div>
    </div>
  );

  // Render yönetici
  const renderManager = (department: Department) => {
    const manager = users.find(u => u.id === department.managerId);
    
    if (!manager) {
      return (
        <span className="text-sm text-gray-400 italic">Atanmamış</span>
      );
    }

    return (
      <div className="flex items-center">
        <div className="flex-shrink-0 h-8 w-8 rounded-full bg-gradient-to-r from-blue-600 to-purple-600 flex items-center justify-center">
          <span className="text-xs font-medium text-white">
            {manager.firstName.charAt(0)}{manager.lastName.charAt(0)}
          </span>
        </div>
        <div className="ml-2">
          <div className="text-sm font-medium text-gray-900">
            {manager.firstName} {manager.lastName}
          </div>
          <div className="text-xs text-gray-500">{manager.position}</div>
        </div>
      </div>
    );
  };

  // Render kullanıcı sayısı
  const renderUserCount = (department: Department) => (
    <div className="flex items-center">
      <Users className="h-4 w-4 text-gray-400 mr-1" />
      <span className="text-sm text-gray-900">{department.userCount}</span>
      <span className="text-xs text-gray-500 ml-1">kullanıcı</span>
    </div>
  );

  // Render durum
  const renderStatus = (department: Department) => (
    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
      department.isActive 
        ? 'bg-green-100 text-green-800' 
        : 'bg-red-100 text-red-800'
    }`}>
      {department.isActive ? 'Aktif' : 'Pasif'}
    </span>
  );

  // Tablo sütunları
  const columns: TableColumn<Department>[] = [
    {
      key: 'department',
      title: 'Departman',
      render: (_, record) => renderDepartment(record)
    },
    {
      key: 'userCount',
      title: 'Kullanıcı Sayısı',
      render: (_, record) => renderUserCount(record)
    },
    {
      key: 'status',
      title: 'Durum',
      render: (_, record) => renderStatus(record)
    },
    {
      key: 'createdAt',
      title: 'Oluşturulma',
      render: (_, record) => (
        <div className="text-sm text-gray-500">
          {record.createdAt.toLocaleDateString('tr-TR')}
        </div>
      )
    }
  ];

  // Tablo aksiyonları
  const actions: TableAction<Department>[] = [
    {
      key: 'view',
      label: 'Görüntüle',
      icon: Eye,
      onClick: (record) => window.location.href = `/departments/${record.id}`,
      variant: 'primary'
    },
    {
      key: 'users',
      label: 'Kullanıcılar',
      icon: Users,
      onClick: (record) => window.location.href = `/departments/${record.id}/users`,
      variant: 'secondary'
    },
    {
      key: 'edit',
      label: 'Düzenle',
      icon: Edit,
      onClick: (record) => window.location.href = `/departments/${record.id}/edit`,
      variant: 'secondary'
    },
    {
      key: 'settings',
      label: 'Ayarlar',
      icon: Settings,
      onClick: (record) => window.location.href = `/departments/${record.id}/settings`,
      variant: 'secondary'
    },
    {
      key: 'activate',
      label: 'Aktifleştir',
      icon: UserCheck,
      onClick: (record) => handleDepartmentStatusChange(record.id, true),
      variant: 'success',
      condition: (record) => !record.isActive
    },
    {
      key: 'deactivate',
      label: 'Pasifleştir',
      onClick: (record) => handleDepartmentStatusChange(record.id, false),
      variant: 'warning',
      condition: (record) => record.isActive && record.userCount === 0
    },
    {
      key: 'delete',
      label: 'Sil',
      icon: Trash2,
      onClick: (record) => handleDepartmentDelete(record.id),
      variant: 'danger',
      condition: (record) => record.userCount === 0
    }
  ];

  // Filtreler
  const filters = [
    {
      key: 'status',
      label: 'Durum',
      value: statusFilter,
      onChange: setStatusFilter,
      options: [
        { value: 'all', label: 'Tüm Durumlar' },
        { value: 'active', label: 'Aktif' },
        { value: 'inactive', label: 'Pasif' }
      ]
    }
  ];

  // Filtrelenmiş departmanlar
  const filteredDepartments = departments.filter(department => {
    const matchesSearch = 
      department.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      department.description?.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchesStatus = statusFilter === 'all' || 
      (statusFilter === 'active' && department.isActive) ||
      (statusFilter === 'inactive' && !department.isActive);
    
    return matchesSearch && matchesStatus;
  });

  // İstatistikler
  const stats = {
    total: departments.length,
    active: departments.filter(d => d.isActive).length,
    totalUsers: departments.reduce((sum, d) => sum + d.userCount, 0),
    avgUsersPerDept: departments.length > 0 
      ? Math.round(departments.reduce((sum, d) => sum + d.userCount, 0) / departments.filter(d => d.isActive).length)
      : 0
  };

  return (
    <div className="p-6">
      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Building2 className="h-8 w-8 text-blue-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Toplam Departman</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.total}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Activity className="h-8 w-8 text-green-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Aktif Departman</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.active}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Users className="h-8 w-8 text-purple-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Toplam Kullanıcı</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.totalUsers}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <TrendingUp className="h-8 w-8 text-orange-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Ortalama Kullanıcı</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.avgUsersPerDept}</p>
            </div>
          </div>
        </div>
      </div>

      <DataTable
        title="Departman Yönetimi"
        description="Acente departmanlarını ve organizasyon yapısını yönetin"
        data={filteredDepartments}
        columns={columns}
        actions={actions}
        loading={loading}
        searchable={true}
        searchPlaceholder="Departman adı veya açıklama ara..."
        searchValue={searchTerm}
        onSearchChange={setSearchTerm}
        filters={filters}
        headerActions={
          <div className="flex items-center space-x-4">
            <Link
              href="/users"
              className="bg-purple-600 text-white px-4 py-2 rounded-lg hover:bg-purple-700 flex items-center gap-2"
            >
              <Users className="h-4 w-4" />
              Kullanıcılar
            </Link>
            <Link
              href="/departments/new"
              className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
            >
              <Plus className="h-4 w-4" />
              Yeni Departman
            </Link>
          </div>
        }
        emptyTitle="Departman bulunamadı"
        emptyDescription="Henüz departman eklenmemiş."
        emptyIcon={Building2}
      />
    </div>
  );
}