'use client';

import React, { useState, useEffect } from 'react';
import { Users, Plus, Eye, Edit, Settings, Trash2, User, Building2, Crown } from 'lucide-react';
import Link from 'next/link';
import DataTable, { type TableColumn, type TableAction } from '@/components/ui/DataTable';
import { renderCustomer } from '@/components/ui/TableRenderers';
import type { Customer } from '@/types';
import PermissionGuard from '@/components/guards/PermissionGuard';

export default function CustomersPage() {
  return (
    <PermissionGuard requiredPermissions={['customers_view', 'customers_view_own']}>
      <CustomersPageContent />
    </PermissionGuard>
  );
}

function CustomersPageContent() {
  const [customers, setCustomers] = useState<Customer[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [typeFilter, setTypeFilter] = useState<string>('all');

  useEffect(() => {
    fetchCustomers();
  }, []);

  const fetchCustomers = async () => {
    try {
      setLoading(true);
      // Mock data - gerçek API ile değiştirilecek
      const mockCustomers: Customer[] = [
        {
          id: '1',
          type: 'individual',
          firstName: 'Ahmet',
          lastName: 'Yılmaz',
          email: 'ahmet@email.com',
          phone: '+90 555 123 4567',
          city: 'İstanbul',
          isVip: true,
          isActive: true,
          notes: 'Düzenli müşteri, her yıl tatil planı yapıyor',
          createdAt: new Date('2024-01-15'),
          updatedAt: new Date('2024-12-20')
        },
        {
          id: '2',
          type: 'corporate',
          companyName: 'ABC Teknoloji A.Ş.',
          taxNumber: '1234567890',
          taxOffice: 'Beşiktaş',
          email: 'info@abcteknoloji.com',
          phone: '+90 212 555 0123',
          address: 'Levent Mahallesi, İş Kuleleri',
          city: 'İstanbul',
          contactPerson: 'Fatma Kaya',
          department: 'İnsan Kaynakları',
          isVip: false,
          isActive: true,
          notes: 'Çalışan seyahatleri için anlaşmalı firma',
          createdAt: new Date('2024-03-10'),
          updatedAt: new Date('2024-12-18')
        },
        {
          id: '3',
          type: 'individual',
          firstName: 'Mehmet',
          lastName: 'Demir',
          email: 'mehmet@email.com',
          phone: '+90 555 987 6543',
          city: 'Ankara',
          isVip: false,
          isActive: true,
          createdAt: new Date('2024-06-22'),
          updatedAt: new Date('2024-11-30')
        },
        {
          id: '4',
          type: 'corporate',
          companyName: 'XYZ İnşaat Ltd.',
          taxNumber: '0987654321',
          taxOffice: 'Çankaya',
          email: 'contact@xyzinsaat.com',
          phone: '+90 312 555 0456',
          address: 'Kızılay Meydanı No:15',
          city: 'Ankara',
          contactPerson: 'Can Özkan',
          department: 'Genel Müdürlük',
          isVip: true,
          isActive: true,
          notes: 'Büyük projeler için grup seyahatleri organize ediyor',
          createdAt: new Date('2024-02-05'),
          updatedAt: new Date('2024-12-15')
        }
      ];
      
      setCustomers(mockCustomers);
    } catch (error) {
      console.error('Error fetching customers:', error);
    } finally {
      setLoading(false);
    }
  };

  // Tablo sütunları
  const columns: TableColumn<Customer>[] = [
    {
      key: 'customer',
      title: 'Müşteri',
      render: (_, record) => renderCustomer(record)
    },
    {
      key: 'contact',
      title: 'İletişim',
      render: (_, record) => (
        <div>
          <div className="text-sm text-gray-900">{record.email}</div>
          <div className="text-sm text-gray-500">{record.phone}</div>
        </div>
      )
    },
    {
      key: 'type',
      title: 'Tür',
      render: (_, record) => (
        <div>
          <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
            record.type === 'corporate' 
              ? 'bg-purple-100 text-purple-800' 
              : 'bg-blue-100 text-blue-800'
          }`}>
            {record.type === 'corporate' ? 'Kurumsal' : 'Bireysel'}
          </span>
          {record.type === 'corporate' && record.taxNumber && (
            <div className="text-xs text-gray-500 mt-1">
              VN: {record.taxNumber}
            </div>
          )}
        </div>
      )
    },
    {
      key: 'location',
      title: 'Konum',
      render: (_, record) => (
        <div>
          <div className="text-sm text-gray-900">{record.city}</div>
          {record.type === 'corporate' && record.taxOffice && (
            <div className="text-xs text-gray-500">{record.taxOffice}</div>
          )}
        </div>
      )
    },
    {
      key: 'status',
      title: 'Durum',
      render: (_, record) => (
        <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
          record.isActive 
            ? 'bg-green-100 text-green-800' 
            : 'bg-red-100 text-red-800'
        }`}>
          {record.isActive ? 'Aktif' : 'Pasif'}
        </span>
      )
    },
    {
      key: 'createdAt',
      title: 'Kayıt Tarihi',
      render: (_, record) => (
        <div className="text-sm text-gray-500">
          {record.createdAt.toLocaleDateString('tr-TR')}
        </div>
      )
    }
  ];

  // Tablo aksiyonları
  const actions: TableAction<Customer>[] = [
    {
      key: 'view',
      label: 'Görüntüle',
      icon: Eye,
      onClick: (record) => window.location.href = `/customers/${record.id}`,
      variant: 'primary'
    },
    {
      key: 'edit',
      label: 'Düzenle',
      icon: Edit,
      onClick: (record) => window.location.href = `/customers/${record.id}/edit`,
      variant: 'secondary'
    },
    {
      key: 'users',
      label: 'Kullanıcıları Yönet',
      icon: Settings,
      onClick: (record) => window.location.href = `/customers/${record.id}/users`,
      variant: 'secondary',
      condition: (record) => record.type === 'corporate'
    },
    {
      key: 'delete',
      label: 'Sil',
      icon: Trash2,
      onClick: (record) => console.log('Delete:', record.id),
      variant: 'danger'
    }
  ];

  // Filtreler
  const filters = [
    {
      key: 'type',
      label: 'Müşteri Türü',
      value: typeFilter,
      onChange: setTypeFilter,
      options: [
        { value: 'all', label: 'Tüm Müşteriler' },
        { value: 'individual', label: 'Bireysel' },
        { value: 'corporate', label: 'Kurumsal' }
      ]
    }
  ];

  // Filtrelenmiş müşteriler
  const filteredCustomers = customers.filter(customer => {
    const searchText = customer.type === 'individual' 
      ? `${customer.firstName} ${customer.lastName}`.toLowerCase()
      : customer.companyName?.toLowerCase() || '';
    
    const matchesSearch = searchText.includes(searchTerm.toLowerCase()) ||
                         customer.email?.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         customer.phone.includes(searchTerm);
    
    const matchesType = typeFilter === 'all' || customer.type === typeFilter;
    
    return matchesSearch && matchesType;
  });

  // İstatistikler
  const stats = {
    total: customers.length,
    individual: customers.filter(c => c.type === 'individual').length,
    corporate: customers.filter(c => c.type === 'corporate').length,
    vip: customers.filter(c => c.isVip).length,
    active: customers.filter(c => c.isActive).length
  };

  return (
    <div className="p-6">
      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-5 gap-4 mb-6">
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Users className="h-8 w-8 text-blue-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Toplam</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.total}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <User className="h-8 w-8 text-green-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Bireysel</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.individual}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Building2 className="h-8 w-8 text-purple-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Kurumsal</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.corporate}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Crown className="h-8 w-8 text-yellow-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">VIP</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.vip}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <div className="h-8 w-8 bg-green-100 rounded-full flex items-center justify-center">
              <div className="h-3 w-3 bg-green-600 rounded-full"></div>
            </div>
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Aktif</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.active}</p>
            </div>
          </div>
        </div>
      </div>

      <DataTable
        title="Müşteri Yönetimi"
        description="Bireysel ve kurumsal müşterilerinizi yönetin"
        data={filteredCustomers}
        columns={columns}
        actions={actions}
        loading={loading}
        searchable={true}
        searchPlaceholder="Müşteri adı, şirket adı, email veya telefon ara..."
        searchValue={searchTerm}
        onSearchChange={setSearchTerm}
        filters={filters}
        headerActions={
          <Link
            href="/customers/new"
            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
          >
            <Plus className="h-4 w-4" />
            Yeni Müşteri
          </Link>
        }
        emptyTitle="Müşteri bulunamadı"
        emptyDescription="Henüz müşteri eklenmemiş."
        emptyIcon={Users}
      />
    </div>
  );
}