'use client';

import React, { useState, useEffect } from 'react';
import { Eye, RotateCcw, Download, Clock, CheckCircle, XCircle, UserCheck, LayoutGrid, List, Plus, Play, Square, Pause, Trash2 } from 'lucide-react';
import { io, Socket } from 'socket.io-client';
import DataTable, { type TableColumn, type TableAction } from '@/components/ui/DataTable';
import KanbanBoard from '@/components/ui/KanbanBoard';
import ManualCheckinForm from '@/components/ui/ManualCheckinForm';
import {
  renderCustomer,
  renderStatusBadge,
  renderDateTime,
  checkinStatusConfig
} from '@/components/ui/TableRenderers';
import type { CheckinTicket } from '@/types';
import { checkinApi } from '@/services/api';
import PermissionGuard from '@/components/guards/PermissionGuard';

export default function CheckinPage() {
  return (
    <PermissionGuard requiredPermissions={['checkin_view', 'checkin_view_own', 'checkin_view_all']}>
      <CheckinPageContent />
    </PermissionGuard>
  );
}

function CheckinPageContent() {
  const [tickets, setTickets] = useState<CheckinTicket[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState<string>('all');
  const [airlineFilter, setAirlineFilter] = useState<string>('all');
  const [viewMode, setViewMode] = useState<'table' | 'kanban'>('table');
  const [showManualForm, setShowManualForm] = useState(false);
  const [submitLoading, setSubmitLoading] = useState(false);
  const [socket, setSocket] = useState<Socket | null>(null);

  // WebSocket bağlantısı kur
  useEffect(() => {
    const newSocket = io(process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002');
    setSocket(newSocket);

    // Check-in sonuçlarını dinle
    newSocket.on('checkin_result', (data) => {
      console.log('Check-in sonucu alındı:', data);

      // Tablodaki ticket'i güncelle
      setTickets(prevTickets =>
        prevTickets.map(ticket =>
          ticket.id === data.jobId || ticket.ticketRequestId === data.jobId
            ? {
                ...ticket,
                checkinStatus: data.success ? 'completed' as const : 'failed' as const,
                smsStatus: data.smsSent ? 'sent' as const : 'failed' as const,
                boardingPassUrl: data.boardingPassUrl,
                updatedAt: new Date()
              }
            : ticket
        )
      );
    });

    // Progress updates dinle
    newSocket.on('checkin_status', (data) => {
      console.log('Check-in progress:', data);
      // İsteğe bağlı olarak progress bar gösterebiliriz
    });

    return () => {
      newSocket.close();
    };
  }, []);

  useEffect(() => {
    fetchCheckinTickets();

    // Her 5 dakikada bir check-in zamanı kontrolü
    const interval = setInterval(() => {
      checkAndProcessReadyTickets();
    }, 5 * 60 * 1000); // 5 dakika

    // İlk yükleme sonrası hemen kontrol et
    setTimeout(checkAndProcessReadyTickets, 2000);

    return () => clearInterval(interval);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  // LocalStorage'den veriyi yükle
  useEffect(() => {
    const savedTickets = localStorage.getItem('checkin-tickets');
    if (savedTickets) {
      try {
        const parsedTickets = JSON.parse(savedTickets).map((ticket: {
          departureDate: string;
          checkinOpenTime: string;
          createdAt: string;
          updatedAt: string;
        }) => ({
          ...ticket,
          departureDate: new Date(ticket.departureDate),
          checkinOpenTime: new Date(ticket.checkinOpenTime),
          createdAt: new Date(ticket.createdAt),
          updatedAt: new Date(ticket.updatedAt)
        }));
        setTickets(parsedTickets);
      } catch (error) {
        console.error('LocalStorage veri parse hatası:', error);
      }
    }
  }, []);

  // Ticket'lar değiştiğinde localStorage'a kaydet
  useEffect(() => {
    if (tickets.length > 0) {
      localStorage.setItem('checkin-tickets', JSON.stringify(tickets));
    }
  }, [tickets]);

  const fetchCheckinTickets = async () => {
    try {
      setLoading(true);

      // API'den check-in ticket'larını getir
      const response = await checkinApi.getCheckinTickets({
        status: statusFilter !== 'all' ? statusFilter : undefined,
        airline: airlineFilter !== 'all' ? airlineFilter : undefined,
        search: searchTerm || undefined
      });

      if (response.success) {
        // API'den gelen veride passengerFirstName/passengerLastName var
        // Bunları passengers array'ine dönüştür
        const ticketsWithPassengers = (response.data || []).map((ticket: {
          id: string;
          passengerFirstName?: string;
          passengerLastName?: string;
          passengerPhone?: string;
          passengerEmail?: string;
        }) => ({
          ...ticket,
          passengers: ticket.passengerFirstName && ticket.passengerLastName
            ? [{
                id: `p_${ticket.id}`,
                firstName: ticket.passengerFirstName,
                lastName: ticket.passengerLastName,
                phone: ticket.passengerPhone,
                email: ticket.passengerEmail
              }]
            : []
        }));
        setTickets(ticketsWithPassengers);
        console.log('🌐 API\'dan', ticketsWithPassengers.length, 'check-in ticket yüklendi');
      }
    } catch (error) {
      console.error('Check-in tickets API hatası:', error);

      // Hata durumunda localStorage'dan yükle
      try {
        const savedTickets = localStorage.getItem('checkin-tickets');
        if (savedTickets) {
          const parsedTickets = JSON.parse(savedTickets);
          setTickets(parsedTickets);
          console.log('📋 LocalStorage\'dan', parsedTickets.length, 'check-in ticket yüklendi');
        }
      } catch (storageError) {
        console.error('LocalStorage okuma hatası:', storageError);
        setTickets([]);
      }
    } finally {
      setLoading(false);
    }
  };


  // Render yolcu check-in durumu
  const renderPassengerCheckin = (record: CheckinTicket) => (
    <div className="space-y-2">
      {record.passengers?.map((passenger, index) => {
        const result = record.passengerResults?.find(r => r.passengerId === passenger.id);
        return (
          <div key={passenger.id} className="text-sm">
            <div className="font-medium text-gray-900">
              ***** {passenger.lastName}
              {record.passengers && record.passengers.length > 1 && (
                <span className="ml-2 px-1 py-0.5 text-xs bg-blue-100 text-blue-800 rounded">
                  {index + 1}
                </span>
              )}
            </div>
            {result?.seat && (
              <div className="text-xs text-green-600">
                Koltuk: {result.seat}
              </div>
            )}
            {passenger.department && (
              <div className="text-xs text-gray-500">
                {passenger.department} • {passenger.position}
              </div>
            )}
          </div>
        );
      }) || []}
      {record.passengers && record.passengers.length > 0 && (
        <div className="text-xs text-blue-600 font-medium mt-2">
          Toplam {record.passengers.length} yolcu
        </div>
      )}
    </div>
  );

  // Render uçuş bilgileri
  const renderFlightInfo = (record: CheckinTicket) => {
    const departureDate = typeof record.departureDate === 'string'
      ? new Date(record.departureDate)
      : record.departureDate;

    return (
      <div>
        <div className="text-sm text-gray-900">
          {record.airline} {record.flightNumber}
        </div>
        <div className="text-sm text-gray-500">
          {record.departureAirport} → {record.arrivalAirport}
        </div>
        <div className="text-xs text-blue-600 font-mono">
          PNR: {record.pnr}
        </div>
        <div className="text-xs text-gray-500">
          {departureDate?.toLocaleDateString('tr-TR')} {record.departureTime}
        </div>
      </div>
    );
  };

  // Render check-in durumu
  const renderCheckinStatus = (record: CheckinTicket) => (
    <div>
      <div className="flex items-center space-x-2 mb-1">
        {renderStatusBadge(record.checkinStatus, checkinStatusConfig)}
      </div>
      <div className="text-xs text-gray-500">
        Deneme: {record.checkinAttempts}
      </div>
      {record.errorMessage && (
        <div className="text-xs text-red-600 mt-1">
          {record.errorMessage}
        </div>
      )}
    </div>
  );

  // Render biniş kartları
  const renderBoardingPasses = (record: CheckinTicket) => {
    const hasAnyBoardingPass = record.passengerResults?.some(r => r.boardingPassUrl);

    if (!hasAnyBoardingPass) return null;

    return (
      <div className="space-y-1">
        {record.passengerResults?.map((result, index) => (
          result.boardingPassUrl && (
            <a
              key={result.passengerId}
              href={result.boardingPassUrl}
              target="_blank"
              rel="noopener noreferrer"
              className="text-blue-600 hover:text-blue-900 text-xs flex items-center gap-1"
            >
              <Download className="h-3 w-3" />
              Biniş Kartı {record.passengers?.length > 1 ? `${index + 1}` : ''}
            </a>
          )
        ))}
      </div>
    );
  };

  // Tablo sütunları
  const columns: TableColumn<CheckinTicket>[] = [
    {
      key: 'pnr',
      title: 'PNR',
      render: (_, record) => (
        <div className="font-mono text-sm font-semibold text-gray-900">
          {record.pnr}
        </div>
      )
    },
    {
      key: 'surname',
      title: 'Soyisim',
      render: (_, record) => {
        const surnames = record.passengers?.map(p => p.lastName).filter(Boolean).join(', ') || 'N/A';
        return (
          <div className="text-sm font-medium text-gray-900">
            {surnames}
          </div>
        );
      }
    },
    {
      key: 'checkinStatus',
      title: 'Check-in Durumu',
      render: (_, record) => {
        const { status, timeStatus } = getCheckinStatusWithTime(record);

        return (
          <div className="space-y-1">
            {renderStatusBadge(status, {
              waiting: { color: 'bg-gray-100 text-gray-800', label: 'Bekliyor' },
              ready: { color: 'bg-blue-100 text-blue-800', label: 'Hazır' },
              processing: { color: 'bg-yellow-100 text-yellow-800', label: 'İşleniyor' },
              completed: { color: 'bg-green-100 text-green-800', label: 'Tamamlandı' },
              failed: { color: 'bg-red-100 text-red-800', label: 'Başarısız' },
              expired: { color: 'bg-orange-100 text-orange-800', label: 'Süresi Geçti' }
            })}
            {timeStatus && (
              <div className="text-xs text-gray-500 mt-1">{timeStatus}</div>
            )}
          </div>
        );
      }
    },
    {
      key: 'flight',
      title: 'Uçuş Bilgileri',
      render: (_, record) => {
        const departureDate = record.departureDate ? new Date(record.departureDate) : null;
        return (
          <div className="space-y-1">
            <div className="flex items-center space-x-2 text-sm font-medium text-gray-900">
              <span className="text-blue-600">{record.departureAirport || 'N/A'}</span>
              <span className="text-gray-400">→</span>
              <span className="text-blue-600">{record.arrivalAirport || 'N/A'}</span>
            </div>
            <div className="flex items-center space-x-3 text-xs text-gray-600">
              {record.airline && (
                <span className="px-2 py-0.5 bg-gray-100 rounded font-medium">{record.airline}</span>
              )}
              {record.flightNumber && record.flightNumber !== 'N/A' && (
                <span className="font-mono">{record.flightNumber}</span>
              )}
            </div>
            {departureDate && (
              <div className="text-xs text-gray-500">
                {departureDate.toLocaleDateString('tr-TR')} {record.departureTime || ''}
              </div>
            )}
          </div>
        );
      }
    },
    {
      key: 'boardingPass',
      title: 'Biniş Kartı & Bildirim',
      render: (_, record) => {
        // Havayoluna göre hangi bildirimlerin kullanıldığını belirle
        const usesEmail = record.airline === 'AJET' || record.airline === 'PC';
        const usesSMS = record.airline === 'AJET' || record.airline === 'THY';

        return (
          <div className="space-y-2">
            {/* Biniş Kartı Durumu */}
            {record.boardingPassUrl ? (
              <a
                href={record.boardingPassUrl}
                target="_blank"
                rel="noopener noreferrer"
                className="inline-flex items-center px-2 py-1 text-xs font-medium text-green-700 bg-green-50 rounded hover:bg-green-100 transition-colors"
              >
                ✓ Biniş Kartı
              </a>
            ) : (
              <span className="inline-flex items-center px-2 py-1 text-xs font-medium text-gray-500 bg-gray-50 rounded">
                - Biniş Kartı
              </span>
            )}

            {/* E-posta Durumu (AJET, Pegasus) */}
            {usesEmail && (
              <div className="flex items-center space-x-1">
                <span className="text-xs text-gray-500">📧</span>
                {renderStatusBadge(record.emailStatus || 'pending', {
                  sent: { color: 'bg-green-100 text-green-800', label: 'Gönderildi' },
                  failed: { color: 'bg-red-100 text-red-800', label: 'Başarısız' },
                  pending: { color: 'bg-gray-100 text-gray-800', label: 'Bekliyor' }
                })}
              </div>
            )}

            {/* SMS Durumu (AJET, THY) */}
            {usesSMS && (
              <div className="flex items-center space-x-1">
                <span className="text-xs text-gray-500">📱</span>
                {renderStatusBadge(record.smsStatus, {
                  sent: { color: 'bg-green-100 text-green-800', label: 'Gönderildi' },
                  failed: { color: 'bg-red-100 text-red-800', label: 'Başarısız' },
                  pending: { color: 'bg-gray-100 text-gray-800', label: 'Bekliyor' }
                })}
              </div>
            )}
          </div>
        );
      }
    }
  ];

  // Check-in başlatma/durdurma fonksiyonları
  const handleStartCheckin = async (record: CheckinTicket) => {
    try {
      console.log(`🚀 Check-in başlatılıyor: ${record.pnr}`);

      // Durumu processing'e güncelle
      setTickets(prevTickets =>
        prevTickets.map(t =>
          t.id === record.id
            ? { ...t, checkinStatus: 'processing' as const, checkinAttempts: t.checkinAttempts + 1 }
            : t
        )
      );

      // API çağrısı
      const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002'}/api/checkin`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          pnr: record.pnr,
          lastName: record.passengers[0]?.lastName || '',
          phone: record.passengers[0]?.phone || '',
          airline: record.airline
        }),
      });

      if (response.ok) {
        const result = await response.json();
        console.log('Check-in başlatıldı:', result);
      } else {
        throw new Error('Check-in başlatılamadı');
      }

    } catch (error) {
      console.error(`❌ Check-in başlatma hatası: ${record.pnr}`, error);

      // Hata durumunda failed olarak işaretle
      setTickets(prevTickets =>
        prevTickets.map(t =>
          t.id === record.id
            ? {
                ...t,
                checkinStatus: 'failed' as const,
                updatedAt: new Date()
              }
            : t
        )
      );
    }
  };

  const handleStopCheckin = async (record: CheckinTicket) => {
    try {
      console.log(`⏹️ Check-in durduruluyor: ${record.pnr}`);

      // ticketRequestId veya id kullan (backend jobId bekliyor)
      const jobId = record.ticketRequestId || record.id;

      // API çağrısı - check-in işlemini iptal et
      const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002'}/api/checkin/${jobId}`, {
        method: 'DELETE'
      });

      if (response.ok) {
        const result = await response.json();
        console.log('Check-in durdurma sonucu:', result);

        // Durumu waiting'e geri çevir
        setTickets(prevTickets =>
          prevTickets.map(t =>
            t.id === record.id
              ? { ...t, checkinStatus: 'waiting' as const, updatedAt: new Date() }
              : t
          )
        );
        console.log(`⏹️ Check-in durduruldu: ${record.pnr}`);
      } else {
        const errorText = await response.text();
        console.error('Durdurma hatası:', errorText);
        throw new Error(`Check-in durdurulamadı: ${response.status}`);
      }

    } catch (error) {
      console.error(`❌ Check-in durdurma hatası: ${record.pnr}`, error);
    }
  };

  const handleDeleteCheckin = async (record: CheckinTicket) => {
    if (!confirm(`"${record.pnr} - ${record.airline} ${record.flightNumber}" check-in kaydını silmek istediğinizden emin misiniz?\n\nBu işlem geri alınamaz.`)) {
      return;
    }

    try {
      console.log(`🗑️ Check-in kaydı siliniyor: ${record.pnr}`);

      // İlk olarak işlem devam ediyorsa durdur
      if (record.checkinStatus === 'processing') {
        await handleStopCheckin(record);
        // Kısa bir gecikme ile işlemin durmasını bekle
        await new Promise(resolve => setTimeout(resolve, 1000));
      }

      // Backend'den sil
      try {
        await checkinApi.deleteCheckinTicket(record.id);
        console.log(`✅ Backend'den silindi: ${record.id}`);
      } catch (apiError) {
        console.error('Backend silme hatası:', apiError);
        // Backend hatası olsa bile devam et (localStorage'dan silmek için)
      }

      // Kaydı localStorage'dan ve state'den sil
      setTickets(prevTickets => {
        const filteredTickets = prevTickets.filter(t => t.id !== record.id);
        // localStorage'ı da güncelle
        localStorage.setItem('checkin-tickets', JSON.stringify(filteredTickets));
        return filteredTickets;
      });

      console.log(`✅ Check-in kaydı silindi: ${record.pnr}`);

    } catch (error) {
      console.error(`❌ Check-in silme hatası: ${record.pnr}`, error);
      alert('Check-in kaydı silinirken bir hata oluştu. Lütfen tekrar deneyin.');
    }
  };

  // Tablo aksiyonları
  const actions: TableAction<CheckinTicket>[] = [
    {
      key: 'start',
      label: 'Check-in Başlat',
      icon: Play,
      onClick: (record) => handleStartCheckin(record),
      variant: 'success',
      condition: (record) => ['waiting', 'ready', 'failed'].includes(record.checkinStatus)
    },
    {
      key: 'stop',
      label: 'Durdur',
      icon: Square,
      onClick: (record) => handleStopCheckin(record),
      variant: 'warning',
      condition: (record) => record.checkinStatus === 'processing'
    },
    {
      key: 'view',
      label: 'Detay',
      icon: Eye,
      onClick: (record) => window.location.href = `/checkin/${record.id}`,
      variant: 'primary'
    },
    {
      key: 'retry',
      label: 'Tekrar Dene',
      icon: RotateCcw,
      onClick: (record) => handleRetryCheckin(record.id),
      variant: 'success',
      condition: (record) => record.checkinStatus === 'failed'
    },
    {
      key: 'delete',
      label: 'Sil',
      icon: Trash2,
      onClick: (record) => handleDeleteCheckin(record),
      variant: 'danger'
    }
  ];

  // Filtreler
  const filters = [
    {
      key: 'status',
      label: 'Durum',
      value: statusFilter,
      onChange: setStatusFilter,
      options: [
        { value: 'all', label: 'Tüm Durumlar' },
        { value: 'waiting', label: 'Bekliyor' },
        { value: 'ready', label: 'Hazır' },
        { value: 'processing', label: 'İşleniyor' },
        { value: 'completed', label: 'Tamamlandı' },
        { value: 'failed', label: 'Başarısız' },
        { value: 'expired', label: 'Süresi Geçti' }
      ]
    },
    {
      key: 'airline',
      label: 'Havayolu',
      value: airlineFilter,
      onChange: setAirlineFilter,
      options: [
        { value: 'all', label: 'Tüm Havayolları' },
        { value: 'THY', label: 'THY' },
        { value: 'PC', label: 'Pegasus' },
        { value: 'AJET', label: 'AnadoluJet' }
      ]
    }
  ];

  // Filtrelenmiş ve sıralanmış biletler
  // Helper: Check-in durumu hesapla (24 saat kontrolü)
  const getCheckinAvailability = (ticket: CheckinTicket) => {
    const now = new Date();
    const flightDate = new Date(ticket.departureDate);

    // Uçuş saatini ekle
    if (ticket.departureTime) {
      const [hours, minutes] = ticket.departureTime.split(':');
      flightDate.setHours(parseInt(hours), parseInt(minutes), 0, 0);
    }

    const diffHours = (flightDate.getTime() - now.getTime()) / (1000 * 60 * 60);

    if (diffHours <= 2) return 'closed'; // Check-in kapandı
    if (diffHours <= 24) return 'open'; // Check-in açık
    return 'waiting'; // Henüz açılmadı (24 saatten fazla)
  };

  const filteredTickets = tickets.filter(ticket => {
    const customerName = ticket.customer?.type === 'individual'
      ? `${ticket.customer?.firstName || ''} ${ticket.customer?.lastName || ''}`
      : ticket.customer?.companyName || '';

    const passengerNames = ticket.passengers?.map(p => `${p.firstName} ${p.lastName}`).join(' ') || '';

    const requestedByName = ticket.requestedBy
      ? `${ticket.requestedBy.firstName || ''} ${ticket.requestedBy.lastName || ''}`
      : '';

    const matchesSearch =
      customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      passengerNames.toLowerCase().includes(searchTerm.toLowerCase()) ||
      requestedByName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      ticket.flightNumber?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      ticket.pnr?.toLowerCase().includes(searchTerm.toLowerCase());

    const matchesStatus = statusFilter === 'all' || ticket.checkinStatus === statusFilter;
    const matchesAirline = airlineFilter === 'all' || ticket.airline === airlineFilter;

    return matchesSearch && matchesStatus && matchesAirline;
  }).sort((a, b) => {
    // Akıllı sıralama: Durum → Check-in açık mı → Tarih yakınlığı

    const now = new Date().getTime();
    const dateA = new Date(a.departureDate).getTime();
    const dateB = new Date(b.departureDate).getTime();

    // Uçuş saatlerini ekle
    if (a.departureTime) {
      const [hours, minutes] = a.departureTime.split(':');
      const flightDateA = new Date(a.departureDate);
      flightDateA.setHours(parseInt(hours), parseInt(minutes), 0, 0);
      const dateAWithTime = flightDateA.getTime();
    }
    if (b.departureTime) {
      const [hours, minutes] = b.departureTime.split(':');
      const flightDateB = new Date(b.departureDate);
      flightDateB.setHours(parseInt(hours), parseInt(minutes), 0, 0);
      const dateBWithTime = flightDateB.getTime();
    }

    const availabilityA = getCheckinAvailability(a);
    const availabilityB = getCheckinAvailability(b);

    // 1. Öncelik: Tamamlananlar ve başarısızlar en altta
    const statusPriorityA = a.checkinStatus === 'completed' ? 1000 :
                            a.checkinStatus === 'failed' ? 1001 : 0;
    const statusPriorityB = b.checkinStatus === 'completed' ? 1000 :
                            b.checkinStatus === 'failed' ? 1001 : 0;

    if (statusPriorityA !== statusPriorityB) {
      return statusPriorityA - statusPriorityB;
    }

    // 2. Öncelik: Check-in açık olanlar en üstte (24 saat içinde)
    if (availabilityA === 'open' && availabilityB !== 'open') return -1;
    if (availabilityA !== 'open' && availabilityB === 'open') return 1;

    // 3. Öncelik: İşlem devam edenler (processing)
    if (a.checkinStatus === 'processing' && b.checkinStatus !== 'processing') return -1;
    if (a.checkinStatus !== 'processing' && b.checkinStatus === 'processing') return 1;

    // 4. Tarih yakınlığına göre sırala
    // Gelecekteki uçuşlar: en yakın tarih önce
    if (dateA >= now && dateB >= now) {
      return dateA - dateB;
    }

    // Geçmiş uçuşlar: en yakın tarih önce
    if (dateA < now && dateB < now) {
      return dateB - dateA;
    }

    // Biri gelecek biri geçmiş: gelecek öncelik
    return dateA >= now ? -1 : 1;
  });

  // Kanban için sütunlar
  const kanbanColumns = [
    { id: 'waiting', title: 'Bekliyor', color: 'bg-gray-500', count: 0 },
    { id: 'ready', title: 'Hazır', color: 'bg-blue-500', count: 0 },
    { id: 'processing', title: 'İşleniyor', color: 'bg-yellow-500', count: 0 },
    { id: 'completed', title: 'Tamamlandı', color: 'bg-green-500', count: 0 },
    { id: 'failed', title: 'Başarısız', color: 'bg-red-500', count: 0 },
    { id: 'expired', title: 'Süresi Geçti', color: 'bg-orange-500', count: 0 },
    { id: 'cancelled', title: 'İptal', color: 'bg-gray-600', count: 0 }
  ];

  // Kanban kartları
  const kanbanCards = filteredTickets.map(ticket => ({
    id: ticket.id,
    title: `${ticket.airline} ${ticket.flightNumber || ''}`,
    subtitle: `${ticket.departureAirport} → ${ticket.arrivalAirport}`,
    customer: ticket.customer,
    status: ticket.checkinStatus,
    date: ticket.departureDate,
    tags: [
      `PNR: ${ticket.pnr}`,
      ticket.passengers?.length ? `${ticket.passengers.length} yolcu` : null,
      ticket.departureTime,
      ticket.smsStatus === 'sent' ? 'SMS Gönderildi' :
      ticket.smsStatus === 'failed' ? 'SMS Başarısız' : 'SMS Bekliyor'
    ].filter(Boolean),
    onClick: () => window.location.href = `/checkin/${ticket.id}`,
    onEdit: () => window.location.href = `/checkin/${ticket.id}/edit`
  }));

  // Manuel check-in form handler
  // Check-in zamanı gelmiş biletleri kontrol et ve işle
  const checkAndProcessReadyTickets = async () => {
    const now = new Date();
    
    tickets.forEach(async (ticket) => {
      if (ticket.checkinStatus === 'waiting') {
        const checkinTime = new Date(ticket.departureDate);
        checkinTime.setHours(checkinTime.getHours() - 24); // 24 saat öncesi
        
        if (now >= checkinTime && ticket.isAutoCheckinEnabled) {
          console.log(`🕐 Check-in zamanı geldi: ${ticket.pnr} - ${ticket.airline} ${ticket.flightNumber}`);
          
          // Durumu ready'ye güncelle
          setTickets(prevTickets => 
            prevTickets.map(t => 
              t.id === ticket.id 
                ? { ...t, checkinStatus: 'ready' as const }
                : t
            )
          );
          
          // Auto check-in başlat
          await handleAutoCheckin(ticket);
        }
      }
    });
  };
  
  // Otomatik check-in işlemi
  const handleAutoCheckin = async (ticket: CheckinTicket) => {
    try {
      console.log(`🚀 Otomatik check-in başlatılıyor: ${ticket.pnr}`);
      
      // Durumu processing'e güncelle
      setTickets(prevTickets => 
        prevTickets.map(t => 
          t.id === ticket.id 
            ? { ...t, checkinStatus: 'processing' as const, checkinAttempts: t.checkinAttempts + 1 }
            : t
        )
      );
      
      // API çağrısı (Python script veya Node.js service)
      const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002'}/api/checkin`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          pnr: ticket.pnr,
          lastName: ticket.passengers[0]?.lastName || '',
          phone: ticket.passengers[0]?.phone || '',
          airline: ticket.airline
        }),
      });
      
      if (response.ok) {
        const result = await response.json();
        
        // Başarılı check-in
        setTickets(prevTickets => 
          prevTickets.map(t => 
            t.id === ticket.id 
              ? { 
                  ...t, 
                  checkinStatus: 'completed' as const,
                  smsStatus: result.smsSent ? 'sent' as const : 'pending' as const,
                  updatedAt: new Date()
                }
              : t
          )
        );
        
        console.log(`✅ Check-in başarılı: ${ticket.pnr}`);
      } else {
        throw new Error('Check-in API hatası');
      }
      
    } catch (error) {
      console.error(`❌ Check-in hatası: ${ticket.pnr}`, error);
      
      // Hata durumunda failed olarak işaretle
      setTickets(prevTickets => 
        prevTickets.map(t => 
          t.id === ticket.id 
            ? { 
                ...t, 
                checkinStatus: 'failed' as const,
                updatedAt: new Date()
              }
            : t
        )
      );
    }
  };
  
  const handleRetryCheckin = async (ticketId: string) => {
    const ticket = tickets.find(t => t.id === ticketId);
    if (ticket) {
      console.log('Manual retry check-in for:', ticketId);
      await handleAutoCheckin(ticket);
    }
  };
  
  // Check-in durumunu kontrol et ve badge rengini belirle
  const getCheckinStatusWithTime = (ticket: CheckinTicket) => {
    const now = new Date();
    const flightDate = new Date(ticket.departureDate);

    // Uçuş saatini ekle
    if (ticket.departureTime) {
      const [hours, minutes] = ticket.departureTime.split(':');
      flightDate.setHours(parseInt(hours), parseInt(minutes), 0, 0);
    }

    const diffMs = flightDate.getTime() - now.getTime();
    const diffHours = diffMs / (1000 * 60 * 60);

    // Tamamlanmış veya başarısız işlemler
    if (ticket.checkinStatus === 'completed') {
      return { status: 'completed', timeStatus: '✓ Check-in yapıldı' };
    }
    if (ticket.checkinStatus === 'failed') {
      return { status: 'failed', timeStatus: '✗ Başarısız oldu' };
    }

    // İşlem devam ediyor
    if (ticket.checkinStatus === 'processing') {
      return { status: 'processing', timeStatus: '⏳ İşleniyor...' };
    }

    // Check-in penceresi kontrolleri
    if (diffHours <= 2) {
      // 2 saat kala kapandı
      return { status: 'expired', timeStatus: '⛔ Check-in kapandı (2 saat içinde)' };
    } else if (diffHours <= 24) {
      // 24 saat içinde - Check-in açık!
      const hours = Math.floor(diffHours);
      const minutes = Math.floor((diffHours - hours) * 60);
      return {
        status: 'ready',
        timeStatus: `🟢 Check-in AÇIK! (Uçuşa ${hours}s ${minutes}dk)`
      };
    } else if (diffHours <= 48) {
      // 24-48 saat arası - Yakında
      const hours = Math.floor(diffHours);
      return {
        status: 'waiting',
        timeStatus: `⏱️ Yakında açılacak (${hours} saat sonra)`
      };
    } else {
      // 48 saatten fazla - Beklemede
      const days = Math.floor(diffHours / 24);
      return {
        status: 'waiting',
        timeStatus: `⏳ Beklemede (${days} gün sonra)`
      };
    }
  };

  const handleManualSubmit = async (formData: {
    airline: string;
    passengerCount: number;
    passengers: Array<{
      id: number;
      pnrOrTicket: string;
      lastName: string;
      // firstName backend'den gelecek
    }>;
    flightDate: string;
    flightTime: string;
    phone: string;
    email: string;
    immediate: boolean;
  }) => {
    setSubmitLoading(true);

    try {
      // Her yolcu için ayrı check-in isteği gönder
      const checkinPromises = formData.passengers.map(async (passenger, index) => {
        const response = await fetch(`${process.env.NEXT_PUBLIC_API_URL || 'http://localhost:3002'}/api/checkin`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            pnr: passenger.pnrOrTicket,
            lastName: passenger.lastName,
            phone: formData.phone,
            email: formData.email,
            airline: formData.airline,
            flightDate: formData.flightDate,
            flightTime: formData.flightTime
            // firstName backend'den Python script çalıştırdıktan sonra gelecek
          }),
        });

        if (response.ok) {
          const result = await response.json();
          console.log(`Yolcu ${index + 1} check-in başlatıldı:`, result);
          return { ...result, passenger };
        } else {
          throw new Error(`Yolcu ${index + 1} için check-in başlatılamadı`);
        }
      });

      const results = await Promise.all(checkinPromises);

      // Başarılı ise modal'ı kapat
      setShowManualForm(false);

      // Her yolcu için yeni ticket'i listeye ekle
      results.forEach((result, index) => {
        const passenger = result.passenger;
        const newTicket: CheckinTicket = {
          id: result.jobId || Date.now().toString() + index,
          ticketRequestId: result.jobId || `tr${Date.now()}_${index}`,
          pnr: passenger.pnrOrTicket,
          customerId: 'manual_user',
          customer: {
            id: 'manual_user',
            type: 'individual',
            firstName: 'Manuel',
            lastName: passenger.lastName,
            phone: formData.phone || '+905551234567',
            email: 'manuel@email.com',
            isVip: false,
            isActive: true,
            createdAt: new Date(),
            updatedAt: new Date()
          },
          passengers: [
            {
              id: `p${Date.now()}`,
              firstName: 'Yolcu',
              lastName: passenger.lastName,
              phone: formData.phone || '+905551234567',
              email: 'yolcu@email.com'
            }
          ],
          airline: formData.airline,
          flightNumber: 'Manual',
          departureAirport: 'IST',
          arrivalAirport: 'ESB',
          departureDate: formData.flightDate ? new Date(formData.flightDate) : new Date(),
          departureTime: formData.flightTime || '14:30',
          checkinStatus: 'processing', // Backend'de işlem başladı
          checkinOpenTime: new Date(),
          checkinAttempts: 1,
          passengerResults: [
            {
              passengerId: `p${Date.now()}`,
              checkinStatus: 'processing'
            }
          ],
          isAutoCheckinEnabled: true,
          smsStatus: 'pending',
          createdAt: new Date(),
          updatedAt: new Date()
        };

        // Listeye ekle
        setTickets(prevTickets => [newTicket, ...prevTickets]);
      });
    } catch (error) {
      console.error('Manuel check-in hatası:', error);
      // TODO: Show error message to user
    } finally {
      setSubmitLoading(false);
    }
  };

  // Header actions with both view toggle and add button
  const headerActions = (
    <div className="flex items-center space-x-3">
      <button
        onClick={() => setShowManualForm(true)}
        className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
      >
        <Plus className="h-4 w-4 mr-2" />
        Manuel Check-in Ekle
      </button>
      <div className="flex items-center bg-gray-100 rounded-lg p-1">
        <button
          onClick={() => setViewMode('table')}
          className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${
            viewMode === 'table'
              ? 'bg-white text-gray-900 shadow-sm'
              : 'text-gray-600 hover:text-gray-900'
          }`}
        >
          <List className="h-4 w-4 mr-2" />
          Tablo
        </button>
        <button
          onClick={() => setViewMode('kanban')}
          className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${
            viewMode === 'kanban'
              ? 'bg-white text-gray-900 shadow-sm'
              : 'text-gray-600 hover:text-gray-900'
          }`}
        >
          <LayoutGrid className="h-4 w-4 mr-2" />
          Kanban
        </button>
      </div>
    </div>
  );

  // Stats hesaplama
  const stats = {
    waiting: tickets.filter(t => t.checkinStatus === 'waiting' || t.checkinStatus === 'ready').length,
    completed: tickets.filter(t => t.checkinStatus === 'completed').length,
    failed: tickets.filter(t => t.checkinStatus === 'failed').length,
    successRate: tickets.length > 0 
      ? Math.round((tickets.filter(t => t.checkinStatus === 'completed').length / tickets.length) * 100)
      : 0
  };

  if (viewMode === 'kanban') {
    return (
      <div className="p-6">
        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
          <div className="bg-white p-4 rounded-lg border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Bekleyen</p>
                <p className="text-2xl font-bold text-gray-900">{stats.waiting}</p>
              </div>
              <Clock className="h-8 w-8 text-gray-400" />
            </div>
          </div>
          
          <div className="bg-white p-4 rounded-lg border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Tamamlanan</p>
                <p className="text-2xl font-bold text-green-600">{stats.completed}</p>
              </div>
              <CheckCircle className="h-8 w-8 text-green-400" />
            </div>
          </div>
          
          <div className="bg-white p-4 rounded-lg border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Başarısız</p>
                <p className="text-2xl font-bold text-red-600">{stats.failed}</p>
              </div>
              <XCircle className="h-8 w-8 text-red-400" />
            </div>
          </div>
          
          <div className="bg-white p-4 rounded-lg border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Başarı Oranı</p>
                <p className="text-2xl font-bold text-blue-600">{stats.successRate}%</p>
              </div>
              <div className="h-8 w-8 bg-blue-100 rounded-full flex items-center justify-center">
                <span className="text-blue-600 font-bold text-sm">%</span>
              </div>
            </div>
          </div>
        </div>

        {/* Header */}
        <div className="flex justify-between items-center mb-6">
          <div>
            <h1 className="text-2xl font-bold text-gray-900 flex items-center">
              <UserCheck className="mr-3 h-8 w-8 text-blue-600" />
              Check-in Listesi
            </h1>
            <p className="text-gray-600 mt-1">
              Biletlenmiş uçuşların otomatik check-in durumunu takip edin
            </p>
          </div>
          <div className="flex items-center space-x-4">
            {headerActions}
          </div>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <div className="relative">
                <input
                  type="text"
                  placeholder="Müşteri adı, yolcu adı, uçuş numarası veya PNR ara..."
                  className="w-full pl-4 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                />
              </div>
            </div>
            <div className="sm:w-48">
              <select
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                value={statusFilter}
                onChange={(e) => setStatusFilter(e.target.value)}
              >
                {filters[0].options.map(option => (
                  <option key={option.value} value={option.value}>
                    {option.label}
                  </option>
                ))}
              </select>
            </div>
            <div className="sm:w-48">
              <select
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                value={airlineFilter}
                onChange={(e) => setAirlineFilter(e.target.value)}
              >
                {filters[1].options.map(option => (
                  <option key={option.value} value={option.value}>
                    {option.label}
                  </option>
                ))}
              </select>
            </div>
          </div>
        </div>

        {/* Kanban Board */}
        <KanbanBoard
          columns={kanbanColumns}
          cards={kanbanCards}
          statusConfig={checkinStatusConfig}
          onCardClick={(card) => window.location.href = `/checkin/${card.id}`}
          onCardEdit={(card) => window.location.href = `/checkin/${card.id}/edit`}
          onStatusChange={(cardId, newStatus) => {
            // Check-in için özel durum yönetimi
            if (newStatus === 'processing') {
              handleRetryCheckin(cardId);
            }
          }}
          loading={loading}
        />
      </div>
    );
  }

  return (
    <div className="p-6">
      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <div className="bg-white p-4 rounded-lg border border-gray-200">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Bekleyen</p>
              <p className="text-2xl font-bold text-gray-900">{stats.waiting}</p>
            </div>
            <Clock className="h-8 w-8 text-gray-400" />
          </div>
        </div>
        
        <div className="bg-white p-4 rounded-lg border border-gray-200">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Tamamlanan</p>
              <p className="text-2xl font-bold text-green-600">{stats.completed}</p>
            </div>
            <CheckCircle className="h-8 w-8 text-green-400" />
          </div>
        </div>
        
        <div className="bg-white p-4 rounded-lg border border-gray-200">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Başarısız</p>
              <p className="text-2xl font-bold text-red-600">{stats.failed}</p>
            </div>
            <XCircle className="h-8 w-8 text-red-400" />
          </div>
        </div>
        
        <div className="bg-white p-4 rounded-lg border border-gray-200">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Başarı Oranı</p>
              <p className="text-2xl font-bold text-blue-600">{stats.successRate}%</p>
            </div>
            <div className="h-8 w-8 bg-blue-100 rounded-full flex items-center justify-center">
              <span className="text-blue-600 font-bold text-sm">%</span>
            </div>
          </div>
        </div>
      </div>

      <DataTable
        title="Check-in Listesi"
        description="Biletlenmiş uçuşların otomatik check-in durumunu takip edin"
        data={filteredTickets}
        columns={columns}
        actions={actions}
        loading={loading}
        searchable={true}
        searchPlaceholder="Müşteri adı, yolcu adı, uçuş numarası veya PNR ara..."
        searchValue={searchTerm}
        onSearchChange={setSearchTerm}
        filters={filters}
        headerActions={headerActions}
        emptyTitle="Check-in bileti bulunamadı"
        emptyDescription="Henüz check-in için bilet bulunmuyor."
        emptyIcon={UserCheck}
      />

      {/* Manuel Check-in Modal */}
      {showManualForm && (
        <ManualCheckinForm
          onSubmit={handleManualSubmit}
          onCancel={() => setShowManualForm(false)}
          loading={submitLoading}
        />
      )}
    </div>
  );
}