'use client';

import React, { useState, useEffect } from 'react';
import {
    CreditCard,
    Plus,
    Eye,
    Edit,
    Download,
    TrendingUp,
    DollarSign,
    FileText,
    Clock,
    LayoutGrid,
    List
} from 'lucide-react';
import Link from 'next/link';
import DataTable, { type TableColumn, type TableAction } from '@/components/ui/DataTable';
import KanbanBoard from '@/components/ui/KanbanBoard';
import {
    renderCustomer,
    renderStatusBadge,
    renderCurrency,
    renderDateTime,
    renderServiceType,
    invoiceStatusConfig,
    paymentStatusConfig
} from '@/components/ui/TableRenderers';
import PermissionGuard from '@/components/guards/PermissionGuard';

interface AccountingRecord {
    id: string;
    customerId: string;
    customer: {
        id: string;
        type: 'individual' | 'corporate';
        firstName?: string;
        lastName?: string;
        companyName?: string;
        phone: string;
        email?: string;
        isVip: boolean;
        isActive: boolean;
        createdAt: Date;
        updatedAt: Date;
    };
    serviceType: 'ticket' | 'accommodation' | 'visa' | 'transfer';
    serviceDetails: string;
    amount: number;
    currency: 'TRY' | 'USD' | 'EUR';
    invoiceStatus: 'not_invoiced' | 'invoiced' | 'paid';
    paymentStatus: 'unpaid' | 'partial' | 'paid' | 'refunded';
    invoiceNumber?: string;
    serviceDate: Date;
    createdAt: Date;
}

export default function AccountingPage() {
  return (
    <PermissionGuard requiredPermissions={['accounting_view']}>
      <AccountingPageContent />
    </PermissionGuard>
  );
}

function AccountingPageContent() {
    const [records, setRecords] = useState<AccountingRecord[]>([]);
    const [loading, setLoading] = useState(true);
    const [searchTerm, setSearchTerm] = useState('');
    const [serviceFilter, setServiceFilter] = useState<string>('all');
    const [statusFilter, setStatusFilter] = useState<string>('all');
    const [viewMode, setViewMode] = useState<'table' | 'kanban'>('table');

    useEffect(() => {
        fetchAccountingRecords();
    }, []);

    const fetchAccountingRecords = async () => {
        try {
            setLoading(true);

            // Mock data for now
            const mockRecords: AccountingRecord[] = [
                {
                    id: '1',
                    customerId: 'c1',
                    customer: {
                        id: 'c1',
                        type: 'individual',
                        firstName: 'Ahmet',
                        lastName: 'Yılmaz',
                        phone: '+905551234567',
                        email: 'ahmet@email.com',
                        isVip: false,
                        isActive: true,
                        createdAt: new Date(),
                        updatedAt: new Date()
                    },
                    serviceType: 'ticket',
                    serviceDetails: 'THY TK1234 - IST → ADB',
                    amount: 850,
                    currency: 'TRY',
                    invoiceStatus: 'invoiced',
                    paymentStatus: 'paid',
                    invoiceNumber: 'INV-2025-001',
                    serviceDate: new Date('2025-01-15'),
                    createdAt: new Date('2025-01-10')
                },
                {
                    id: '2',
                    customerId: 'c2',
                    customer: {
                        id: 'c2',
                        type: 'corporate',
                        companyName: 'ABC Teknoloji A.Ş.',
                        phone: '+905559876543',
                        email: 'info@abcteknoloji.com',
                        isVip: true,
                        isActive: true,
                        createdAt: new Date(),
                        updatedAt: new Date()
                    },
                    serviceType: 'accommodation',
                    serviceDetails: 'Bodrum Bay Resort - 4 gece',
                    amount: 9200,
                    currency: 'TRY',
                    invoiceStatus: 'invoiced',
                    paymentStatus: 'unpaid',
                    invoiceNumber: 'INV-2025-002',
                    serviceDate: new Date('2025-02-01'),
                    createdAt: new Date('2025-01-12')
                },
                {
                    id: '3',
                    customerId: 'c3',
                    customer: {
                        id: 'c3',
                        type: 'individual',
                        firstName: 'Mehmet',
                        lastName: 'Özkan',
                        phone: '+905557891234',
                        email: 'mehmet@email.com',
                        isVip: false,
                        isActive: true,
                        createdAt: new Date(),
                        updatedAt: new Date()
                    },
                    serviceType: 'visa',
                    serviceDetails: 'ABD İş Vizesi',
                    amount: 660,
                    currency: 'USD',
                    invoiceStatus: 'not_invoiced',
                    paymentStatus: 'unpaid',
                    serviceDate: new Date('2025-03-15'),
                    createdAt: new Date('2025-01-10')
                }
            ];

            setRecords(mockRecords);
        } catch (error) {
            console.error('Error fetching accounting records:', error);
        } finally {
            setLoading(false);
        }
    };

    const handleStatusChange = async (recordId: string, newStatus: string, type: 'invoice' | 'payment') => {
        try {
            console.log('Status change:', recordId, newStatus, type);

            setRecords(prev => prev.map(record =>
                record.id === recordId
                    ? {
                        ...record,
                        [type === 'invoice' ? 'invoiceStatus' : 'paymentStatus']: newStatus,
                        ...(type === 'invoice' && newStatus === 'invoiced' && !record.invoiceNumber ? {
                            invoiceNumber: `INV-2025-${String(Date.now()).slice(-3)}`
                        } : {})
                    }
                    : record
            ));
        } catch (error) {
            console.error('Error updating status:', error);
        }
    };

    // Tablo sütunları
    const columns: TableColumn<AccountingRecord>[] = [
        {
            key: 'customer',
            title: 'Müşteri',
            render: (_, record) => renderCustomer(record.customer)
        },
        {
            key: 'service',
            title: 'Hizmet',
            render: (_, record) => renderServiceType(record.serviceType, record.serviceDetails)
        },
        {
            key: 'amount',
            title: 'Tutar',
            render: (_, record) => renderCurrency(record.amount, record.currency)
        },
        {
            key: 'serviceDate',
            title: 'Hizmet Tarihi',
            render: (_, record) => renderDateTime(record.serviceDate)
        },
        {
            key: 'invoiceStatus',
            title: 'Fatura Durumu',
            render: (_, record) => (
                <div>
                    {renderStatusBadge(record.invoiceStatus, invoiceStatusConfig)}
                    {record.invoiceNumber && (
                        <div className="text-xs text-blue-600 font-mono mt-1">
                            {record.invoiceNumber}
                        </div>
                    )}
                </div>
            )
        },
        {
            key: 'paymentStatus',
            title: 'Ödeme Durumu',
            render: (_, record) => renderStatusBadge(record.paymentStatus, paymentStatusConfig)
        }
    ];

    // Tablo aksiyonları
    const actions: TableAction<AccountingRecord>[] = [
        {
            key: 'view',
            label: 'Görüntüle',
            icon: Eye,
            onClick: (record) => console.log('View:', record.id),
            variant: 'primary'
        },
        {
            key: 'edit',
            label: 'Düzenle',
            icon: Edit,
            onClick: (record) => console.log('Edit:', record.id),
            variant: 'secondary'
        },
        {
            key: 'invoice',
            label: 'Faturala',
            icon: FileText,
            onClick: (record) => handleStatusChange(record.id, 'invoiced', 'invoice'),
            variant: 'success',
            condition: (record) => record.invoiceStatus === 'not_invoiced'
        },
        {
            key: 'mark_paid',
            label: 'Ödendi İşaretle',
            onClick: (record) => handleStatusChange(record.id, 'paid', 'payment'),
            variant: 'success',
            condition: (record) => record.paymentStatus === 'unpaid'
        },
        {
            key: 'download',
            label: 'Fatura İndir',
            icon: Download,
            onClick: (record) => console.log('Download invoice:', record.invoiceNumber),
            variant: 'secondary',
            condition: (record) => record.invoiceStatus === 'invoiced'
        }
    ];

    // Filtreler
    const filters = [
        {
            key: 'service',
            label: 'Hizmet Türü',
            value: serviceFilter,
            onChange: setServiceFilter,
            options: [
                { value: 'all', label: 'Tüm Hizmetler' },
                { value: 'ticket', label: 'Bilet' },
                { value: 'accommodation', label: 'Konaklama' },
                { value: 'visa', label: 'Vize' },
                { value: 'transfer', label: 'Transfer' }
            ]
        },
        {
            key: 'status',
            label: 'Ödeme Durumu',
            value: statusFilter,
            onChange: setStatusFilter,
            options: [
                { value: 'all', label: 'Tüm Durumlar' },
                { value: 'unpaid', label: 'Ödenmedi' },
                { value: 'paid', label: 'Ödendi' },
                { value: 'partial', label: 'Kısmi Ödendi' },
                { value: 'refunded', label: 'İade Edildi' }
            ]
        }
    ];

    // Filtrelenmiş kayıtlar
    const filteredRecords = records.filter(record => {
        const customerName = record.customer.type === 'individual'
            ? `${record.customer.firstName} ${record.customer.lastName}`
            : record.customer.companyName || '';

        const matchesSearch =
            customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
            record.serviceDetails.toLowerCase().includes(searchTerm.toLowerCase()) ||
            record.invoiceNumber?.toLowerCase().includes(searchTerm.toLowerCase());

        const matchesService = serviceFilter === 'all' || record.serviceType === serviceFilter;
        const matchesStatus = statusFilter === 'all' || record.paymentStatus === statusFilter;

        return matchesSearch && matchesService && matchesStatus;
    });

    // Kanban için sütunlar
    const kanbanColumns = [
        { id: 'not_invoiced', title: 'Faturalanmadı', color: 'bg-gray-500', count: 0 },
        { id: 'invoiced_unpaid', title: 'Faturalı - Ödenmedi', color: 'bg-blue-500', count: 0 },
        { id: 'invoiced_partial', title: 'Faturalı - Kısmi', color: 'bg-yellow-500', count: 0 },
        { id: 'invoiced_paid', title: 'Faturalı - Ödendi', color: 'bg-green-500', count: 0 },
        { id: 'refunded', title: 'İade Edildi', color: 'bg-purple-500', count: 0 }
    ];

    // Kanban kartları için durum kombinasyonu
    const getCardStatus = (record: AccountingRecord) => {
        if (record.paymentStatus === 'refunded') return 'refunded';
        if (record.invoiceStatus === 'not_invoiced') return 'not_invoiced';
        if (record.invoiceStatus === 'invoiced') {
            if (record.paymentStatus === 'paid') return 'invoiced_paid';
            if (record.paymentStatus === 'partial') return 'invoiced_partial';
            return 'invoiced_unpaid';
        }
        return 'not_invoiced';
    };

    // Kanban kartları
    const kanbanCards = filteredRecords.map(record => ({
        id: record.id,
        title: record.serviceDetails,
        subtitle: record.customer.type === 'individual'
            ? `${record.customer.firstName} ${record.customer.lastName}`
            : record.customer.companyName || '',
        customer: record.customer,
        status: getCardStatus(record),
        amount: record.amount,
        currency: record.currency,
        date: record.serviceDate,
        tags: [
            record.serviceType === 'ticket' ? 'Bilet' :
                record.serviceType === 'accommodation' ? 'Konaklama' :
                    record.serviceType === 'visa' ? 'Vize' :
                        record.serviceType === 'transfer' ? 'Transfer' : record.serviceType,
            record.invoiceNumber || '',
            record.serviceDate.toLocaleDateString('tr-TR')
        ].filter(Boolean),
        onClick: () => console.log('View record:', record.id),
        onEdit: () => console.log('Edit record:', record.id)
    }));

    // View mode toggle
    const viewModeToggle = (
        <div className="flex items-center bg-gray-100 rounded-lg p-1">
            <button
                onClick={() => setViewMode('table')}
                className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${viewMode === 'table'
                    ? 'bg-white text-gray-900 shadow-sm'
                    : 'text-gray-600 hover:text-gray-900'
                    }`}
            >
                <List className="h-4 w-4 mr-2" />
                Tablo
            </button>
            <button
                onClick={() => setViewMode('kanban')}
                className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${viewMode === 'kanban'
                    ? 'bg-white text-gray-900 shadow-sm'
                    : 'text-gray-600 hover:text-gray-900'
                    }`}
            >
                <LayoutGrid className="h-4 w-4 mr-2" />
                Kanban
            </button>
        </div>
    );

    // İstatistikler
    const stats = {
        totalRevenue: records.reduce((sum, record) => sum + record.amount, 0),
        paidRevenue: records.filter(r => r.paymentStatus === 'paid').reduce((sum, record) => sum + record.amount, 0),
        unpaidRevenue: records.filter(r => r.paymentStatus === 'unpaid').reduce((sum, record) => sum + record.amount, 0),
        invoicedCount: records.filter(r => r.invoiceStatus === 'invoiced').length
    };

    if (viewMode === 'kanban') {
        return (
            <div className="p-6">
                {/* Stats Cards */}
                <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                    <div className="bg-white p-4 rounded-lg border border-gray-200">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600">Toplam Gelir</p>
                                <p className="text-2xl font-bold text-gray-900">
                                    {stats.totalRevenue.toLocaleString()} TRY
                                </p>
                            </div>
                            <TrendingUp className="h-8 w-8 text-gray-400" />
                        </div>
                    </div>

                    <div className="bg-white p-4 rounded-lg border border-gray-200">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600">Tahsil Edilen</p>
                                <p className="text-2xl font-bold text-green-600">
                                    {stats.paidRevenue.toLocaleString()} TRY
                                </p>
                            </div>
                            <DollarSign className="h-8 w-8 text-green-400" />
                        </div>
                    </div>

                    <div className="bg-white p-4 rounded-lg border border-gray-200">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600">Bekleyen Ödeme</p>
                                <p className="text-2xl font-bold text-red-600">
                                    {stats.unpaidRevenue.toLocaleString()} TRY
                                </p>
                            </div>
                            <Clock className="h-8 w-8 text-red-400" />
                        </div>
                    </div>

                    <div className="bg-white p-4 rounded-lg border border-gray-200">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600">Faturalanan</p>
                                <p className="text-2xl font-bold text-blue-600">{stats.invoicedCount}</p>
                            </div>
                            <FileText className="h-8 w-8 text-blue-400" />
                        </div>
                    </div>
                </div>

                {/* Header */}
                <div className="flex justify-between items-center mb-6">
                    <div>
                        <h1 className="text-2xl font-bold text-gray-900 flex items-center">
                            <CreditCard className="mr-3 h-8 w-8 text-blue-600" />
                            Muhasebe Kayıtları
                        </h1>
                        <p className="text-gray-600 mt-1">
                            Hizmet gelirleri ve fatura durumlarını yönetin
                        </p>
                    </div>
                    <div className="flex items-center space-x-4">
                        {viewModeToggle}
                        <Link
                            href="/accounting/new"
                            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
                        >
                            <Plus className="h-4 w-4" />
                            Yeni Kayıt
                        </Link>
                    </div>
                </div>

                {/* Filters */}
                <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
                    <div className="flex flex-col sm:flex-row gap-4">
                        <div className="flex-1">
                            <div className="relative">
                                <input
                                    type="text"
                                    placeholder="Müşteri adı, hizmet detayı veya fatura numarası ara..."
                                    className="w-full pl-4 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                                    value={searchTerm}
                                    onChange={(e) => setSearchTerm(e.target.value)}
                                />
                            </div>
                        </div>
                        <div className="sm:w-48">
                            <select
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                                value={serviceFilter}
                                onChange={(e) => setServiceFilter(e.target.value)}
                            >
                                {filters[0].options.map(option => (
                                    <option key={option.value} value={option.value}>
                                        {option.label}
                                    </option>
                                ))}
                            </select>
                        </div>
                        <div className="sm:w-48">
                            <select
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                                value={statusFilter}
                                onChange={(e) => setStatusFilter(e.target.value)}
                            >
                                {filters[1].options.map(option => (
                                    <option key={option.value} value={option.value}>
                                        {option.label}
                                    </option>
                                ))}
                            </select>
                        </div>
                    </div>
                </div>

                {/* Kanban Board */}
                <KanbanBoard
                    columns={kanbanColumns}
                    cards={kanbanCards}
                    statusConfig={{
                        not_invoiced: { color: 'bg-gray-100 text-gray-800', label: 'Faturalanmadı' },
                        invoiced_unpaid: { color: 'bg-red-100 text-red-800', label: 'Ödenmedi' },
                        invoiced_partial: { color: 'bg-yellow-100 text-yellow-800', label: 'Kısmi Ödendi' },
                        invoiced_paid: { color: 'bg-green-100 text-green-800', label: 'Ödendi' },
                        refunded: { color: 'bg-purple-100 text-purple-800', label: 'İade Edildi' }
                    }}
                    onCardClick={(card) => console.log('View record:', card.id)}
                    onCardEdit={(card) => console.log('Edit record:', card.id)}
                    onStatusChange={(cardId, newStatus) => {
                        // Muhasebe için özel durum yönetimi
                        console.log('Status change:', cardId, newStatus);

                        // Kanban'da durum değişikliği mantığı
                        if (newStatus === 'invoiced_unpaid') {
                            handleStatusChange(cardId, 'invoiced', 'invoice');
                        } else if (newStatus === 'invoiced_paid') {
                            handleStatusChange(cardId, 'paid', 'payment');
                        }
                    }}
                    loading={loading}
                />
            </div>
        );
    }

    return (
        <div className="p-6">
            {/* Stats Cards */}
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                <div className="bg-white p-4 rounded-lg border border-gray-200">
                    <div className="flex items-center justify-between">
                        <div>
                            <p className="text-sm text-gray-600">Toplam Gelir</p>
                            <p className="text-2xl font-bold text-gray-900">
                                {stats.totalRevenue.toLocaleString()} TRY
                            </p>
                        </div>
                        <TrendingUp className="h-8 w-8 text-gray-400" />
                    </div>
                </div>

                <div className="bg-white p-4 rounded-lg border border-gray-200">
                    <div className="flex items-center justify-between">
                        <div>
                            <p className="text-sm text-gray-600">Tahsil Edilen</p>
                            <p className="text-2xl font-bold text-green-600">
                                {stats.paidRevenue.toLocaleString()} TRY
                            </p>
                        </div>
                        <DollarSign className="h-8 w-8 text-green-400" />
                    </div>
                </div>

                <div className="bg-white p-4 rounded-lg border border-gray-200">
                    <div className="flex items-center justify-between">
                        <div>
                            <p className="text-sm text-gray-600">Bekleyen Ödeme</p>
                            <p className="text-2xl font-bold text-red-600">
                                {stats.unpaidRevenue.toLocaleString()} TRY
                            </p>
                        </div>
                        <Clock className="h-8 w-8 text-red-400" />
                    </div>
                </div>

                <div className="bg-white p-4 rounded-lg border border-gray-200">
                    <div className="flex items-center justify-between">
                        <div>
                            <p className="text-sm text-gray-600">Faturalanan</p>
                            <p className="text-2xl font-bold text-blue-600">{stats.invoicedCount}</p>
                        </div>
                        <FileText className="h-8 w-8 text-blue-400" />
                    </div>
                </div>
            </div>

            <DataTable
                title="Muhasebe Kayıtları"
                description="Hizmet gelirleri ve fatura durumlarını yönetin"
                data={filteredRecords}
                columns={columns}
                actions={actions}
                loading={loading}
                searchable={true}
                searchPlaceholder="Müşteri adı, hizmet detayı veya fatura numarası ara..."
                searchValue={searchTerm}
                onSearchChange={setSearchTerm}
                filters={filters}
                headerActions={
                    <div className="flex items-center space-x-4">
                        {viewModeToggle}
                        <Link
                            href="/accounting/new"
                            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
                        >
                            <Plus className="h-4 w-4" />
                            Yeni Kayıt
                        </Link>
                    </div>
                }
                emptyTitle="Muhasebe kaydı bulunamadı"
                emptyDescription="Henüz muhasebe kaydı bulunmuyor."
                emptyIcon={CreditCard}
            />
        </div>
    );
}