'use client';

import React, { useState, useEffect } from 'react';
import { Hotel, Plus, Eye, Edit, CheckCircle, XCircle, LayoutGrid, List } from 'lucide-react';
import Link from 'next/link';
import DataTable, { type TableColumn, type TableAction } from '@/components/ui/DataTable';
import KanbanBoard from '@/components/ui/KanbanBoard';
import { 
  renderCustomer, 
  renderStatusBadge, 
  renderAccommodationInfo,
  renderGuestInfo,
  renderBudget,
  renderHotelInfo,
  accommodationStatusConfig 
} from '@/components/ui/TableRenderers';
import type { AccommodationRequest } from '@/types';
import PermissionGuard from '@/components/guards/PermissionGuard';

export default function AccommodationRequestsPage() {
  return (
    <PermissionGuard requiredPermissions={['accommodation_view']}>
      <AccommodationRequestsPageContent />
    </PermissionGuard>
  );
}

function AccommodationRequestsPageContent() {
  const [requests, setRequests] = useState<AccommodationRequest[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState<string>('all');
  const [viewMode, setViewMode] = useState<'table' | 'kanban'>('table');

  useEffect(() => {
    fetchAccommodationRequests();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  const fetchAccommodationRequests = async () => {
    try {
      setLoading(true);
      
      // Mock data for now
      const mockRequests: AccommodationRequest[] = [
        {
          id: '1',
          customerId: 'c1',
          customer: {
            id: 'c1',
            type: 'individual',
            firstName: 'Ahmet',
            lastName: 'Yılmaz',
            phone: '+905551234567',
            email: 'ahmet@email.com',
            isVip: false,
            isActive: true,
            createdAt: new Date(),
            updatedAt: new Date()
          },
          destination: 'Antalya',
          checkInDate: new Date('2025-01-15'),
          checkOutDate: new Date('2025-01-20'),
          nights: 5,
          guests: [
            {
              id: 'g1',
              firstName: 'Ahmet',
              lastName: 'Yılmaz',
              isChild: false
            },
            {
              id: 'g2',
              firstName: 'Ayşe',
              lastName: 'Yılmaz',
              isChild: false
            }
          ],
          totalGuests: 2,
          roomType: 'double',
          roomCount: 1,
          budgetMin: 5000,
          budgetMax: 7000,
          currency: 'TRY',
          specialRequests: 'Deniz manzaralı oda tercihi',
          mealPlan: 'breakfast',
          hotelCategory: 'standard',
          status: 'pending',
          createdAt: new Date('2025-01-10'),
          updatedAt: new Date('2025-01-10')
        },
        {
          id: '2',
          customerId: 'c2',
          customer: {
            id: 'c2',
            type: 'corporate',
            companyName: 'ABC Teknoloji A.Ş.',
            contactPerson: 'Fatma Kaya',
            phone: '+905559876543',
            email: 'info@abcteknoloji.com',
            isVip: true,
            isActive: true,
            createdAt: new Date(),
            updatedAt: new Date()
          },
          requestedBy: {
            id: 'u1',
            customerId: 'c2',
            firstName: 'Mehmet',
            lastName: 'Özkan',
            email: 'mehmet.ozkan@abcteknoloji.com',
            department: 'Satış',
            position: 'Satış Temsilcisi',
            permissions: ['create_accommodation_request'],
            isActive: true,
            createdAt: new Date(),
            updatedAt: new Date()
          },
          destination: 'Bodrum',
          checkInDate: new Date('2025-02-01'),
          checkOutDate: new Date('2025-02-05'),
          nights: 4,
          guests: [
            {
              id: 'g3',
              firstName: 'Mehmet',
              lastName: 'Özkan',
              isChild: false
            },
            {
              id: 'g4',
              firstName: 'Zeynep',
              lastName: 'Özkan',
              isChild: false
            },
            {
              id: 'g5',
              firstName: 'Ali',
              lastName: 'Özkan',
              age: 8,
              isChild: true
            },
            {
              id: 'g6',
              firstName: 'Elif',
              lastName: 'Özkan',
              age: 5,
              isChild: true
            }
          ],
          totalGuests: 4,
          roomType: 'family',
          roomCount: 1,
          budgetMin: 8000,
          budgetMax: 10000,
          currency: 'TRY',
          mealPlan: 'half_board',
          hotelCategory: 'comfort',
          status: 'booked',
          hotelName: 'Bodrum Bay Resort',
          confirmationNumber: 'BDR2025001',
          totalPrice: 9200,
          agentNotes: 'Kurumsal müşteri - aile tatili',
          createdAt: new Date('2025-01-08'),
          updatedAt: new Date('2025-01-12')
        }
      ];
      
      setRequests(mockRequests);
    } catch (error) {
      console.error('Error fetching accommodation requests:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleStatusChange = async (requestId: string, newStatus: string) => {
    try {
      console.log('Status change:', requestId, newStatus);

      setRequests(prev => prev.map(request =>
        request.id === requestId
          ? { ...request, status: newStatus as AccommodationRequest['status'], updatedAt: new Date() }
          : request
      ));
    } catch (error) {
      console.error('Error updating status:', error);
    }
  };

  // Filtrelenmiş talepler
  const filteredRequests = requests.filter(request => {
    const customerName = request.customer.type === 'individual' 
      ? `${request.customer.firstName} ${request.customer.lastName}`
      : request.customer.companyName || '';
    
    const requestedByName = request.requestedBy 
      ? `${request.requestedBy.firstName} ${request.requestedBy.lastName}`
      : '';
    
    const matchesSearch = 
      customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      requestedByName.toLowerCase().includes(searchTerm.toLowerCase()) ||
      request.destination.toLowerCase().includes(searchTerm.toLowerCase()) ||
      request.hotelName?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      request.confirmationNumber?.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchesStatus = statusFilter === 'all' || request.status === statusFilter;
    
    return matchesSearch && matchesStatus;
  });

  // Kanban için sütunlar
  const kanbanColumns = [
    { id: 'pending', title: 'Beklemede', color: 'bg-yellow-500', count: 0 },
    { id: 'searching', title: 'Aranıyor', color: 'bg-blue-500', count: 0 },
    { id: 'quoted', title: 'Fiyat Verildi', color: 'bg-purple-500', count: 0 },
    { id: 'confirmed', title: 'Onaylandı', color: 'bg-green-500', count: 0 },
    { id: 'booked', title: 'Rezerve Edildi', color: 'bg-green-600', count: 0 },
    { id: 'cancelled', title: 'İptal Edildi', color: 'bg-red-500', count: 0 }
  ];

  // Kanban kartları
  const kanbanCards = filteredRequests.map(request => ({
    id: request.id,
    title: request.destination,
    subtitle: `${request.nights} gece - ${request.totalGuests} misafir`,
    customer: request.customer as unknown as Record<string, unknown>,
    status: request.status,
    amount: request.totalPrice || request.budgetMax,
    currency: request.currency,
    date: request.checkInDate,
    tags: [
      `${request.roomCount} oda`,
      request.roomType,
      request.hotelName || ''
    ].filter(Boolean),
    onClick: () => window.location.href = `/accommodation/requests/${request.id}`,
    onEdit: () => window.location.href = `/accommodation/requests/${request.id}/edit`
  }));

  // View mode toggle
  const viewModeToggle = (
    <div className="flex items-center bg-gray-100 rounded-lg p-1">
      <button
        onClick={() => setViewMode('table')}
        className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${
          viewMode === 'table'
            ? 'bg-white text-gray-900 shadow-sm'
            : 'text-gray-600 hover:text-gray-900'
        }`}
      >
        <List className="h-4 w-4 mr-2" />
        Tablo
      </button>
      <button
        onClick={() => setViewMode('kanban')}
        className={`flex items-center px-3 py-2 rounded-md text-sm font-medium transition-colors ${
          viewMode === 'kanban'
            ? 'bg-white text-gray-900 shadow-sm'
            : 'text-gray-600 hover:text-gray-900'
        }`}
      >
        <LayoutGrid className="h-4 w-4 mr-2" />
        Kanban
      </button>
    </div>
  );

  // Tablo sütunları
  const columns: TableColumn<AccommodationRequest>[] = [
    {
      key: 'customer',
      title: 'Müşteri/Talep Eden',
      render: (_, record) => renderCustomer(record.customer)
    },
    {
      key: 'accommodation',
      title: 'Konaklama Bilgileri',
      render: (_, record) => renderAccommodationInfo(
        record.destination, 
        record.checkInDate, 
        record.checkOutDate, 
        record.nights
      )
    },
    {
      key: 'guests',
      title: 'Misafir Bilgileri',
      render: (_, record) => renderGuestInfo(
        record.totalGuests, 
        record.roomCount, 
        record.roomType
      )
    },
    {
      key: 'budget',
      title: 'Bütçe',
      render: (_, record) => renderBudget(
        record.budgetMin, 
        record.budgetMax, 
        record.currency
      )
    },
    {
      key: 'hotel',
      title: 'Otel Bilgileri',
      render: (_, record) => renderHotelInfo(
        record.hotelName,
        record.confirmationNumber,
        record.totalPrice,
        record.currency
      )
    },
    {
      key: 'status',
      title: 'Durum',
      render: (_, record) => renderStatusBadge(record.status, accommodationStatusConfig)
    }
  ];

  // Tablo aksiyonları
  const actions: TableAction<AccommodationRequest>[] = [
    {
      key: 'view',
      label: 'Görüntüle',
      icon: Eye,
      onClick: (record) => window.location.href = `/accommodation/requests/${record.id}`,
      variant: 'primary'
    },
    {
      key: 'edit',
      label: 'Düzenle',
      icon: Edit,
      onClick: (record) => console.log('Edit:', record.id),
      variant: 'secondary'
    },
    {
      key: 'quote',
      label: 'Fiyat Ver',
      onClick: (record) => handleStatusChange(record.id, 'quoted'),
      variant: 'success',
      condition: (record) => record.status === 'pending' || record.status === 'searching'
    },
    {
      key: 'confirm',
      label: 'Onayla',
      icon: CheckCircle,
      onClick: (record) => handleStatusChange(record.id, 'confirmed'),
      variant: 'success',
      condition: (record) => record.status === 'quoted'
    },
    {
      key: 'book',
      label: 'Rezerve Et',
      onClick: (record) => handleStatusChange(record.id, 'booked'),
      variant: 'success',
      condition: (record) => record.status === 'confirmed'
    },
    {
      key: 'cancel',
      label: 'İptal Et',
      icon: XCircle,
      onClick: (record) => handleStatusChange(record.id, 'cancelled'),
      variant: 'danger',
      condition: (record) => ['pending', 'searching', 'quoted', 'confirmed'].includes(record.status)
    }
  ];

  // Filtreler
  const filters = [
    {
      key: 'status',
      label: 'Durum',
      value: statusFilter,
      onChange: setStatusFilter,
      options: [
        { value: 'all', label: 'Tüm Durumlar' },
        { value: 'pending', label: 'Beklemede' },
        { value: 'searching', label: 'Aranıyor' },
        { value: 'quoted', label: 'Fiyat Verildi' },
        { value: 'confirmed', label: 'Onaylandı' },
        { value: 'booked', label: 'Rezerve Edildi' },
        { value: 'cancelled', label: 'İptal Edildi' },
        { value: 'completed', label: 'Tamamlandı' }
      ]
    }
  ];

  if (viewMode === 'kanban') {
    return (
      <div className="p-6">
        {/* Header */}
        <div className="flex justify-between items-center mb-6">
          <div>
            <h1 className="text-2xl font-bold text-gray-900 flex items-center">
              <Hotel className="mr-3 h-8 w-8 text-purple-600" />
              Konaklama Talepleri
            </h1>
            <p className="text-gray-600 mt-1">
              Müşterilerden gelen konaklama taleplerini yönetin
            </p>
          </div>
          <div className="flex items-center space-x-4">
            {viewModeToggle}
            <Link
              href="/accommodation/requests/new"
              className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
            >
              <Plus className="h-4 w-4" />
              Yeni Talep
            </Link>
          </div>
        </div>

        {/* Filters */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <div className="relative">
                <input
                  type="text"
                  placeholder="Müşteri adı, destinasyon, otel adı veya rezervasyon numarası ara..."
                  className="w-full pl-4 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                />
              </div>
            </div>
            <div className="sm:w-48">
              <select
                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                value={statusFilter}
                onChange={(e) => setStatusFilter(e.target.value)}
              >
                {filters[0].options.map(option => (
                  <option key={option.value} value={option.value}>
                    {option.label}
                  </option>
                ))}
              </select>
            </div>
          </div>
        </div>

        {/* Kanban Board */}
        <KanbanBoard
          columns={kanbanColumns}
          cards={kanbanCards}
          statusConfig={accommodationStatusConfig}
          onCardClick={(card) => window.location.href = `/accommodation/requests/${card.id}`}
          onCardEdit={(card) => window.location.href = `/accommodation/requests/${card.id}/edit`}
          onStatusChange={handleStatusChange}
          loading={loading}
        />
      </div>
    );
  }

  return (
    <div className="p-6">
      <DataTable
        title="Konaklama Talepleri"
        description="Müşterilerden gelen konaklama taleplerini yönetin"
        data={filteredRequests}
        columns={columns}
        actions={actions}
        loading={loading}
        searchable={true}
        searchPlaceholder="Müşteri adı, destinasyon, otel adı veya rezervasyon numarası ara..."
        searchValue={searchTerm}
        onSearchChange={setSearchTerm}
        filters={filters}
        headerActions={
          <div className="flex items-center space-x-4">
            {viewModeToggle}
            <Link
              href="/accommodation/requests/new"
              className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center gap-2"
            >
              <Plus className="h-4 w-4" />
              Yeni Talep
            </Link>
          </div>
        }
        emptyTitle="Konaklama talebi bulunamadı"
        emptyDescription="Henüz konaklama talebi bulunmuyor."
        emptyIcon={Hotel}
      />
    </div>
  );
}