-- vTravel Panel - Complete Database Schema (SQLite)
-- Production-ready schema for all entities

-- ============================================
-- CUSTOMERS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS customers (
  id TEXT PRIMARY KEY,
  type TEXT NOT NULL CHECK(type IN ('individual', 'corporate')),

  -- Bireysel müşteri bilgileri
  first_name TEXT,
  last_name TEXT,

  -- Kurumsal müşteri bilgileri
  company_name TEXT,
  tax_number TEXT,
  tax_office TEXT,
  contact_person TEXT,
  department TEXT,
  parent_company_id TEXT,

  -- Ortak bilgiler
  email TEXT,
  phone TEXT NOT NULL,
  address TEXT,
  city TEXT,
  country TEXT,
  notes TEXT,
  is_vip INTEGER DEFAULT 0,
  is_active INTEGER DEFAULT 1,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  FOREIGN KEY (parent_company_id) REFERENCES customers(id)
);

CREATE INDEX IF NOT EXISTS idx_customers_type ON customers(type);
CREATE INDEX IF NOT EXISTS idx_customers_phone ON customers(phone);
CREATE INDEX IF NOT EXISTS idx_customers_email ON customers(email);
CREATE INDEX IF NOT EXISTS idx_customers_active ON customers(is_active);

-- ============================================
-- CORPORATE USERS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS corporate_users (
  id TEXT PRIMARY KEY,
  customer_id TEXT NOT NULL,
  first_name TEXT NOT NULL,
  last_name TEXT NOT NULL,
  email TEXT NOT NULL UNIQUE,
  phone TEXT,
  department TEXT,
  position TEXT,
  permissions TEXT, -- JSON array
  is_active INTEGER DEFAULT 1,
  last_login_at DATETIME,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  FOREIGN KEY (customer_id) REFERENCES customers(id)
);

CREATE INDEX IF NOT EXISTS idx_corporate_users_customer ON corporate_users(customer_id);
CREATE INDEX IF NOT EXISTS idx_corporate_users_email ON corporate_users(email);
CREATE INDEX IF NOT EXISTS idx_corporate_users_active ON corporate_users(is_active);

-- ============================================
-- AGENCY USERS (Internal Staff) TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS agency_users (
  id TEXT PRIMARY KEY,
  first_name TEXT NOT NULL,
  last_name TEXT NOT NULL,
  email TEXT NOT NULL UNIQUE,
  phone TEXT,
  avatar TEXT,

  -- Department and position
  department_id TEXT NOT NULL,
  position TEXT NOT NULL,

  -- Authorization
  role TEXT NOT NULL CHECK(role IN ('super_admin', 'admin', 'manager', 'senior_agent', 'agent', 'intern')),
  permissions TEXT, -- JSON array

  -- Status
  is_active INTEGER DEFAULT 1,
  is_online INTEGER DEFAULT 0,
  last_login_at DATETIME,

  -- Manager info
  manager_id TEXT,

  -- Statistics
  total_tickets INTEGER DEFAULT 0,
  total_revenue REAL DEFAULT 0,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  created_by TEXT,

  FOREIGN KEY (department_id) REFERENCES departments(id),
  FOREIGN KEY (manager_id) REFERENCES agency_users(id)
);

CREATE INDEX IF NOT EXISTS idx_agency_users_email ON agency_users(email);
CREATE INDEX IF NOT EXISTS idx_agency_users_department ON agency_users(department_id);
CREATE INDEX IF NOT EXISTS idx_agency_users_role ON agency_users(role);
CREATE INDEX IF NOT EXISTS idx_agency_users_active ON agency_users(is_active);

-- ============================================
-- DEPARTMENTS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS departments (
  id TEXT PRIMARY KEY,
  name TEXT NOT NULL,
  description TEXT,
  color TEXT NOT NULL,

  -- Hierarchy
  parent_department_id TEXT,

  -- Manager
  manager_id TEXT,

  -- Status
  is_active INTEGER DEFAULT 1,

  -- Statistics
  user_count INTEGER DEFAULT 0,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  FOREIGN KEY (parent_department_id) REFERENCES departments(id)
);

CREATE INDEX IF NOT EXISTS idx_departments_active ON departments(is_active);
CREATE INDEX IF NOT EXISTS idx_departments_parent ON departments(parent_department_id);

-- ============================================
-- PASSENGERS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS passengers (
  id TEXT PRIMARY KEY,
  first_name TEXT NOT NULL,
  last_name TEXT NOT NULL,
  phone TEXT,
  email TEXT,

  -- Corporate employee info
  employee_id TEXT,
  department TEXT,
  position TEXT,

  -- Child passenger info
  age INTEGER,
  is_child INTEGER DEFAULT 0,

  -- Special needs
  special_needs TEXT,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX IF NOT EXISTS idx_passengers_name ON passengers(first_name, last_name);
CREATE INDEX IF NOT EXISTS idx_passengers_employee ON passengers(employee_id);

-- ============================================
-- TICKET REQUESTS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS ticket_requests (
  id TEXT PRIMARY KEY,
  customer_id TEXT NOT NULL,
  requested_by_user_id TEXT, -- Corporate user ID

  -- Flight details
  airline TEXT NOT NULL CHECK(airline IN ('THY', 'PC', 'AJET', 'OTHER')),
  flight_number TEXT,
  departure_airport TEXT NOT NULL,
  arrival_airport TEXT NOT NULL,
  departure_date DATETIME NOT NULL,
  departure_time TEXT,

  -- Request details
  status TEXT NOT NULL CHECK(status IN ('pending', 'confirmed', 'ticketed', 'cancelled', 'rejected')),
  request_date DATETIME NOT NULL,
  ticket_price REAL,
  currency TEXT DEFAULT 'TRY',

  -- Agent notes
  agent_notes TEXT,
  pnr TEXT,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  FOREIGN KEY (customer_id) REFERENCES customers(id),
  FOREIGN KEY (requested_by_user_id) REFERENCES corporate_users(id)
);

CREATE INDEX IF NOT EXISTS idx_ticket_requests_customer ON ticket_requests(customer_id);
CREATE INDEX IF NOT EXISTS idx_ticket_requests_status ON ticket_requests(status);
CREATE INDEX IF NOT EXISTS idx_ticket_requests_airline ON ticket_requests(airline);
CREATE INDEX IF NOT EXISTS idx_ticket_requests_departure_date ON ticket_requests(departure_date);
CREATE INDEX IF NOT EXISTS idx_ticket_requests_pnr ON ticket_requests(pnr);

-- ============================================
-- TICKET REQUEST PASSENGERS (Many-to-Many)
-- ============================================
CREATE TABLE IF NOT EXISTS ticket_request_passengers (
  id TEXT PRIMARY KEY,
  ticket_request_id TEXT NOT NULL,
  passenger_id TEXT NOT NULL,

  FOREIGN KEY (ticket_request_id) REFERENCES ticket_requests(id),
  FOREIGN KEY (passenger_id) REFERENCES passengers(id)
);

CREATE INDEX IF NOT EXISTS idx_trp_ticket ON ticket_request_passengers(ticket_request_id);
CREATE INDEX IF NOT EXISTS idx_trp_passenger ON ticket_request_passengers(passenger_id);

-- ============================================
-- CHECKIN TICKETS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS checkin_tickets (
  id TEXT PRIMARY KEY,
  ticket_request_id TEXT,
  pnr TEXT NOT NULL,
  customer_id TEXT NOT NULL,
  requested_by_user_id TEXT,

  -- Flight details
  airline TEXT NOT NULL CHECK(airline IN ('THY', 'PC', 'AJET', 'OTHER')),
  flight_number TEXT NOT NULL,
  departure_airport TEXT NOT NULL,
  arrival_airport TEXT NOT NULL,
  departure_date DATETIME NOT NULL,
  departure_time TEXT NOT NULL,

  -- Check-in details
  checkin_status TEXT NOT NULL CHECK(checkin_status IN ('waiting', 'ready', 'processing', 'completed', 'failed', 'expired', 'cancelled')),
  checkin_open_time DATETIME,
  checkin_completed_at DATETIME,
  checkin_attempts INTEGER DEFAULT 0,
  last_checkin_attempt DATETIME,

  -- SMS
  sms_status TEXT DEFAULT 'pending' CHECK(sms_status IN ('pending', 'sent', 'failed')),
  error_message TEXT,

  -- Settings
  is_auto_checkin_enabled INTEGER DEFAULT 1,
  notes TEXT,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  FOREIGN KEY (ticket_request_id) REFERENCES ticket_requests(id),
  FOREIGN KEY (customer_id) REFERENCES customers(id),
  FOREIGN KEY (requested_by_user_id) REFERENCES corporate_users(id)
);

CREATE INDEX IF NOT EXISTS idx_checkin_tickets_pnr ON checkin_tickets(pnr);
CREATE INDEX IF NOT EXISTS idx_checkin_tickets_customer ON checkin_tickets(customer_id);
CREATE INDEX IF NOT EXISTS idx_checkin_tickets_status ON checkin_tickets(checkin_status);
CREATE INDEX IF NOT EXISTS idx_checkin_tickets_airline ON checkin_tickets(airline);
CREATE INDEX IF NOT EXISTS idx_checkin_tickets_departure_date ON checkin_tickets(departure_date);
CREATE INDEX IF NOT EXISTS idx_checkin_tickets_auto_enabled ON checkin_tickets(is_auto_checkin_enabled);

-- ============================================
-- CHECKIN TICKET PASSENGERS (Many-to-Many)
-- ============================================
CREATE TABLE IF NOT EXISTS checkin_ticket_passengers (
  id TEXT PRIMARY KEY,
  checkin_ticket_id TEXT NOT NULL,
  passenger_id TEXT NOT NULL,

  -- Check-in result for this passenger
  checkin_status TEXT CHECK(checkin_status IN ('waiting', 'ready', 'processing', 'completed', 'failed', 'expired', 'cancelled')),
  seat TEXT,
  boarding_pass_url TEXT,
  error_message TEXT,

  FOREIGN KEY (checkin_ticket_id) REFERENCES checkin_tickets(id),
  FOREIGN KEY (passenger_id) REFERENCES passengers(id)
);

CREATE INDEX IF NOT EXISTS idx_ctp_ticket ON checkin_ticket_passengers(checkin_ticket_id);
CREATE INDEX IF NOT EXISTS idx_ctp_passenger ON checkin_ticket_passengers(passenger_id);

-- ============================================
-- AUTO CHECKIN JOBS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS auto_checkin_jobs (
  id TEXT PRIMARY KEY,
  checkin_ticket_id TEXT NOT NULL,
  scheduled_for DATETIME NOT NULL,
  status TEXT NOT NULL CHECK(status IN ('scheduled', 'queued', 'processing', 'completed', 'failed', 'cancelled')),
  attempts INTEGER DEFAULT 0,
  last_attempt DATETIME,
  next_retry DATETIME,
  parent_job_id TEXT, -- For retries
  retry_reason TEXT,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  FOREIGN KEY (checkin_ticket_id) REFERENCES checkin_tickets(id),
  FOREIGN KEY (parent_job_id) REFERENCES auto_checkin_jobs(id)
);

CREATE INDEX IF NOT EXISTS idx_acj_ticket ON auto_checkin_jobs(checkin_ticket_id);
CREATE INDEX IF NOT EXISTS idx_acj_status ON auto_checkin_jobs(status);
CREATE INDEX IF NOT EXISTS idx_acj_scheduled ON auto_checkin_jobs(scheduled_for);

-- ============================================
-- CHECKIN LOGS TABLE (Already exists)
-- ============================================
-- This table is already created in database.js

-- ============================================
-- FLIGHTS TABLE (For tracking)
-- ============================================
CREATE TABLE IF NOT EXISTS flights (
  id TEXT PRIMARY KEY,
  airline TEXT NOT NULL,
  flight_number TEXT NOT NULL,
  departure_airport TEXT NOT NULL,
  arrival_airport TEXT NOT NULL,
  departure_date DATETIME NOT NULL,
  departure_time TEXT,
  arrival_date DATETIME,
  arrival_time TEXT,

  -- Status tracking
  status TEXT CHECK(status IN ('scheduled', 'delayed', 'departed', 'arrived', 'cancelled')),
  delay_minutes INTEGER,
  gate TEXT,
  terminal TEXT,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX IF NOT EXISTS idx_flights_airline_number ON flights(airline, flight_number);
CREATE INDEX IF NOT EXISTS idx_flights_departure_date ON flights(departure_date);
CREATE INDEX IF NOT EXISTS idx_flights_status ON flights(status);

-- ============================================
-- ACCOMMODATION REQUESTS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS accommodation_requests (
  id TEXT PRIMARY KEY,
  customer_id TEXT NOT NULL,
  requested_by_user_id TEXT,

  -- Accommodation details
  destination TEXT NOT NULL,
  check_in_date DATETIME NOT NULL,
  check_out_date DATETIME NOT NULL,
  nights INTEGER NOT NULL,
  total_guests INTEGER NOT NULL,

  -- Room preferences
  room_type TEXT CHECK(room_type IN ('single', 'double', 'twin', 'triple', 'family', 'suite', 'connecting')),
  room_count INTEGER NOT NULL,

  -- Budget
  budget_min REAL,
  budget_max REAL,
  currency TEXT DEFAULT 'TRY',

  -- Preferences
  special_requests TEXT,
  meal_plan TEXT CHECK(meal_plan IN ('room_only', 'breakfast', 'half_board', 'full_board', 'all_inclusive')),
  hotel_category TEXT CHECK(hotel_category IN ('budget', 'standard', 'comfort', 'luxury', 'boutique')),

  -- Status
  status TEXT NOT NULL CHECK(status IN ('pending', 'searching', 'quoted', 'confirmed', 'booked', 'cancelled', 'completed')),

  -- Booking details
  hotel_name TEXT,
  hotel_address TEXT,
  confirmation_number TEXT,
  total_price REAL,

  -- Notes
  agent_notes TEXT,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  FOREIGN KEY (customer_id) REFERENCES customers(id),
  FOREIGN KEY (requested_by_user_id) REFERENCES corporate_users(id)
);

CREATE INDEX IF NOT EXISTS idx_accommodation_customer ON accommodation_requests(customer_id);
CREATE INDEX IF NOT EXISTS idx_accommodation_status ON accommodation_requests(status);
CREATE INDEX IF NOT EXISTS idx_accommodation_checkin ON accommodation_requests(check_in_date);

-- ============================================
-- ACCOMMODATION GUESTS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS accommodation_guests (
  id TEXT PRIMARY KEY,
  accommodation_request_id TEXT NOT NULL,
  first_name TEXT NOT NULL,
  last_name TEXT NOT NULL,
  age INTEGER,
  is_child INTEGER DEFAULT 0,
  special_needs TEXT,

  FOREIGN KEY (accommodation_request_id) REFERENCES accommodation_requests(id)
);

CREATE INDEX IF NOT EXISTS idx_ag_accommodation ON accommodation_guests(accommodation_request_id);

-- ============================================
-- VISA REQUESTS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS visa_requests (
  id TEXT PRIMARY KEY,
  customer_id TEXT NOT NULL,
  requested_by_user_id TEXT,

  -- Applicant (stored as JSON for simplicity)
  applicant_data TEXT NOT NULL, -- JSON

  -- Visa details
  country TEXT NOT NULL,
  visa_type TEXT CHECK(visa_type IN ('tourist', 'business', 'student', 'work', 'transit', 'family', 'medical')),
  application_date DATETIME NOT NULL,
  travel_date DATETIME NOT NULL,
  return_date DATETIME,

  -- Application status
  status TEXT NOT NULL CHECK(status IN ('pending', 'document_review', 'appointment_scheduled', 'interview_completed', 'processing', 'approved', 'rejected', 'delivered', 'cancelled')),
  priority TEXT DEFAULT 'medium' CHECK(priority IN ('low', 'medium', 'high', 'urgent')),

  -- Documents (stored as JSON array)
  documents TEXT, -- JSON array
  required_documents TEXT, -- JSON array

  -- Appointment
  appointment_date DATETIME,
  appointment_location TEXT,

  -- Result
  decision_date DATETIME,
  visa_number TEXT,
  valid_until DATETIME,

  -- Pricing
  application_fee REAL,
  service_fee REAL,
  total_amount REAL,
  currency TEXT DEFAULT 'TRY',

  -- Notes
  agent_notes TEXT,
  special_requests TEXT,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  FOREIGN KEY (customer_id) REFERENCES customers(id),
  FOREIGN KEY (requested_by_user_id) REFERENCES corporate_users(id)
);

CREATE INDEX IF NOT EXISTS idx_visa_customer ON visa_requests(customer_id);
CREATE INDEX IF NOT EXISTS idx_visa_status ON visa_requests(status);
CREATE INDEX IF NOT EXISTS idx_visa_country ON visa_requests(country);
CREATE INDEX IF NOT EXISTS idx_visa_travel_date ON visa_requests(travel_date);
CREATE INDEX IF NOT EXISTS idx_visa_priority ON visa_requests(priority);

-- ============================================
-- TRANSFER REQUESTS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS transfer_requests (
  id TEXT PRIMARY KEY,
  customer_id TEXT NOT NULL,
  requested_by_user_id TEXT,

  -- Transfer details
  transfer_type TEXT CHECK(transfer_type IN ('airport_pickup', 'airport_dropoff', 'hotel_transfer', 'city_tour', 'point_to_point', 'hourly', 'daily')),
  pickup_location TEXT NOT NULL,
  dropoff_location TEXT NOT NULL,
  pickup_date DATETIME NOT NULL,
  pickup_time TEXT NOT NULL,

  -- Passengers (count)
  total_passengers INTEGER NOT NULL,

  -- Vehicle
  vehicle_type TEXT CHECK(vehicle_type IN ('economy', 'comfort', 'luxury', 'minibus', 'bus', 'vip')),
  vehicle_count INTEGER DEFAULT 1,

  -- Special requests
  special_requests TEXT,
  has_luggage INTEGER DEFAULT 0,
  needs_child_seat INTEGER DEFAULT 0,
  is_wheelchair_accessible INTEGER DEFAULT 0,

  -- Status
  status TEXT NOT NULL CHECK(status IN ('pending', 'quoted', 'confirmed', 'assigned', 'in_progress', 'completed', 'cancelled')),

  -- Assignment (stored as JSON)
  assigned_driver_data TEXT, -- JSON
  assigned_vehicle_data TEXT, -- JSON

  -- Pricing
  estimated_price REAL,
  final_price REAL,
  currency TEXT DEFAULT 'TRY',

  -- Notes
  agent_notes TEXT,
  driver_notes TEXT,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  FOREIGN KEY (customer_id) REFERENCES customers(id),
  FOREIGN KEY (requested_by_user_id) REFERENCES corporate_users(id)
);

CREATE INDEX IF NOT EXISTS idx_transfer_customer ON transfer_requests(customer_id);
CREATE INDEX IF NOT EXISTS idx_transfer_status ON transfer_requests(status);
CREATE INDEX IF NOT EXISTS idx_transfer_pickup_date ON transfer_requests(pickup_date);

-- ============================================
-- TRANSFER PASSENGERS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS transfer_passengers (
  id TEXT PRIMARY KEY,
  transfer_request_id TEXT NOT NULL,
  first_name TEXT NOT NULL,
  last_name TEXT NOT NULL,
  phone TEXT,
  is_child INTEGER DEFAULT 0,
  age INTEGER,
  special_needs TEXT,

  FOREIGN KEY (transfer_request_id) REFERENCES transfer_requests(id)
);

CREATE INDEX IF NOT EXISTS idx_tp_transfer ON transfer_passengers(transfer_request_id);

-- ============================================
-- INVOICES TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS invoices (
  id TEXT PRIMARY KEY,
  invoice_number TEXT NOT NULL UNIQUE,
  customer_id TEXT NOT NULL,

  -- Invoice details
  issue_date DATETIME NOT NULL,
  due_date DATETIME NOT NULL,
  amount REAL NOT NULL,
  currency TEXT DEFAULT 'TRY',
  tax_amount REAL DEFAULT 0,
  total_amount REAL NOT NULL,

  -- Status
  status TEXT NOT NULL CHECK(status IN ('draft', 'sent', 'viewed', 'overdue', 'cancelled')),
  payment_status TEXT NOT NULL CHECK(payment_status IN ('unpaid', 'partial', 'paid', 'refunded')),

  -- Payment
  paid_amount REAL DEFAULT 0,
  paid_date DATETIME,
  payment_method TEXT CHECK(payment_method IN ('cash', 'bank_transfer', 'credit_card', 'check', 'other')),

  -- Notes
  notes TEXT,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  FOREIGN KEY (customer_id) REFERENCES customers(id)
);

CREATE INDEX IF NOT EXISTS idx_invoices_customer ON invoices(customer_id);
CREATE INDEX IF NOT EXISTS idx_invoices_number ON invoices(invoice_number);
CREATE INDEX IF NOT EXISTS idx_invoices_status ON invoices(status);
CREATE INDEX IF NOT EXISTS idx_invoices_payment_status ON invoices(payment_status);
CREATE INDEX IF NOT EXISTS idx_invoices_due_date ON invoices(due_date);

-- ============================================
-- INVOICE ITEMS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS invoice_items (
  id TEXT PRIMARY KEY,
  invoice_id TEXT NOT NULL,
  service_type TEXT CHECK(service_type IN ('ticket', 'accommodation', 'visa', 'transfer')),
  service_id TEXT,
  description TEXT NOT NULL,
  quantity INTEGER DEFAULT 1,
  unit_price REAL NOT NULL,
  total_price REAL NOT NULL,
  tax_rate REAL DEFAULT 0,
  tax_amount REAL DEFAULT 0,

  FOREIGN KEY (invoice_id) REFERENCES invoices(id)
);

CREATE INDEX IF NOT EXISTS idx_invoice_items_invoice ON invoice_items(invoice_id);
CREATE INDEX IF NOT EXISTS idx_invoice_items_service ON invoice_items(service_type, service_id);

-- ============================================
-- REVENUE RECORDS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS revenue_records (
  id TEXT PRIMARY KEY,
  type TEXT NOT NULL CHECK(type IN ('ticket', 'accommodation', 'visa', 'transfer')),
  customer_id TEXT NOT NULL,
  requested_by_user_id TEXT,

  -- Transaction details
  description TEXT NOT NULL,
  amount REAL NOT NULL,
  currency TEXT DEFAULT 'TRY',
  commission REAL,
  net_amount REAL,

  -- Reference
  reference_id TEXT NOT NULL,
  reference_type TEXT NOT NULL,

  -- Status
  status TEXT NOT NULL CHECK(status IN ('pending', 'confirmed', 'completed', 'cancelled', 'refunded')),
  payment_status TEXT NOT NULL CHECK(payment_status IN ('unpaid', 'partial', 'paid', 'overdue', 'refunded')),
  invoice_number TEXT,

  -- Dates
  service_date DATETIME NOT NULL,
  completed_at DATETIME,
  invoiced_at DATETIME,
  paid_at DATETIME,

  -- Notes
  notes TEXT,

  -- Meta
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  FOREIGN KEY (customer_id) REFERENCES customers(id),
  FOREIGN KEY (requested_by_user_id) REFERENCES corporate_users(id)
);

CREATE INDEX IF NOT EXISTS idx_revenue_type ON revenue_records(type);
CREATE INDEX IF NOT EXISTS idx_revenue_customer ON revenue_records(customer_id);
CREATE INDEX IF NOT EXISTS idx_revenue_status ON revenue_records(status);
CREATE INDEX IF NOT EXISTS idx_revenue_payment_status ON revenue_records(payment_status);
CREATE INDEX IF NOT EXISTS idx_revenue_service_date ON revenue_records(service_date);
CREATE INDEX IF NOT EXISTS idx_revenue_reference ON revenue_records(reference_id);

-- ============================================
-- USER ACTIVITIES TABLE (Audit log)
-- ============================================
CREATE TABLE IF NOT EXISTS user_activities (
  id TEXT PRIMARY KEY,
  user_id TEXT NOT NULL,
  action TEXT NOT NULL,
  description TEXT NOT NULL,
  entity_type TEXT,
  entity_id TEXT,
  ip_address TEXT,
  user_agent TEXT,

  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  FOREIGN KEY (user_id) REFERENCES agency_users(id)
);

CREATE INDEX IF NOT EXISTS idx_user_activities_user ON user_activities(user_id);
CREATE INDEX IF NOT EXISTS idx_user_activities_created ON user_activities(created_at);
CREATE INDEX IF NOT EXISTS idx_user_activities_entity ON user_activities(entity_type, entity_id);

-- ============================================
-- USER SESSIONS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS user_sessions (
  id TEXT PRIMARY KEY,
  user_id TEXT NOT NULL,
  token TEXT NOT NULL UNIQUE,
  ip_address TEXT NOT NULL,
  user_agent TEXT NOT NULL,
  is_active INTEGER DEFAULT 1,
  expires_at DATETIME NOT NULL,
  last_activity_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,

  FOREIGN KEY (user_id) REFERENCES agency_users(id)
);

CREATE INDEX IF NOT EXISTS idx_user_sessions_user ON user_sessions(user_id);
CREATE INDEX IF NOT EXISTS idx_user_sessions_token ON user_sessions(token);
CREATE INDEX IF NOT EXISTS idx_user_sessions_active ON user_sessions(is_active);
CREATE INDEX IF NOT EXISTS idx_user_sessions_expires ON user_sessions(expires_at);
