-- vTravel Panel - MySQL Database Schema
-- Converted from SQLite schema

-- ============================================
-- CUSTOMERS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS customers (
  id VARCHAR(36) PRIMARY KEY,
  type ENUM('individual', 'corporate') NOT NULL,

  -- Bireysel müşteri bilgileri
  first_name VARCHAR(255),
  last_name VARCHAR(255),

  -- Kurumsal müşteri bilgileri
  company_name VARCHAR(255),
  tax_number VARCHAR(50),
  tax_office VARCHAR(255),
  contact_person VARCHAR(255),
  department VARCHAR(255),
  parent_company_id VARCHAR(36),

  -- Ortak bilgiler
  email VARCHAR(255),
  phone VARCHAR(50) NOT NULL,
  address TEXT,
  city VARCHAR(100),
  country VARCHAR(100),
  notes TEXT,
  is_vip TINYINT(1) DEFAULT 0,
  is_active TINYINT(1) DEFAULT 1,

  -- Meta
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

  INDEX idx_customers_type (type),
  INDEX idx_customers_phone (phone),
  INDEX idx_customers_email (email),
  INDEX idx_customers_active (is_active),
  FOREIGN KEY (parent_company_id) REFERENCES customers(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- CORPORATE USERS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS corporate_users (
  id VARCHAR(36) PRIMARY KEY,
  customer_id VARCHAR(36) NOT NULL,
  first_name VARCHAR(255) NOT NULL,
  last_name VARCHAR(255) NOT NULL,
  email VARCHAR(255) NOT NULL UNIQUE,
  phone VARCHAR(50),
  department VARCHAR(255),
  position VARCHAR(255),
  permissions JSON,
  is_active TINYINT(1) DEFAULT 1,
  last_login_at TIMESTAMP NULL,

  -- Meta
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

  INDEX idx_corporate_users_customer (customer_id),
  INDEX idx_corporate_users_email (email),
  INDEX idx_corporate_users_active (is_active),
  FOREIGN KEY (customer_id) REFERENCES customers(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- DEPARTMENTS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS departments (
  id VARCHAR(36) PRIMARY KEY,
  name VARCHAR(255) NOT NULL,
  description TEXT,
  color VARCHAR(50) NOT NULL,

  -- Hierarchy
  parent_department_id VARCHAR(36),

  -- Manager
  manager_id VARCHAR(36),

  -- Status
  is_active TINYINT(1) DEFAULT 1,

  -- Statistics
  user_count INT DEFAULT 0,

  -- Meta
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

  INDEX idx_departments_active (is_active),
  INDEX idx_departments_parent (parent_department_id),
  FOREIGN KEY (parent_department_id) REFERENCES departments(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- AGENCY USERS (Internal Staff) TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS agency_users (
  id VARCHAR(36) PRIMARY KEY,
  first_name VARCHAR(255) NOT NULL,
  last_name VARCHAR(255) NOT NULL,
  email VARCHAR(255) NOT NULL UNIQUE,
  phone VARCHAR(50),
  avatar VARCHAR(500),

  -- Department and position
  department_id VARCHAR(36) NOT NULL,
  position VARCHAR(255) NOT NULL,

  -- Authorization
  role ENUM('super_admin', 'admin', 'manager', 'senior_agent', 'agent', 'intern') NOT NULL,
  permissions JSON,

  -- Status
  is_active TINYINT(1) DEFAULT 1,
  is_online TINYINT(1) DEFAULT 0,
  last_login_at TIMESTAMP NULL,

  -- Manager info
  manager_id VARCHAR(36),

  -- Statistics
  total_tickets INT DEFAULT 0,
  total_revenue DECIMAL(10,2) DEFAULT 0,

  -- Meta
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_by VARCHAR(36),

  INDEX idx_agency_users_email (email),
  INDEX idx_agency_users_department (department_id),
  INDEX idx_agency_users_role (role),
  INDEX idx_agency_users_active (is_active),
  FOREIGN KEY (department_id) REFERENCES departments(id) ON DELETE RESTRICT,
  FOREIGN KEY (manager_id) REFERENCES agency_users(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- PASSENGERS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS passengers (
  id VARCHAR(36) PRIMARY KEY,
  first_name VARCHAR(255) NOT NULL,
  last_name VARCHAR(255) NOT NULL,
  phone VARCHAR(50),
  email VARCHAR(255),

  -- Corporate employee info
  employee_id VARCHAR(50),
  department VARCHAR(255),
  position VARCHAR(255),

  -- Child passenger info
  age INT,
  is_child TINYINT(1) DEFAULT 0,

  -- Special needs
  special_needs TEXT,

  -- Meta
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

  INDEX idx_passengers_name (first_name, last_name),
  INDEX idx_passengers_employee (employee_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TICKET REQUESTS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS ticket_requests (
  id VARCHAR(36) PRIMARY KEY,
  customer_id VARCHAR(36) NOT NULL,
  requested_by_user_id VARCHAR(36),

  -- Flight details
  airline ENUM('THY', 'PC', 'AJET', 'OTHER') NOT NULL,
  flight_number VARCHAR(20),
  departure_airport VARCHAR(10) NOT NULL,
  arrival_airport VARCHAR(10) NOT NULL,
  departure_date TIMESTAMP NOT NULL,
  departure_time VARCHAR(10),

  -- Request details
  status ENUM('pending', 'confirmed', 'ticketed', 'cancelled', 'rejected') NOT NULL,
  request_date TIMESTAMP NOT NULL,
  ticket_price DECIMAL(10,2),
  currency VARCHAR(10) DEFAULT 'TRY',

  -- Agent notes
  agent_notes TEXT,
  pnr VARCHAR(20),

  -- Meta
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

  INDEX idx_ticket_requests_customer (customer_id),
  INDEX idx_ticket_requests_status (status),
  INDEX idx_ticket_requests_airline (airline),
  INDEX idx_ticket_requests_departure_date (departure_date),
  INDEX idx_ticket_requests_pnr (pnr),
  FOREIGN KEY (customer_id) REFERENCES customers(id) ON DELETE CASCADE,
  FOREIGN KEY (requested_by_user_id) REFERENCES corporate_users(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TICKET REQUEST PASSENGERS (Many-to-Many)
-- ============================================
CREATE TABLE IF NOT EXISTS ticket_request_passengers (
  id VARCHAR(36) PRIMARY KEY,
  ticket_request_id VARCHAR(36) NOT NULL,
  passenger_id VARCHAR(36) NOT NULL,

  INDEX idx_trp_ticket (ticket_request_id),
  INDEX idx_trp_passenger (passenger_id),
  FOREIGN KEY (ticket_request_id) REFERENCES ticket_requests(id) ON DELETE CASCADE,
  FOREIGN KEY (passenger_id) REFERENCES passengers(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- CHECKIN TICKETS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS checkin_tickets (
  id VARCHAR(36) PRIMARY KEY,
  ticket_request_id VARCHAR(36),
  pnr VARCHAR(20) NOT NULL,
  customer_id VARCHAR(36) NOT NULL,
  requested_by_user_id VARCHAR(36),

  -- Flight details
  airline ENUM('THY', 'PC', 'AJET', 'OTHER') NOT NULL,
  flight_number VARCHAR(20) NOT NULL,
  departure_airport VARCHAR(10) NOT NULL,
  arrival_airport VARCHAR(10) NOT NULL,
  departure_date TIMESTAMP NOT NULL,
  departure_time VARCHAR(10) NOT NULL,

  -- Check-in details
  checkin_status ENUM('waiting', 'ready', 'processing', 'completed', 'failed', 'expired', 'cancelled') NOT NULL,
  checkin_open_time TIMESTAMP NULL,
  checkin_completed_at TIMESTAMP NULL,
  checkin_attempts INT DEFAULT 0,
  last_checkin_attempt TIMESTAMP NULL,

  -- SMS
  sms_status ENUM('pending', 'sent', 'failed') DEFAULT 'pending',
  error_message TEXT,

  -- Settings
  is_auto_checkin_enabled TINYINT(1) DEFAULT 1,
  notes TEXT,

  -- Meta
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

  INDEX idx_checkin_tickets_pnr (pnr),
  INDEX idx_checkin_tickets_customer (customer_id),
  INDEX idx_checkin_tickets_status (checkin_status),
  INDEX idx_checkin_tickets_airline (airline),
  INDEX idx_checkin_tickets_departure_date (departure_date),
  INDEX idx_checkin_tickets_auto_enabled (is_auto_checkin_enabled),
  FOREIGN KEY (ticket_request_id) REFERENCES ticket_requests(id) ON DELETE SET NULL,
  FOREIGN KEY (customer_id) REFERENCES customers(id) ON DELETE CASCADE,
  FOREIGN KEY (requested_by_user_id) REFERENCES corporate_users(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- CHECKIN TICKET PASSENGERS (Many-to-Many)
-- ============================================
CREATE TABLE IF NOT EXISTS checkin_ticket_passengers (
  id VARCHAR(36) PRIMARY KEY,
  checkin_ticket_id VARCHAR(36) NOT NULL,
  passenger_id VARCHAR(36) NOT NULL,

  -- Check-in result for this passenger
  checkin_status ENUM('waiting', 'ready', 'processing', 'completed', 'failed', 'expired', 'cancelled'),
  seat VARCHAR(10),
  boarding_pass_url VARCHAR(500),
  error_message TEXT,

  INDEX idx_ctp_ticket (checkin_ticket_id),
  INDEX idx_ctp_passenger (passenger_id),
  FOREIGN KEY (checkin_ticket_id) REFERENCES checkin_tickets(id) ON DELETE CASCADE,
  FOREIGN KEY (passenger_id) REFERENCES passengers(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- AUTO CHECKIN JOBS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS auto_checkin_jobs (
  id VARCHAR(36) PRIMARY KEY,
  checkin_ticket_id VARCHAR(36) NOT NULL,
  scheduled_for TIMESTAMP NOT NULL,
  status ENUM('scheduled', 'queued', 'processing', 'completed', 'failed', 'cancelled') NOT NULL,
  attempts INT DEFAULT 0,
  last_attempt TIMESTAMP NULL,
  next_retry TIMESTAMP NULL,
  parent_job_id VARCHAR(36),
  retry_reason TEXT,

  -- Meta
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

  INDEX idx_acj_ticket (checkin_ticket_id),
  INDEX idx_acj_status (status),
  INDEX idx_acj_scheduled (scheduled_for),
  FOREIGN KEY (checkin_ticket_id) REFERENCES checkin_tickets(id) ON DELETE CASCADE,
  FOREIGN KEY (parent_job_id) REFERENCES auto_checkin_jobs(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- CHECKIN LOGS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS checkin_logs (
  id VARCHAR(36) PRIMARY KEY,
  checkin_ticket_id VARCHAR(36),
  pnr VARCHAR(20) NOT NULL,
  airline ENUM('THY', 'PC', 'AJET', 'OTHER') NOT NULL,
  flight_number VARCHAR(20) NOT NULL,

  status ENUM('success', 'failure', 'pending', 'processing') NOT NULL,
  error_message TEXT,
  passenger_count INT DEFAULT 1,

  attempt_number INT DEFAULT 1,
  execution_time_ms INT,

  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

  INDEX idx_checkin_logs_ticket (checkin_ticket_id),
  INDEX idx_checkin_logs_pnr (pnr),
  INDEX idx_checkin_logs_status (status),
  INDEX idx_checkin_logs_created (created_at),
  FOREIGN KEY (checkin_ticket_id) REFERENCES checkin_tickets(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
