const cron = require('node-cron');
const { Worker } = require('worker_threads');
const path = require('path');
const { saveLog, updateJobStatus } = require('./database');

class WorkerSchedulerService {
  constructor() {
    this.scheduledJobs = new Map();
    this.tickets = [];
    this.customers = [];
    this.checkInJobs = [];
    this.activeWorkers = new Map();
    this.maxConcurrentWorkers = 3; // Limit concurrent operations
    this.workerPool = [];
    this.startScheduler();
  }

  startScheduler() {
    console.log('🕐 Starting optimized worker-based scheduler...');

    // Non-blocking cron jobs with immediate execution wrapped in setImmediate
    cron.schedule('* * * * *', () => {
      // Use setImmediate to prevent blocking the event loop
      setImmediate(() => this.queueScheduledCheckIns());
    }, {
      scheduled: true,
      timezone: "Europe/Istanbul" // THY timezone
    });

    cron.schedule('0 * * * *', () => {
      // Use setImmediate for non-blocking execution
      setImmediate(() => this.scheduleUpcomingCheckIns());
    }, {
      scheduled: true,
      timezone: "Europe/Istanbul"
    });

    // Clean up completed jobs every 6 hours to prevent memory buildup
    cron.schedule('0 */6 * * *', () => {
      setImmediate(() => this.cleanupCompletedJobs());
    }, {
      scheduled: true,
      timezone: "Europe/Istanbul"
    });

    console.log('✅ Worker scheduler started successfully');
  }

  // Non-blocking: Just identify and queue jobs with optimized filtering
  queueScheduledCheckIns() {
    try {
      const now = new Date();
      const readyJobs = [];

      // Optimized loop to avoid creating large intermediate arrays
      for (let i = 0; i < this.checkInJobs.length; i++) {
        const job = this.checkInJobs[i];
        if (job.status === 'scheduled' && job.scheduledFor <= now) {
          readyJobs.push(job);
          // Limit processing to prevent overwhelming the system
          if (readyJobs.length >= 10) break;
        }
      }

      // Process ready jobs without blocking
      for (const job of readyJobs) {
        if (this.activeWorkers.size < this.maxConcurrentWorkers) {
          // Use setImmediate to prevent blocking
          setImmediate(() => this.executeCheckinJobWorker(job));
        } else {
          // Mark as queued if workers are busy
          job.status = 'queued';
          console.log(`⏳ Job ${job.id} queued - workers busy (${this.activeWorkers.size}/${this.maxConcurrentWorkers})`);
        }
      }
    } catch (error) {
      console.error('❌ Error in queueScheduledCheckIns:', error.message);
    }
  }

  // Execute check-in using worker thread (non-blocking)
  async executeCheckinJobWorker(job) {
    const ticket = this.tickets.find(t => t.id === job.ticketId);
    if (!ticket) {
      job.status = 'failed';
      return;
    }

    console.log(`🚀 Spawning worker for check-in: ${ticket.pnr}`);

    job.status = 'processing';
    job.attempts += 1;
    job.lastAttempt = new Date();

    // Database'e log kaydet
    try {
      await saveLog({
        jobId: job.id,
        pnr: ticket.pnr,
        airline: ticket.airline || 'THY',
        percentage: 10,
        level: 'info',
        message: `Worker thread başlatılıyor - PNR: ${ticket.pnr}`,
        details: `Attempt ${job.attempts}`
      });
    } catch (err) {
      console.error('❌ Database log error:', err);
    }

    // Create worker for Selenium operations
    const worker = new Worker(path.join(__dirname, 'checkin-worker.js'), {
      workerData: {
        jobId: job.id,
        ticketData: {
          pnr: ticket.pnr,
          lastName: ticket.customer.lastName.toUpperCase(),
          firstName: ticket.customer.firstName,
          phone: ticket.customer.phone
        }
      }
    });

    this.activeWorkers.set(job.id, worker);

    // Handle worker completion (non-blocking)
    worker.on('message', (result) => {
      this.handleWorkerResult(job, ticket, result);
    });

    worker.on('error', (error) => {
      console.error(`❌ Worker error for ${ticket.pnr}:`, error);
      this.handleWorkerError(job, ticket, error);
    });

    worker.on('exit', (code) => {
      this.activeWorkers.delete(job.id);
      if (code !== 0) {
        console.error(`❌ Worker stopped with exit code ${code}`);
      }

      // Process queued jobs if worker slots available
      this.processQueuedJobs();
    });

    // Set worker timeout (30 seconds max)
    setTimeout(() => {
      if (this.activeWorkers.has(job.id)) {
        console.log(`⏰ Terminating worker for ${ticket.pnr} - timeout`);
        worker.terminate();
        this.handleWorkerTimeout(job, ticket);
      }
    }, 30000);
  }

  async handleWorkerResult(job, ticket, result) {
    console.log(`✅ Worker completed for ${ticket.pnr}:`, result.success);

    // Update ticket and job status
    if (result.success) {
      ticket.checkinStatus = 'completed';
      ticket.checkinCompletedAt = new Date();
      ticket.seat = result.seat;
      ticket.boardingPassUrl = result.boardingPassUrl;
      job.status = 'completed';

      // Database'e başarı log'u kaydet
      try {
        await saveLog({
          jobId: job.id,
          pnr: ticket.pnr,
          airline: ticket.airline || 'THY',
          percentage: 100,
          level: 'info',
          message: `✅ Check-in başarıyla tamamlandı - Koltuk: ${result.seat || 'N/A'}`,
          details: JSON.stringify({ seat: result.seat, boardingPassUrl: result.boardingPassUrl })
        });
      } catch (err) {
        console.error('❌ Database log error:', err);
      }
    } else {
      ticket.checkinStatus = 'failed';
      ticket.errorMessage = result.message;
      job.status = 'failed';

      // Database'e hata log'u kaydet
      try {
        await saveLog({
          jobId: job.id,
          pnr: ticket.pnr,
          airline: ticket.airline || 'THY',
          percentage: 0,
          level: 'error',
          message: `❌ Check-in başarısız: ${result.message}`,
          details: result.message
        });
      } catch (err) {
        console.error('❌ Database log error:', err);
      }

      this.scheduleRetry(job, ticket, result.message);
    }

    // Emit real-time update
    if (global.io) {
      global.io.emit('ticket_status_update', {
        ticketId: ticket.id,
        status: ticket.checkinStatus,
        message: result.message,
        ticket
      });
    }
  }

  handleWorkerError(job, ticket, error) {
    ticket.checkinStatus = 'failed';
    ticket.errorMessage = error.message;
    job.status = 'failed';

    this.scheduleRetry(job, ticket, error.message);

    if (global.io) {
      global.io.emit('ticket_status_update', {
        ticketId: ticket.id,
        status: 'failed',
        message: `Worker error: ${error.message}`,
        ticket
      });
    }
  }

  handleWorkerTimeout(job, ticket) {
    ticket.checkinStatus = 'failed';
    ticket.errorMessage = 'Check-in timeout';
    job.status = 'failed';

    this.scheduleRetry(job, ticket, 'Operation timeout');
  }

  // Process any queued jobs when workers become available (optimized)
  processQueuedJobs() {
    try {
      if (this.activeWorkers.size >= this.maxConcurrentWorkers) return;

      // Process up to available worker slots, avoiding full array scan
      const availableSlots = this.maxConcurrentWorkers - this.activeWorkers.size;
      let processed = 0;

      for (let i = 0; i < this.checkInJobs.length && processed < availableSlots; i++) {
        const job = this.checkInJobs[i];
        if (job.status === 'queued') {
          job.status = 'scheduled'; // Reset to scheduled
          setImmediate(() => this.executeCheckinJobWorker(job));
          processed++;
        }
      }

      if (processed > 0) {
        console.log(`🔄 Processed ${processed} queued jobs`);
      }
    } catch (error) {
      console.error('❌ Error in processQueuedJobs:', error.message);
    }
  }

  // Lightweight scheduling with performance optimizations
  scheduleUpcomingCheckIns() {
    try {
      const now = new Date();
      const in25Hours = new Date(now.getTime() + 25 * 60 * 60 * 1000);
      let scheduledCount = 0;
      const maxNewSchedules = 20; // Prevent overwhelming scheduling

      // Use for loop instead of filter for better performance
      for (let i = 0; i < this.tickets.length && scheduledCount < maxNewSchedules; i++) {
        const ticket = this.tickets[i];

        // Early exits for performance
        if (!ticket.isAutoCheckinEnabled) continue;
        if (ticket.checkinStatus !== 'waiting') continue;

        const checkinOpenTime = new Date(ticket.checkinOpenTime);
        if (checkinOpenTime >= now && checkinOpenTime <= in25Hours) {
          // Use setImmediate to prevent blocking
          setImmediate(() => this.scheduleCheckIn(ticket));
          scheduledCount++;
        }
      }

      if (scheduledCount > 0) {
        console.log(`📅 Scheduled ${scheduledCount} upcoming check-ins`);
      }
    } catch (error) {
      console.error('❌ Error in scheduleUpcomingCheckIns:', error.message);
    }
  }

  scheduleCheckIn(ticket) {
    const existingJob = this.checkInJobs.find(job =>
      job.ticketId === ticket.id &&
      job.status === 'scheduled'
    );

    if (existingJob) {
      console.log(`⏰ Check-in already scheduled for ticket ${ticket.pnr}`);
      return;
    }

    const checkinOpenTime = new Date(ticket.checkinOpenTime);
    const scheduledFor = new Date(checkinOpenTime.getTime() + 30 * 1000);

    const job = {
      id: this.generateId(),
      ticketId: ticket.id,
      scheduledFor,
      status: 'scheduled',
      attempts: 0,
      createdAt: new Date(),
      updatedAt: new Date()
    };

    this.checkInJobs.push(job);
    console.log(`⏰ Auto check-in scheduled for ${ticket.pnr} at ${scheduledFor.toISOString()}`);
  }

  scheduleRetry(originalJob, ticket, errorMessage) {
    if (originalJob.attempts >= 3) {
      console.log(`❌ Max retry attempts reached for ${ticket.pnr}`);
      return;
    }

    const retryDelays = [5 * 60 * 1000, 15 * 60 * 1000, 30 * 60 * 1000];
    const retryDelay = retryDelays[originalJob.attempts - 1] || retryDelays[retryDelays.length - 1];
    const retryTime = new Date(Date.now() + retryDelay);

    const retryJob = {
      id: this.generateId(),
      ticketId: ticket.id,
      scheduledFor: retryTime,
      status: 'scheduled',
      attempts: 0,
      parentJobId: originalJob.id,
      retryReason: errorMessage,
      createdAt: new Date(),
      updatedAt: new Date()
    };

    this.checkInJobs.push(retryJob);
    console.log(`🔄 Retry scheduled for ${ticket.pnr} at ${retryTime.toISOString()}`);
  }

  // Status and utility methods (unchanged from original)
  getStatus() {
    const now = new Date();

    const stats = {
      totalJobs: this.checkInJobs.length,
      scheduledJobs: this.checkInJobs.filter(j => j.status === 'scheduled').length,
      queuedJobs: this.checkInJobs.filter(j => j.status === 'queued').length,
      processingJobs: this.checkInJobs.filter(j => j.status === 'processing').length,
      completedJobs: this.checkInJobs.filter(j => j.status === 'completed').length,
      failedJobs: this.checkInJobs.filter(j => j.status === 'failed').length,
      activeWorkers: this.activeWorkers.size,
      maxWorkers: this.maxConcurrentWorkers
    };

    return {
      isRunning: true,
      stats,
      nextJobTime: this.getNextJobTime()
    };
  }

  getNextJobTime() {
    const now = new Date();
    const upcomingJobs = this.checkInJobs
      .filter(j => j.status === 'scheduled' && j.scheduledFor > now)
      .sort((a, b) => a.scheduledFor.getTime() - b.scheduledFor.getTime());

    return upcomingJobs.length > 0 ? upcomingJobs[0].scheduledFor : null;
  }

  setDataSources(tickets, customers, checkInJobs) {
    this.tickets = tickets;
    this.customers = customers;
    this.checkInJobs = checkInJobs;
  }

  generateId() {
    return Date.now().toString() + Math.random().toString(36).substr(2, 9);
  }

  // Clean up completed and old failed jobs to prevent memory leaks
  cleanupCompletedJobs() {
    try {
      const now = new Date();
      const oneDayAgo = new Date(now.getTime() - 24 * 60 * 60 * 1000);
      const initialCount = this.checkInJobs.length;

      // Remove completed jobs older than 1 day and failed jobs older than 1 day
      this.checkInJobs = this.checkInJobs.filter(job => {
        if (job.status === 'completed' && job.updatedAt < oneDayAgo) {
          return false; // Remove old completed jobs
        }
        if (job.status === 'failed' && job.attempts >= 3 && job.updatedAt < oneDayAgo) {
          return false; // Remove old failed jobs that won't retry
        }
        return true; // Keep job
      });

      const removedCount = initialCount - this.checkInJobs.length;
      if (removedCount > 0) {
        console.log(`🧹 Cleaned up ${removedCount} old jobs (${this.checkInJobs.length} remaining)`);

        // Force garbage collection if available
        if (global.gc) {
          global.gc();
        }
      }
    } catch (error) {
      console.error('❌ Error in cleanupCompletedJobs:', error.message);
    }
  }

  // Cleanup and shutdown
  async shutdown() {
    console.log('🛑 Shutting down worker scheduler...');

    // Terminate all active workers
    for (const [jobId, worker] of this.activeWorkers) {
      await worker.terminate();
    }

    this.activeWorkers.clear();
    console.log('✅ All workers terminated');
  }
}

module.exports = WorkerSchedulerService;