/**
 * Timeout wrapper for preventing blocking operations in cron jobs
 * Ensures all external API calls and operations have proper timeout controls
 */

class TimeoutWrapper {
  constructor(defaultTimeout = 5000) {
    this.defaultTimeout = defaultTimeout;
  }

  /**
   * Wrap any promise with a timeout
   */
  async withTimeout(promise, timeout = this.defaultTimeout, operation = 'Operation') {
    const timeoutPromise = new Promise((_, reject) => {
      const id = setTimeout(() => {
        reject(new Error(`${operation} timeout after ${timeout}ms`));
      }, timeout);

      // Clear timeout if the promise resolves
      promise.finally(() => clearTimeout(id));
    });

    try {
      return await Promise.race([promise, timeoutPromise]);
    } catch (error) {
      if (error.message.includes('timeout')) {
        console.warn(`⏰ ${operation} timed out after ${timeout}ms`);
      }
      throw error;
    }
  }

  /**
   * Wrap API calls with circuit breaker pattern
   */
  async withCircuitBreaker(apiCall, options = {}) {
    const {
      timeout = this.defaultTimeout,
      maxRetries = 2,
      retryDelay = 1000,
      operation = 'API call'
    } = options;

    let lastError;

    for (let attempt = 1; attempt <= maxRetries + 1; attempt++) {
      try {
        return await this.withTimeout(apiCall(), timeout, operation);
      } catch (error) {
        lastError = error;

        if (attempt <= maxRetries) {
          console.warn(`⚠️ ${operation} attempt ${attempt} failed: ${error.message}, retrying in ${retryDelay}ms`);
          await this.delay(retryDelay * attempt); // Exponential backoff
        }
      }
    }

    throw new Error(`${operation} failed after ${maxRetries + 1} attempts: ${lastError.message}`);
  }

  /**
   * Wrap database operations
   */
  async withDatabaseTimeout(dbOperation, timeout = 3000) {
    return this.withTimeout(dbOperation(), timeout, 'Database operation');
  }

  /**
   * Wrap file operations
   */
  async withFileTimeout(fileOperation, timeout = 2000) {
    return this.withTimeout(fileOperation(), timeout, 'File operation');
  }

  /**
   * Batch operations with timeout and concurrency control
   */
  async batchWithTimeout(operations, options = {}) {
    const {
      maxConcurrency = 3,
      timeout = this.defaultTimeout,
      operation = 'Batch operation'
    } = options;

    const chunks = this.chunk(operations, maxConcurrency);
    const results = [];

    for (const chunk of chunks) {
      const chunkPromises = chunk.map(op =>
        this.withTimeout(op(), timeout, operation).catch(err => ({ error: err.message }))
      );

      const chunkResults = await Promise.all(chunkPromises);
      results.push(...chunkResults);
    }

    return results;
  }

  /**
   * Non-blocking delay
   */
  async delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }

  /**
   * Utility: Split array into chunks
   */
  chunk(array, size) {
    const chunks = [];
    for (let i = 0; i < array.length; i += size) {
      chunks.push(array.slice(i, i + size));
    }
    return chunks;
  }

  /**
   * Monitor operation duration
   */
  async monitor(operation, name = 'Operation') {
    const startTime = Date.now();

    try {
      const result = await operation();
      const duration = Date.now() - startTime;

      if (duration > 5000) {
        console.warn(`🐌 Slow ${name}: ${duration}ms`);
      } else if (duration > 1000) {
        console.log(`⏱️ ${name}: ${duration}ms`);
      }

      return result;
    } catch (error) {
      const duration = Date.now() - startTime;
      console.error(`❌ ${name} failed after ${duration}ms: ${error.message}`);
      throw error;
    }
  }

  /**
   * Safe async wrapper that never throws
   */
  async safe(operation, fallback = null, operation_name = 'Operation') {
    try {
      return await this.withTimeout(operation(), this.defaultTimeout, operation_name);
    } catch (error) {
      console.error(`❌ Safe ${operation_name} failed: ${error.message}`);
      return fallback;
    }
  }

  /**
   * Memory-safe operation wrapper
   */
  async withMemoryLimit(operation, maxMemoryMB = 100) {
    const initialMemory = process.memoryUsage().heapUsed;

    try {
      const result = await operation();

      const finalMemory = process.memoryUsage().heapUsed;
      const memoryDelta = (finalMemory - initialMemory) / 1024 / 1024;

      if (memoryDelta > maxMemoryMB) {
        console.warn(`🧠 High memory usage: ${memoryDelta.toFixed(2)}MB`);

        // Force garbage collection if available
        if (global.gc) {
          global.gc();
        }
      }

      return result;
    } catch (error) {
      throw error;
    }
  }
}

// Singleton instance
const timeoutWrapper = new TimeoutWrapper();

// Example usage for Amadeus service
class AmadeusServiceWithTimeout {
  constructor(amadeusService) {
    this.amadeus = amadeusService;
    this.timeout = timeoutWrapper;
  }

  async getFlightStatus(airline, flightNumber, date) {
    return this.timeout.withCircuitBreaker(
      () => this.amadeus.getFlightStatus(airline, flightNumber, date),
      {
        timeout: 8000,
        maxRetries: 2,
        operation: `Amadeus flight status ${airline}${flightNumber}`
      }
    );
  }

  async searchFlights(params) {
    return this.timeout.withTimeout(
      this.amadeus.searchFlights(params),
      10000,
      'Amadeus flight search'
    );
  }
}

// Example usage for FlightStatusService
class FlightStatusServiceWithTimeout {
  constructor(flightStatusService) {
    this.service = flightStatusService;
    this.timeout = timeoutWrapper;
  }

  async getFlightStatus(airline, flightNumber, date) {
    return this.timeout.monitor(
      () => this.timeout.withTimeout(
        this.service.getFlightStatus(airline, flightNumber, date),
        6000,
        'Flight status lookup'
      ),
      `Flight status ${airline}${flightNumber}`
    );
  }

  async updateMultipleTickets(tickets) {
    const operations = tickets.map(ticket =>
      () => this.service.updateTicketWithFlightStatus(ticket)
    );

    return this.timeout.batchWithTimeout(operations, {
      maxConcurrency: 3,
      timeout: 5000,
      operation: 'Ticket status update'
    });
  }
}

module.exports = {
  TimeoutWrapper,
  timeoutWrapper,
  AmadeusServiceWithTimeout,
  FlightStatusServiceWithTimeout
};