const { Builder, By, until } = require('selenium-webdriver');
const chrome = require('selenium-webdriver/chrome');
const fs = require('fs').promises;
const path = require('path');

class SeleniumService {
  constructor() {
    this.driver = null;
    this.isRunning = false;
  }

  async initDriver(headless = true) {
    const options = new chrome.Options();
    
    if (headless) {
      options.addArguments('--headless');
      options.addArguments('--no-sandbox');
      options.addArguments('--disable-dev-shm-usage');
      options.addArguments('--disable-gpu');
      options.addArguments('--window-size=1920,1080');
    }
    
    options.addArguments('--disable-web-security');
    options.addArguments('--allow-running-insecure-content');
    
    // Download ayarları
    const downloadDir = path.resolve('./downloads/boarding-passes');
    await this.ensureDirectoryExists(downloadDir);
    
    options.setUserPreferences({
      'download.default_directory': downloadDir,
      'download.prompt_for_download': false,
      'plugins.always_open_pdf_externally': true
    });

    this.driver = await new Builder()
      .forBrowser('chrome')
      .setChromeOptions(options)
      .build();
    
    this.isRunning = true;
    return this.driver;
  }

  async ensureDirectoryExists(dir) {
    try {
      await fs.mkdir(dir, { recursive: true });
    } catch (err) {
      if (err.code !== 'EEXIST') throw err;
    }
  }

  async performCheckin({ pnr, lastName, firstName, phone }) {
    if (!this.driver) {
      await this.initDriver();
    }

    const result = {
      success: false,
      message: '',
      flightInfo: null,
      boardingPassUrl: null,
      screenshots: []
    };

    try {
      console.log(`🚀 Starting check-in for PNR: ${pnr}, Passenger: ${lastName}`);
      
      // 1. THY ana sayfasına git
      await this.driver.get('https://www.turkishairlines.com');
      await this.driver.sleep(2000);
      
      // Screenshot al
      const homeScreenshot = await this.takeScreenshot('homepage');
      result.screenshots.push(homeScreenshot);

      // 2. Cookie popup'ını kapat
      try {
        const cookieButton = await this.driver.wait(
          until.elementLocated(By.xpath("//button[contains(text(), 'Tüm çerezleri')]")), 
          5000
        );
        await cookieButton.click();
        await this.driver.sleep(1000);
      } catch (err) {
        console.log('Cookie popup bulunamadı veya kapatılamadı');
      }

      // 3. Check-in tab'ına tıkla
      const checkinTab = await this.driver.wait(
        until.elementLocated(By.xpath("//button[contains(@class, 'tab') and contains(., 'Check-in')]")),
        10000
      );
      await checkinTab.click();
      await this.driver.sleep(3000);

      // 4. Form alanlarını doldur
      const pnrInput = await this.driver.wait(
        until.elementLocated(By.id('pnrnumberinput')),
        10000
      );
      await pnrInput.clear();
      await pnrInput.sendKeys(pnr);

      const surnameInput = await this.driver.findElement(By.id('pnrsurnameinput'));
      await surnameInput.clear();
      await surnameInput.sendKeys(lastName);

      // Screenshot al
      const formScreenshot = await this.takeScreenshot('form_filled');
      result.screenshots.push(formScreenshot);

      // 5. Check-in butonuna tıkla
      const checkinButton = await this.driver.findElement(
        By.xpath("//button[contains(text(), 'Check-in')]")
      );
      await checkinButton.click();
      await this.driver.sleep(5000);

      // 6. Sonuç sayfasını kontrol et
      const currentUrl = await this.driver.getCurrentUrl();
      console.log('Current URL:', currentUrl);

      if (currentUrl.includes('binis-karti') || currentUrl.includes('boarding')) {
        // Başarılı check-in
        result.success = true;
        result.message = 'Check-in başarılı! Biniş kartı hazırlandı.';
        
        // Screenshot al
        const successScreenshot = await this.takeScreenshot('success');
        result.screenshots.push(successScreenshot);

        // Uçuş bilgilerini çıkar
        try {
          const flightInfo = await this.extractFlightInfo();
          result.flightInfo = flightInfo;
        } catch (err) {
          console.log('Uçuş bilgileri çıkarılamadı:', err.message);
        }

        // SMS ile gönder (isteğe bağlı)
        if (phone) {
          try {
            await this.sendSMS(phone);
            result.message += ' SMS gönderildi.';
          } catch (err) {
            console.log('SMS gönderilemedi:', err.message);
          }
        }

      } else {
        // Hata durumu
        result.success = false;
        result.message = 'Check-in başarısız. PNR veya soyisim bilgileri hatalı olabilir.';
        
        const errorScreenshot = await this.takeScreenshot('error');
        result.screenshots.push(errorScreenshot);
      }

    } catch (error) {
      console.error('Check-in hatası:', error);
      result.success = false;
      result.message = `Check-in sırasında hata: ${error.message}`;
      
      const errorScreenshot = await this.takeScreenshot('exception');
      result.screenshots.push(errorScreenshot);
    }

    return result;
  }

  async extractFlightInfo() {
    // Uçuş bilgilerini sayfadan çıkarmaya çalış
    try {
      await this.driver.sleep(2000);
      
      // Bu kısım THY'nin actual DOM yapısına göre güncellenecek
      const flightInfo = {
        flight: 'TK 1234', // Placeholder
        from: 'İstanbul (IST)',
        to: 'Antalya (AYT)', 
        date: new Date().toLocaleDateString('tr-TR'),
        seat: '12A'
      };
      
      return flightInfo;
    } catch (err) {
      throw new Error('Uçuş bilgileri çıkarılamadı');
    }
  }

  async sendSMS(phone) {
    try {
      // SMS gönder butonunu bul ve tıkla
      const smsButton = await this.driver.findElement(
        By.xpath("//button[contains(text(), 'SMS ile gönder')]")
      );
      await smsButton.click();
      await this.driver.sleep(2000);

      // Telefon numarasını gir
      const phoneInput = await this.driver.findElement(
        By.css("input[type='tel'], input[type='text']")
      );
      await phoneInput.clear();
      await phoneInput.sendKeys(phone);

      // Gönder butonuna tıkla
      const sendButton = await this.driver.findElement(
        By.xpath("//button[contains(text(), 'Gönder')]")
      );
      await sendButton.click();
      await this.driver.sleep(3000);

      console.log(`📱 SMS sent to ${phone}`);
    } catch (err) {
      throw new Error(`SMS gönderilemedi: ${err.message}`);
    }
  }

  async takeScreenshot(name) {
    try {
      const screenshot = await this.driver.takeScreenshot();
      const filename = `${name}_${Date.now()}.png`;
      const filepath = path.resolve('./downloads/screenshots', filename);
      
      await this.ensureDirectoryExists(path.dirname(filepath));
      await fs.writeFile(filepath, screenshot, 'base64');
      
      return {
        filename,
        path: filepath,
        url: `/downloads/screenshots/${filename}`
      };
    } catch (err) {
      console.error('Screenshot alınamadı:', err);
      return null;
    }
  }

  async close() {
    if (this.driver) {
      await this.driver.quit();
      this.driver = null;
      this.isRunning = false;
    }
  }

  // Cleanup için process exit handler
  setupCleanup() {
    process.on('SIGINT', async () => {
      console.log('🛑 Shutting down Selenium service...');
      await this.close();
      process.exit(0);
    });
  }
}

module.exports = SeleniumService;