const cron = require('node-cron');
const SeleniumService = require('./SeleniumService');

class SchedulerService {
  constructor() {
    this.scheduledJobs = new Map();
    this.tickets = []; // Will be replaced with database
    this.customers = []; // Will be replaced with database
    this.checkInJobs = [];
    this.startScheduler();
  }

  startScheduler() {
    // Run every minute to check for pending check-ins
    console.log('🕐 Starting automated scheduler...');
    
    cron.schedule('* * * * *', () => {
      this.processScheduledCheckIns();
    });

    // Run every hour to schedule upcoming check-ins
    cron.schedule('0 * * * *', () => {
      this.scheduleUpcomingCheckIns();
    });

    console.log('✅ Scheduler started successfully');
  }

  // Process check-ins that are ready to execute
  async processScheduledCheckIns() {
    const now = new Date();
    
    // Find jobs that are scheduled and ready to execute
    const readyJobs = this.checkInJobs.filter(job => 
      job.status === 'scheduled' && 
      job.scheduledFor <= now
    );

    for (const job of readyJobs) {
      await this.executeCheckinJob(job);
    }
  }

  // Schedule check-ins for tickets that have reached the 24-hour window
  scheduleUpcomingCheckIns() {
    const now = new Date();
    const in25Hours = new Date(now.getTime() + 25 * 60 * 60 * 1000);

    // Find tickets that will have check-in open in the next hour
    const ticketsToSchedule = this.tickets.filter(ticket => {
      if (!ticket.isAutoCheckinEnabled) return false;
      if (ticket.checkinStatus !== 'waiting') return false;
      
      const checkinOpenTime = new Date(ticket.checkinOpenTime);
      return checkinOpenTime >= now && checkinOpenTime <= in25Hours;
    });

    for (const ticket of ticketsToSchedule) {
      this.scheduleCheckIn(ticket);
    }
  }

  // Schedule a single check-in
  scheduleCheckIn(ticket) {
    // Check if already scheduled
    const existingJob = this.checkInJobs.find(job => 
      job.ticketId === ticket.id && 
      job.status === 'scheduled'
    );

    if (existingJob) {
      console.log(`⏰ Check-in already scheduled for ticket ${ticket.pnr}`);
      return;
    }

    const checkinOpenTime = new Date(ticket.checkinOpenTime);
    // Schedule 30 seconds after check-in opens to ensure it's available
    const scheduledFor = new Date(checkinOpenTime.getTime() + 30 * 1000);

    const job = {
      id: this.generateId(),
      ticketId: ticket.id,
      scheduledFor,
      status: 'scheduled',
      attempts: 0,
      createdAt: new Date(),
      updatedAt: new Date()
    };

    this.checkInJobs.push(job);
    
    console.log(`⏰ Auto check-in scheduled for ${ticket.pnr} at ${scheduledFor.toISOString()}`);
  }

  // Execute a scheduled check-in job
  async executeCheckinJob(job) {
    const ticket = this.tickets.find(t => t.id === job.ticketId);
    if (!ticket) {
      console.log(`❌ Ticket not found for job ${job.id}`);
      job.status = 'failed';
      return;
    }

    console.log(`🤖 Executing auto check-in for ${ticket.pnr}`);
    
    job.status = 'processing';
    job.attempts += 1;
    job.lastAttempt = new Date();
    job.updatedAt = new Date();

    // Update ticket status
    ticket.checkinStatus = 'processing';
    ticket.checkinAttempts += 1;
    ticket.lastCheckinAttempt = new Date();
    ticket.updatedAt = new Date();

    try {
      const seleniumService = new SeleniumService();
      
      const result = await seleniumService.performCheckin({
        pnr: ticket.pnr,
        lastName: ticket.customer.lastName.toUpperCase(),
        firstName: ticket.customer.firstName,
        phone: ticket.customer.phone
      });

      // Update ticket based on result
      if (result.success) {
        ticket.checkinStatus = 'completed';
        ticket.checkinCompletedAt = new Date();
        ticket.seat = result.seat;
        ticket.boardingPassUrl = result.boardingPassUrl;
        ticket.errorMessage = null;
        job.status = 'completed';
        
        console.log(`✅ Auto check-in completed for ${ticket.pnr}: ${result.message}`);
      } else {
        ticket.checkinStatus = 'failed';
        ticket.errorMessage = result.message;
        job.status = 'failed';
        
        console.log(`❌ Auto check-in failed for ${ticket.pnr}: ${result.message}`);
        
        // Schedule retry if it's a temporary failure
        this.scheduleRetry(job, ticket, result.message);
      }

      ticket.updatedAt = new Date();
      job.updatedAt = new Date();

      await seleniumService.close();

      // Emit real-time update if WebSocket is available
      if (global.io) {
        global.io.emit('ticket_status_update', {
          ticketId: ticket.id,
          status: ticket.checkinStatus,
          message: result.message,
          ticket
        });
      }

    } catch (error) {
      console.error(`❌ Check-in error for ${ticket.pnr}:`, error);
      
      ticket.checkinStatus = 'failed';
      ticket.errorMessage = error.message;
      ticket.updatedAt = new Date();
      
      job.status = 'failed';
      job.updatedAt = new Date();

      // Schedule retry for system errors
      this.scheduleRetry(job, ticket, error.message);

      if (global.io) {
        global.io.emit('ticket_status_update', {
          ticketId: ticket.id,
          status: 'failed',
          message: `System error: ${error.message}`,
          ticket
        });
      }
    }
  }

  // Schedule a retry for failed check-ins
  scheduleRetry(originalJob, ticket, errorMessage) {
    if (originalJob.attempts >= 3) {
      console.log(`❌ Max retry attempts reached for ${ticket.pnr}`);
      return;
    }

    // Retry after increasing intervals: 5min, 15min, 30min
    const retryDelays = [5 * 60 * 1000, 15 * 60 * 1000, 30 * 60 * 1000];
    const retryDelay = retryDelays[originalJob.attempts - 1] || retryDelays[retryDelays.length - 1];
    
    const retryTime = new Date(Date.now() + retryDelay);

    const retryJob = {
      id: this.generateId(),
      ticketId: ticket.id,
      scheduledFor: retryTime,
      status: 'scheduled',
      attempts: 0,
      parentJobId: originalJob.id,
      retryReason: errorMessage,
      createdAt: new Date(),
      updatedAt: new Date()
    };

    this.checkInJobs.push(retryJob);
    
    console.log(`🔄 Retry scheduled for ${ticket.pnr} at ${retryTime.toISOString()} (attempt ${originalJob.attempts})`);
  }

  // Get scheduler status
  getStatus() {
    const now = new Date();
    
    const stats = {
      totalJobs: this.checkInJobs.length,
      scheduledJobs: this.checkInJobs.filter(j => j.status === 'scheduled').length,
      processingJobs: this.checkInJobs.filter(j => j.status === 'processing').length,
      completedJobs: this.checkInJobs.filter(j => j.status === 'completed').length,
      failedJobs: this.checkInJobs.filter(j => j.status === 'failed').length,
      upcomingJobs: this.checkInJobs.filter(j => 
        j.status === 'scheduled' && 
        j.scheduledFor > now
      ).length,
      overdueJobs: this.checkInJobs.filter(j => 
        j.status === 'scheduled' && 
        j.scheduledFor <= now
      ).length
    };

    return {
      isRunning: true,
      stats,
      nextJobTime: this.getNextJobTime()
    };
  }

  // Get next scheduled job time
  getNextJobTime() {
    const now = new Date();
    const upcomingJobs = this.checkInJobs
      .filter(j => j.status === 'scheduled' && j.scheduledFor > now)
      .sort((a, b) => a.scheduledFor.getTime() - b.scheduledFor.getTime());

    return upcomingJobs.length > 0 ? upcomingJobs[0].scheduledFor : null;
  }

  // Set data sources (to be called by the routes)
  setDataSources(tickets, customers, checkInJobs) {
    this.tickets = tickets;
    this.customers = customers;
    this.checkInJobs = checkInJobs;
  }

  // Utility function
  generateId() {
    return Date.now().toString() + Math.random().toString(36).substr(2, 9);
  }

  // Cancel scheduled job
  cancelJob(jobId) {
    const jobIndex = this.checkInJobs.findIndex(j => j.id === jobId);
    if (jobIndex !== -1) {
      const job = this.checkInJobs[jobIndex];
      if (job.status === 'scheduled') {
        job.status = 'cancelled';
        job.updatedAt = new Date();
        console.log(`❌ Cancelled scheduled job ${jobId}`);
        return true;
      }
    }
    return false;
  }

  // Force execute job
  async forceExecuteJob(jobId) {
    const job = this.checkInJobs.find(j => j.id === jobId);
    if (job && job.status === 'scheduled') {
      console.log(`⚡ Force executing job ${jobId}`);
      await this.executeCheckinJob(job);
      return true;
    }
    return false;
  }

  // Get scheduled jobs for a ticket
  getJobsForTicket(ticketId) {
    return this.checkInJobs.filter(j => j.ticketId === ticketId);
  }

  // Clean up old jobs (keep last 1000)
  cleanup() {
    const maxJobs = 1000;
    if (this.checkInJobs.length > maxJobs) {
      this.checkInJobs.sort((a, b) => b.createdAt.getTime() - a.createdAt.getTime());
      this.checkInJobs = this.checkInJobs.slice(0, maxJobs);
      console.log(`🧹 Cleaned up old jobs, keeping ${maxJobs} most recent`);
    }
  }
}

module.exports = SchedulerService;