const { Builder, By, until } = require('selenium-webdriver');
const chrome = require('selenium-webdriver/chrome');
const fs = require('fs').promises;
const path = require('path');

class PegasusSeleniumService {
  constructor() {
    this.driver = null;
    this.isRunning = false;
  }

  async initDriver(headless = true) {
    const options = new chrome.Options();
    
    if (headless) {
      options.addArguments('--headless');
      options.addArguments('--no-sandbox');
      options.addArguments('--disable-dev-shm-usage');
      options.addArguments('--disable-gpu');
      options.addArguments('--window-size=1920,1080');
    }
    
    // Anti-bot detection
    options.addArguments('--disable-web-security');
    options.addArguments('--allow-running-insecure-content');
    options.addArguments('--disable-blink-features=AutomationControlled');
    options.addArguments('--user-agent=Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36');
    options.addArguments('--disable-extensions');
    options.addArguments('--disable-plugins');
    options.addArguments('--disable-images'); // Hız için resimleri yüklememe
    
    // Exclude automation switches
    options.excludeSwitches(['enable-automation']);
    options.addArguments('--disable-blink-features=AutomationControlled');
    
    // Dil ve lokasyon ayarları
    options.addArguments('--lang=tr-TR');
    options.addArguments('--accept-lang=tr-TR,tr;q=0.9,en;q=0.8');
    
    // Download ayarları
    const downloadDir = path.resolve('./downloads/boarding-passes');
    await this.ensureDirectoryExists(downloadDir);
    
    options.setUserPreferences({
      'download.default_directory': downloadDir,
      'download.prompt_for_download': false,
      'plugins.always_open_pdf_externally': true
    });

    this.driver = await new Builder()
      .forBrowser('chrome')
      .setChromeOptions(options)
      .build();
    
    // Anti-detection JavaScript injection
    await this.driver.executeScript(`
      Object.defineProperty(navigator, 'webdriver', {
        get: () => undefined,
      });
      
      // Chrome runtime'ı gizle
      window.chrome = {
        runtime: {}
      };
      
      // Permissions query'yi override et
      const originalQuery = window.navigator.permissions.query;
      window.navigator.permissions.query = (parameters) => (
        parameters.name === 'notifications' ?
          Promise.resolve({ state: Notification.permission }) :
          originalQuery(parameters)
      );
      
      // Languages ayarı
      Object.defineProperty(navigator, 'languages', {
        get: () => ['tr-TR', 'tr', 'en-US', 'en'],
      });
    `);
    
    this.isRunning = true;
    return this.driver;
  }

  async ensureDirectoryExists(dir) {
    try {
      await fs.mkdir(dir, { recursive: true });
    } catch (err) {
      if (err.code !== 'EEXIST') throw err;
    }
  }

  // Human-like random delay
  async humanDelay(min = 1000, max = 3000) {
    const delay = Math.floor(Math.random() * (max - min + 1)) + min;
    await new Promise(resolve => setTimeout(resolve, delay));
  }

  async performCheckin({ pnr, lastName, firstName, phone, flightType, passportNumber, visaNumber }) {
    if (!this.driver) {
      await this.initDriver(false); // Görünür browser ile test
    }

    const result = {
      success: false,
      message: '',
      flightInfo: null,
      boardingPassUrl: null,
      screenshots: []
    };

    try {
      console.log(`🚀 Starting Pegasus check-in for PNR: ${pnr}, Passenger: ${lastName}`);
      
      // 1. Pegasus ana sayfasına git - human-like behavior
      console.log('🌍 Pegasus ana sayfasına gidiliyor...');
      await this.driver.get('https://www.flypgs.com/');
      await this.humanDelay(3000, 5000); // Random delay 3-5 saniye
      
      // Screenshot al
      const homeScreenshot = await this.takeScreenshot('pegasus_homepage');
      result.screenshots.push(homeScreenshot);

      // 2. Cookie popup'ını kapat
      try {
        const cookieButtons = [
          "//button[contains(text(), 'Kabul Et')]",
          "//button[contains(text(), 'Accept')]",
          "//button[contains(@class, 'accept')]",
          "//button[contains(@class, 'cookie')]"
        ];

        for (const selector of cookieButtons) {
          try {
            const cookieButton = await this.driver.wait(
              until.elementLocated(By.xpath(selector)), 
              3000
            );
            await cookieButton.click();
            await this.driver.sleep(1000);
            console.log('Cookie popup kapatıldı');
            break;
          } catch (err) {
            // Try next selector
          }
        }
      } catch (err) {
        console.log('Cookie popup bulunamadı veya kapatılamadı');
      }

      // 3. Check-in tab'ına tıkla - Katalon'dan
      try {
        const checkinTab = await this.driver.wait(
          until.elementLocated(By.xpath("//ul[@id='rc_tabWrapper']/li[2]")), 
          10000
        );
        await checkinTab.click();
        console.log('✅ Check-in tab\'ı tıklandı');
        await this.driver.sleep(2000);
      } catch (err) {
        console.log('❌ Check-in tab bulunamadı:', err.message);
      }

      // 4. Soyisim alanını bul ve doldur - Katalon: id=OnlineCheckInSurnameField
      console.log('📍 Soyisim alanı dolduruluyor...');
      try {
        const surnameField = await this.driver.wait(
          until.elementLocated(By.id('OnlineCheckInSurnameField')), 
          10000
        );
        await surnameField.click();
        await surnameField.clear();
        await surnameField.sendKeys(lastName.toUpperCase());
        console.log('✅ Soyisim girildi:', lastName);
        await this.driver.sleep(1000);
      } catch (err) {
        console.log('❌ Soyisim alanı bulunamadı:', err.message);
        throw new Error('Soyisim alanı bulunamadı');
      }

      // 5. PNR alanını bul ve doldur - Katalon: id=OnlineCheckInCode  
      console.log('📍 PNR alanı doldurluyor...');
      const checkinSelectors = [
        "OnlineCheckInCode" // Katalon'dan ID
      ];

      let pnrInput = null;
      
      for (const selector of checkinSelectors) {
        try {
          pnrInput = await this.driver.wait(
            until.elementLocated(By.id(selector)),
            10000
          );
          console.log(`PNR input bulundu: ${selector}`);
          break;
        } catch (err) {
          console.log(`PNR input bulunamadı: ${selector}`);
        }
      }

      if (!pnrInput) {
        throw new Error('PNR input alanı bulunamadı');
      }

      await pnrInput.clear();
      await pnrInput.sendKeys(pnr);
      console.log('✅ PNR girildi:', pnr);
      await this.driver.sleep(1000);

      // 6. Devam butonunu bul ve tıkla - Katalon'dan başarılı selector
      console.log('📍 Devam butonu aranıyor...');
      const katalonSelectors = [
        "button.SearchButton__base-button.SearchButton__custom-size-button.SearchButton__orange-button.js-online-checkIn-search-button", // Katalon'dan başarılı CSS
        ".js-online-checkIn-search-button", // CSS class
        "[class*='SearchButton'][class*='orange-button']" // CSS partial match
      ];

      let continueButton = null;
      for (const selector of katalonSelectors) {
        try {
          console.log(`🔍 Deneniyor: ${selector}`);
          continueButton = await this.driver.wait(until.elementLocated(By.css(selector)), 5000);
          
          const isDisplayed = await continueButton.isDisplayed();
          const isEnabled = await continueButton.isEnabled();
          console.log(`✅ Buton bulundu: ${selector} - Görünür: ${isDisplayed}, Aktif: ${isEnabled}`);
          
          if (isDisplayed && isEnabled) {
            break;
          } else {
            continueButton = null;
          }
          
        } catch (err) {
          console.log('❌ Bulunamadı:', selector);
        }
      }

      if (!continueButton) {
        throw new Error('Devam butonu bulunamadı');
      }

      await continueButton.click();
      console.log('✅ Devam butonu tıklandı');
      await this.driver.sleep(5000);

      // Screenshot al
      const formScreenshot = await this.takeScreenshot('pegasus_form_filled');
      result.screenshots.push(formScreenshot);

      // 7. Yönlendirme kontolü
      const newUrl = await this.driver.getCurrentUrl();
      console.log('🔍 Yeni URL:', newUrl);
      
      if (!newUrl.includes('check-in')) {
        throw new Error('Check-in sayfasına yönlendirilemedi');
      }
      
      console.log('✅ Check-in sayfasına yönlendirildi!');

      // 8. Sonraki sayfada CHECK-IN butonunu bul ve tıkla
      const checkinButtonSelectors = [
        "//button[contains(text(), 'CHECK-IN')]",
        "//button[contains(text(), 'Check-in')]",
        "//button[contains(text(), 'CHECK IN')]",
        "//button[contains(@class, 'checkin')]",
        "//input[@value*='CHECK' or @value*='check']"
      ];

      let checkinButton = null;
      
      for (const selector of checkinButtonSelectors) {
        try {
          checkinButton = await this.driver.wait(
            until.elementLocated(By.xpath(selector)),
            10000
          );
          console.log(`Check-in button bulundu: ${selector}`);
          break;
        } catch (err) {
          console.log(`Check-in button bulunamadı: ${selector}`);
        }
      }

      if (checkinButton) {
        await checkinButton.click();
        await this.driver.sleep(5000);
        console.log('Check-in butonu tıklandı');
      } else {
        console.log('Check-in butonu bulunamadı, devam ediliyor...');
      }

      // 9. Yolcu listesi sayfasında tüm yolcuları seç
      await this.selectAllPassengers();

      // 10. Check-in formunda pasaport bilgilerini doldur (dış hat için)
      if (flightType === 'international' && passportNumber) {
        await this.fillInternationalDetails({ passportNumber, visaNumber });
      }

      // 11. Final check-in işlemini tamamla (yolcu seçimi sonrası devam)
      await this.completeCheckin();

      // 11. Sonuç sayfasını kontrol et
      const finalUrl = await this.driver.getCurrentUrl();
      console.log('Final URL:', finalUrl);

      if (finalUrl.includes('boarding') || finalUrl.includes('binis') || finalUrl.includes('checkin-success')) {
        // Başarılı check-in
        result.success = true;
        result.message = 'Pegasus check-in başarılı! Biniş kartı hazırlandı.';
        
        const successScreenshot = await this.takeScreenshot('pegasus_success');
        result.screenshots.push(successScreenshot);

        // Uçuş bilgilerini çıkar
        try {
          const flightInfo = await this.extractFlightInfo();
          result.flightInfo = flightInfo;
        } catch (err) {
          console.log('Uçuş bilgileri çıkarılamadı:', err.message);
        }

        // SMS ile gönder (isteğe bağlı)
        if (phone) {
          try {
            await this.sendSMS(phone);
            result.message += ' SMS gönderildi.';
          } catch (err) {
            console.log('SMS gönderilemedi:', err.message);
          }
        }

      } else {
        // Hata durumu
        result.success = false;
        result.message = 'Pegasus check-in başarısız. PNR veya soyisim bilgileri hatalı olabilir.';
        
        const errorScreenshot = await this.takeScreenshot('pegasus_error');
        result.screenshots.push(errorScreenshot);
      }

    } catch (error) {
      console.error('Pegasus check-in hatası:', error);
      result.success = false;
      result.message = `Pegasus check-in sırasında hata: ${error.message}`;
      
      const errorScreenshot = await this.takeScreenshot('pegasus_exception');
      result.screenshots.push(errorScreenshot);
    }

    return result;
  }

  async selectAllPassengers() {
    try {
      console.log('🧑‍🤝‍🧑 Yolcu listesinde tüm yolcuları seçiliyor...');
      
      // Screenshot al - yolcu listesi sayfası
      await this.takeScreenshot('pegasus_passenger_list');
      
      // Tüm yolcuları seçmek için farklı stratejiler
      const passengerSelectionStrategies = [
        // Strateji 1: Katalon'dan yolcu seçim butonu
        async () => {
          try {
            // Katalon'dan: xpath=//div[@id='boarding-card-body']/div[2]/div[3]/div/div[2]/div[2]/div/button/div
            const passengerButton = await this.driver.wait(
              until.elementLocated(By.xpath("//div[@id='boarding-card-body']/div[2]/div[3]/div/div[2]/div[2]/div/button/div")), 
              10000
            );
            await passengerButton.click();
            console.log('✅ Katalon yolcu seçim butonu tıklandı');
            await this.driver.sleep(2000);
            return true;
          } catch (err) {
            console.log('❌ Katalon yolcu seçim butonu bulunamadı');
            return false;
          }
        },

        // Strateji 2: Katalon'dan checkbox'ları seç
        async () => {
          try {
            // Katalon'dan checkbox selector'lar - yeni kayıtlara göre güncellenmiş
            const katalonCheckboxSelectors = [
              "//input[@type='checkbox']", // Katalon'dan kesin çalışan
              "xpath=(.//*[normalize-space(text()) and normalize-space(.)='Check-in yapmak istediğiniz yolcuları seçin'])[1]/following::label[1]"
            ];
            
            for (const selector of katalonCheckboxSelectors) {
              try {
                let elements;
                if (selector.startsWith('xpath=')) {
                  elements = await this.driver.findElements(By.xpath(selector.replace('xpath=', '')));
                } else {
                  elements = await this.driver.findElements(By.xpath(selector));
                }
                
                if (elements.length > 0) {
                  console.log(`${elements.length} Katalon checkbox bulundu`);
                  for (let element of elements) {
                    try {
                      const isDisplayed = await element.isDisplayed();
                      const isEnabled = await element.isEnabled();
                      console.log(`Checkbox durumu - Görünür: ${isDisplayed}, Aktif: ${isEnabled}`);
                      
                      if (isDisplayed && isEnabled) {
                        await element.click();
                        await this.driver.sleep(500);
                        console.log('✅ Katalon yolcu seçildi');
                      }
                    } catch (err) {
                      console.log('Checkbox tıklama hatası:', err.message);
                    }
                  }
                  return true;
                }
              } catch (err) {
                console.log('Katalon checkbox seçimi hatası:', err.message);
              }
            }
            return false;
          } catch (err) {
            return false;
          }
        },

        // Strateji 3: Yolcu isimlerine tıkla
        async () => {
          const passengerSelectors = [
            "//div[contains(@class, 'passenger')]//input[@type='radio' or @type='checkbox']",
            "//tr[contains(@class, 'passenger')]//input[@type='radio' or @type='checkbox']",
            "//li[contains(@class, 'passenger')]//input[@type='radio' or @type='checkbox']"
          ];

          for (const selector of passengerSelectors) {
            try {
              const passengerInputs = await this.driver.findElements(By.xpath(selector));
              if (passengerInputs.length > 0) {
                console.log(`${passengerInputs.length} yolcu seçim alanı bulundu`);
                
                for (let input of passengerInputs) {
                  try {
                    await input.click();
                    await this.driver.sleep(300);
                  } catch (err) {
                    console.log('Yolcu seçim alanı tıklanamadı:', err.message);
                  }
                }
                console.log('✅ Tüm yolcular seçildi');
                return true;
              }
            } catch (err) {
              // Try next selector
            }
          }
          return false;
        }
      ];

      // Stratejileri sırayla dene
      for (let i = 0; i < passengerSelectionStrategies.length; i++) {
        console.log(`Strateji ${i + 1} deneniyor...`);
        const success = await passengerSelectionStrategies[i]();
        if (success) {
          await this.driver.sleep(2000);
          
          // Screenshot al - yolcular seçili
          await this.takeScreenshot('pegasus_passengers_selected');
          
          // Devam butonunu bul ve tıkla
          await this.clickContinueAfterPassengerSelection();
          return;
        }
      }

      console.log('⚠️ Yolcu seçimi yapılamadı, devam ediliyor...');
      
    } catch (error) {
      console.log('Yolcu seçimi hatası:', error.message);
    }
  }

  async clickContinueAfterPassengerSelection() {
    try {
      // Katalon'dan sonraki adımların selector'larını kullan
      console.log('📍 Yolcu seçimi sonrası devam butonu aranıyor...');
      
      const katalonContinueSelectors = [
        "//div[@id='boarding-card-body']/div[2]/div[2]/div/form/div/div[2]/button", // Katalon'dan kesin çalışan
        "//div[@id='boarding-card-body']/div[2]/div[2]/div/form/div/div[2]/button/div", // Katalon alternatif
        "//button[contains(text(), 'Devam')]",
        "//button[contains(text(), 'Continue')]",
        "//button[contains(text(), 'İlerle')]",
        "//input[@type='submit']",
        "//button[@type='submit']"
      ];

      for (const selector of katalonContinueSelectors) {
        try {
          console.log(`🔍 Deneniyor: ${selector}`);
          const continueBtn = await this.driver.wait(
            until.elementLocated(By.xpath(selector)),
            5000
          );
          
          const isDisplayed = await continueBtn.isDisplayed();
          const isEnabled = await continueBtn.isEnabled();
          console.log(`Buton durumu - Görünür: ${isDisplayed}, Aktif: ${isEnabled}`);
          
          if (isDisplayed && isEnabled) {
            await continueBtn.click();
            await this.driver.sleep(3000);
            console.log('✅ Katalon yolcu seçimi sonrası devam butonu tıklandı');
            
            // Katalon'dan sonraki adımlar
            await this.handleSSRPage();
            await this.handlePaymentPage();
            return;
          }
        } catch (err) {
          console.log('❌ Bulunamadı:', selector);
        }
      }

      console.log('⚠️ Yolcu seçimi sonrası devam butonu bulunamadı');

    } catch (error) {
      console.log('Yolcu seçimi sonrası devam butonu hatası:', error.message);
    }
  }

  async handleSSRPage() {
    try {
      const currentUrl = await this.driver.getCurrentUrl();
      console.log('📍 SSR sayfası kontrol ediliyor...', currentUrl);
      
      // SSR sayfasındaysa devam et
      if (currentUrl.includes('/check-in/ssr')) {
        console.log('✅ SSR sayfasına yönlendirildi');
        
        // Screenshot al
        await this.takeScreenshot('pegasus_ssr_page');
        
        // Katalon'dan SSR devam butonu
        const ssrContinueSelector = "//div[@id='boarding-card-body']/div[2]/div[2]/div/div[3]/div[4]/button";
        
        try {
          const ssrButton = await this.driver.wait(
            until.elementLocated(By.xpath(ssrContinueSelector)),
            10000
          );
          await ssrButton.click();
          console.log('✅ SSR sayfası devam butonu tıklandı');
          await this.driver.sleep(3000);
        } catch (err) {
          console.log('⚠️ SSR devam butonu bulunamadı, devam ediliyor...');
        }
      }
      
    } catch (error) {
      console.log('SSR sayfası işlemi hatası:', error.message);
    }
  }

  async handlePaymentPage() {
    try {
      const currentUrl = await this.driver.getCurrentUrl();
      console.log('📍 Payment sayfası kontrol ediliyor...', currentUrl);
      
      // Payment sayfasındaysa final butonuna tıkla
      if (currentUrl.includes('/check-in/payment')) {
        console.log('✅ Payment sayfasına yönlendirildi');
        
        // Screenshot al
        await this.takeScreenshot('pegasus_payment_page');
        
        // Katalon'dan final butonu
        const finalButtonSelector = "//div[@id='boarding-card-body']/div[2]/div[2]/div/div[4]/button";
        
        try {
          const finalButton = await this.driver.wait(
            until.elementLocated(By.xpath(finalButtonSelector)),
            10000
          );
          await finalButton.click();
          console.log('✅ Payment sayfası final butonu tıklandı');
          await this.driver.sleep(5000);
        } catch (err) {
          console.log('⚠️ Payment final butonu bulunamadı');
        }
      }
      
    } catch (error) {
      console.log('Payment sayfası işlemi hatası:', error.message);
    }
  }

  async fillInternationalDetails({ passportNumber, visaNumber }) {
    try {
      // Pasaport numarası alanını bul
      const passportSelectors = [
        "//input[@placeholder*='Pasaport' or @placeholder*='Passport']",
        "//input[@id*='passport']",
        "//input[@name*='passport']"
      ];

      for (const selector of passportSelectors) {
        try {
          const passportInput = await this.driver.wait(
            until.elementLocated(By.xpath(selector)),
            3000
          );
          await passportInput.clear();
          await passportInput.sendKeys(passportNumber);
          console.log('Pasaport numarası girildi');
          break;
        } catch (err) {
          // Try next selector
        }
      }

      // Vize numarası (varsa)
      if (visaNumber) {
        const visaSelectors = [
          "//input[@placeholder*='Vize' or @placeholder*='Visa']",
          "//input[@id*='visa']",
          "//input[@name*='visa']"
        ];

        for (const selector of visaSelectors) {
          try {
            const visaInput = await this.driver.wait(
              until.elementLocated(By.xpath(selector)),
              3000
            );
            await visaInput.clear();
            await visaInput.sendKeys(visaNumber);
            console.log('Vize numarası girildi');
            break;
          } catch (err) {
            // Try next selector
          }
        }
      }

    } catch (error) {
      console.log('Uluslararası bilgiler girilemedi:', error.message);
    }
  }

  async completeCheckin() {
    try {
      // Final check-in butonunu bul
      const finalCheckinSelectors = [
        "//button[contains(text(), 'Check-in Tamamla')]",
        "//button[contains(text(), 'Complete')]",
        "//button[contains(text(), 'Finish')]",
        "//button[contains(text(), 'Onayla')]",
        "//input[@type='submit']",
        "//button[@type='submit']"
      ];

      for (const selector of finalCheckinSelectors) {
        try {
          const finalButton = await this.driver.wait(
            until.elementLocated(By.xpath(selector)),
            3000
          );
          await finalButton.click();
          await this.driver.sleep(3000);
          console.log('Final check-in butonu tıklandı');
          break;
        } catch (err) {
          // Try next selector
        }
      }

    } catch (error) {
      console.log('Final check-in tamamlanamadı:', error.message);
    }
  }

  async extractFlightInfo() {
    try {
      await this.driver.sleep(2000);
      
      // Pegasus'tan uçuş bilgilerini çıkarmaya çalış
      const flightInfo = {
        flight: 'PC 1234', // Placeholder - gerçek veriler DOM'dan çekilecek
        from: 'İstanbul (SAW)',
        to: 'Ankara (ESB)', 
        date: new Date().toLocaleDateString('tr-TR'),
        seat: '15F'
      };
      
      return flightInfo;
    } catch (err) {
      throw new Error('Pegasus uçuş bilgileri çıkarılamadı');
    }
  }

  async sendSMS(phone) {
    try {
      // Pegasus SMS gönder butonunu bul ve tıkla
      const smsSelectors = [
        "//button[contains(text(), 'SMS')]",
        "//a[contains(text(), 'SMS')]",
        "//button[contains(text(), 'Telefona Gönder')]"
      ];

      for (const selector of smsSelectors) {
        try {
          const smsButton = await this.driver.wait(
            until.elementLocated(By.xpath(selector)),
            3000
          );
          await smsButton.click();
          await this.driver.sleep(2000);

          // Telefon numarasını gir
          const phoneInput = await this.driver.findElement(
            By.css("input[type='tel'], input[type='text']")
          );
          await phoneInput.clear();
          await phoneInput.sendKeys(phone);

          // Gönder
          const sendButton = await this.driver.findElement(
            By.xpath("//button[contains(text(), 'Gönder')]")
          );
          await sendButton.click();
          await this.driver.sleep(3000);

          console.log(`📱 Pegasus SMS sent to ${phone}`);
          break;

        } catch (err) {
          // Try next selector
        }
      }

    } catch (err) {
      throw new Error(`Pegasus SMS gönderilemedi: ${err.message}`);
    }
  }

  async takeScreenshot(name) {
    try {
      const screenshot = await this.driver.takeScreenshot();
      const filename = `pegasus_${name}_${Date.now()}.png`;
      const filepath = path.resolve('./downloads/screenshots', filename);
      
      await this.ensureDirectoryExists(path.dirname(filepath));
      await fs.writeFile(filepath, screenshot, 'base64');
      
      return {
        filename,
        path: filepath,
        url: `/downloads/screenshots/${filename}`
      };
    } catch (err) {
      console.error('Pegasus screenshot alınamadı:', err);
      return null;
    }
  }

  async close() {
    if (this.driver) {
      await this.driver.quit();
      this.driver = null;
      this.isRunning = false;
    }
  }

  // Cleanup için process exit handler
  setupCleanup() {
    process.on('SIGINT', async () => {
      console.log('🛑 Shutting down Pegasus Selenium service...');
      await this.close();
      process.exit(0);
    });
  }
}

module.exports = PegasusSeleniumService;