const { Builder, By, until } = require('selenium-webdriver');
const chrome = require('selenium-webdriver/chrome');
const fs = require('fs').promises;
const path = require('path');

class OptimizedSeleniumService {
  constructor() {
    this.driver = null;
    this.isRunning = false;
    this.defaultTimeout = 10000; // Reduced from implicit waits
    this.pageLoadTimeout = 15000; // Maximum page load time
  }

  async initDriver(headless = true) {
    const options = new chrome.Options();

    if (headless) {
      options.addArguments('--headless');
      options.addArguments('--no-sandbox');
      options.addArguments('--disable-dev-shm-usage');
      options.addArguments('--disable-gpu');
      options.addArguments('--window-size=1920,1080');
    }

    // Performance optimizations
    options.addArguments('--disable-web-security');
    options.addArguments('--allow-running-insecure-content');
    options.addArguments('--disable-features=TranslateUI');
    options.addArguments('--disable-iframes');
    options.addArguments('--disable-plugins');
    options.addArguments('--disable-extensions');
    options.addArguments('--disable-images'); // Faster loading
    options.addArguments('--disable-javascript-harmony-shipping');
    options.addArguments('--disable-background-timer-throttling');
    options.addArguments('--disable-renderer-backgrounding');
    options.addArguments('--disable-backgrounding-occluded-windows');

    // Memory optimizations
    options.addArguments('--memory-pressure-off');
    options.addArguments('--max_old_space_size=512');

    const downloadDir = path.resolve('./downloads/boarding-passes');
    await this.ensureDirectoryExists(downloadDir);

    options.setUserPreferences({
      'download.default_directory': downloadDir,
      'download.prompt_for_download': false,
      'plugins.always_open_pdf_externally': true,
      // Disable resource-heavy features
      'profile.default_content_setting_values.notifications': 2,
      'profile.default_content_settings.popups': 0,
      'profile.managed_default_content_settings.images': 2 // Block images
    });

    this.driver = await new Builder()
      .forBrowser('chrome')
      .setChromeOptions(options)
      .build();

    // Set optimized timeouts
    await this.driver.manage().setTimeouts({
      implicit: this.defaultTimeout,
      pageLoad: this.pageLoadTimeout,
      script: this.defaultTimeout
    });

    this.isRunning = true;
    return this.driver;
  }

  async ensureDirectoryExists(dir) {
    try {
      await fs.mkdir(dir, { recursive: true });
    } catch (err) {
      if (err.code !== 'EEXIST') throw err;
    }
  }

  async performCheckin({ pnr, lastName, firstName, phone }) {
    if (!this.driver) {
      await this.initDriver();
    }

    const result = {
      success: false,
      message: '',
      flightInfo: null,
      boardingPassUrl: null,
      screenshots: []
    };

    const startTime = Date.now();

    try {
      console.log(`🚀 Optimized check-in started for PNR: ${pnr}`);

      // 1. Navigate with timeout protection
      await Promise.race([
        this.driver.get('https://www.turkishairlines.com'),
        this.timeoutPromise(8000, 'Page load timeout')
      ]);

      // Minimal delay instead of fixed sleep
      await this.smartWait(1000);

      // 2. Handle cookie popup with timeout
      try {
        const cookieButton = await this.driver.wait(
          until.elementLocated(By.xpath("//button[contains(text(), 'Tüm çerezleri')]")),
          3000
        );
        await cookieButton.click();
        await this.smartWait(500);
      } catch (err) {
        // Continue if cookie popup not found
        console.log('Cookie popup not found - continuing');
      }

      // 3. Click check-in tab with optimized wait
      const checkinTab = await this.driver.wait(
        until.elementLocated(By.xpath("//button[contains(@class, 'tab') and contains(., 'Check-in')]")),
        8000
      );
      await checkinTab.click();
      await this.smartWait(1500);

      // 4. Fill form fields with faster input
      const pnrInput = await this.driver.wait(
        until.elementLocated(By.id('pnrnumberinput')),
        8000
      );
      await this.fastInput(pnrInput, pnr);

      const surnameInput = await this.driver.findElement(By.id('pnrsurnameinput'));
      await this.fastInput(surnameInput, lastName);

      // Optional screenshot for debugging
      if (process.env.NODE_ENV === 'development') {
        const formScreenshot = await this.takeScreenshot('form_filled');
        result.screenshots.push(formScreenshot);
      }

      // 5. Submit with timeout
      const checkinButton = await this.driver.findElement(
        By.xpath("//button[contains(text(), 'Check-in')]")
      );
      await checkinButton.click();

      // Wait for result with timeout
      await Promise.race([
        this.waitForResult(),
        this.timeoutPromise(12000, 'Check-in response timeout')
      ]);

      // 6. Evaluate result
      const currentUrl = await this.driver.getCurrentUrl();
      console.log(`Result URL: ${currentUrl}`);

      if (currentUrl.includes('binis-karti') || currentUrl.includes('boarding')) {
        result.success = true;
        result.message = 'Check-in successful! Boarding pass ready.';

        // Extract flight info with timeout
        try {
          result.flightInfo = await Promise.race([
            this.extractFlightInfo(),
            this.timeoutPromise(5000, 'Flight info extraction timeout')
          ]);
        } catch (err) {
          console.log('Flight info extraction failed:', err.message);
        }

        // Optional SMS with timeout
        if (phone) {
          try {
            await Promise.race([
              this.sendSMS(phone),
              this.timeoutPromise(8000, 'SMS send timeout')
            ]);
            result.message += ' SMS sent.';
          } catch (err) {
            console.log('SMS failed:', err.message);
          }
        }

      } else {
        result.success = false;
        result.message = 'Check-in failed. PNR or surname may be incorrect.';
      }

      const executionTime = Date.now() - startTime;
      console.log(`⏱️ Check-in completed in ${executionTime}ms`);

    } catch (error) {
      const executionTime = Date.now() - startTime;
      console.error(`❌ Check-in error for ${pnr} after ${executionTime}ms:`, error.message);

      result.success = false;
      result.message = `Check-in error: ${error.message}`;
    }

    return result;
  }

  // Optimized input method
  async fastInput(element, text) {
    await element.clear();
    // Use JavaScript for faster input
    await this.driver.executeScript(
      "arguments[0].value = arguments[1]; arguments[0].dispatchEvent(new Event('input', { bubbles: true }));",
      element,
      text
    );
  }

  // Smart wait - reduces unnecessary delays
  async smartWait(ms) {
    const delay = Math.min(ms, 2000); // Cap at 2 seconds
    await new Promise(resolve => setTimeout(resolve, delay));
  }

  // Timeout promise helper
  timeoutPromise(ms, message) {
    return new Promise((_, reject) =>
      setTimeout(() => reject(new Error(message)), ms)
    );
  }

  // Wait for check-in result with smart polling
  async waitForResult() {
    const maxAttempts = 10;
    const pollInterval = 800;

    for (let i = 0; i < maxAttempts; i++) {
      const currentUrl = await this.driver.getCurrentUrl();

      // Check for success or error indicators
      if (currentUrl.includes('binis-karti') ||
          currentUrl.includes('boarding') ||
          currentUrl.includes('error') ||
          currentUrl.includes('hata')) {
        return;
      }

      await this.smartWait(pollInterval);
    }

    throw new Error('Check-in result determination timeout');
  }

  async extractFlightInfo() {
    await this.smartWait(1000);

    try {
      // Use JavaScript to extract data faster
      const flightData = await this.driver.executeScript(`
        try {
          return {
            flight: document.querySelector('[data-flight-number]')?.textContent || 'N/A',
            from: document.querySelector('[data-departure-city]')?.textContent || 'N/A',
            to: document.querySelector('[data-arrival-city]')?.textContent || 'N/A',
            seat: document.querySelector('[data-seat-number]')?.textContent || 'TBD'
          };
        } catch (e) {
          return {
            flight: 'TK XXX',
            from: 'Istanbul',
            to: 'Destination',
            seat: 'Auto-assigned'
          };
        }
      `);

      return {
        ...flightData,
        date: new Date().toLocaleDateString('tr-TR')
      };

    } catch (err) {
      // Fallback data
      return {
        flight: 'TK XXX',
        from: 'Istanbul',
        to: 'Destination',
        date: new Date().toLocaleDateString('tr-TR'),
        seat: 'Auto-assigned'
      };
    }
  }

  async sendSMS(phone) {
    try {
      const smsButton = await this.driver.wait(
        until.elementLocated(By.xpath("//button[contains(text(), 'SMS ile gönder')]")),
        5000
      );
      await smsButton.click();
      await this.smartWait(1000);

      const phoneInput = await this.driver.findElement(
        By.css("input[type='tel'], input[type='text']")
      );
      await this.fastInput(phoneInput, phone);

      const sendButton = await this.driver.findElement(
        By.xpath("//button[contains(text(), 'Gönder')]")
      );
      await sendButton.click();
      await this.smartWait(2000);

      console.log(`📱 SMS sent to ${phone}`);
    } catch (err) {
      throw new Error(`SMS send failed: ${err.message}`);
    }
  }

  async takeScreenshot(name) {
    if (process.env.NODE_ENV === 'production') {
      return null; // Skip screenshots in production
    }

    try {
      const screenshot = await this.driver.takeScreenshot();
      const filename = `${name}_${Date.now()}.png`;
      const filepath = path.resolve('./downloads/screenshots', filename);

      await this.ensureDirectoryExists(path.dirname(filepath));
      await fs.writeFile(filepath, screenshot, 'base64');

      return {
        filename,
        path: filepath,
        url: `/downloads/screenshots/${filename}`
      };
    } catch (err) {
      console.error('Screenshot failed:', err);
      return null;
    }
  }

  async close() {
    if (this.driver) {
      try {
        await this.driver.quit();
      } catch (err) {
        console.error('Driver close error:', err.message);
      } finally {
        this.driver = null;
        this.isRunning = false;
      }
    }
  }

  // Health check for the service
  isHealthy() {
    return this.isRunning && this.driver !== null;
  }
}

module.exports = OptimizedSeleniumService;