const AmadeusConfig = require('../config/amadeus');

/**
 * Amadeus API Service
 * Provides flight search, booking, and status functionality
 */
class AmadeusService {
  constructor() {
    this.amadeus = new AmadeusConfig();
  }

  /**
   * Search flight offers
   */
  async searchFlights(searchParams) {
    const {
      originLocationCode,
      destinationLocationCode,
      departureDate,
      returnDate,
      adults = 1,
      children = 0,
      infants = 0,
      travelClass = 'ECONOMY',
      nonStop = false,
      currencyCode = 'TRY',
      max = 250
    } = searchParams;

    const params = {
      originLocationCode,
      destinationLocationCode,
      departureDate,
      adults,
      max,
      currencyCode,
      nonStop
    };

    if (returnDate) {
      params.returnDate = returnDate;
    }

    if (children > 0) {
      params.children = children;
    }

    if (infants > 0) {
      params.infants = infants;
    }

    if (travelClass !== 'ECONOMY') {
      params.travelClass = travelClass;
    }

    try {
      const response = await this.amadeus.makeRequest(
        this.amadeus.endpoints.flightOffers,
        'GET',
        null,
        params
      );

      return {
        success: true,
        data: response.data || [],
        meta: response.meta || {},
        count: response.data?.length || 0
      };

    } catch (error) {
      console.error('Flight search error:', error);
      return {
        success: false,
        error: 'Flight search failed',
        message: error.response?.data?.errors?.[0]?.detail || error.message
      };
    }
  }

  /**
   * Get flight offer pricing
   */
  async getFlightPricing(flightOffers) {
    try {
      const requestBody = {
        data: {
          type: 'flight-offers-pricing',
          flightOffers: Array.isArray(flightOffers) ? flightOffers : [flightOffers]
        }
      };

      const response = await this.amadeus.makeRequest(
        this.amadeus.endpoints.flightOffersPrice,
        'POST',
        requestBody
      );

      return {
        success: true,
        data: response.data,
        included: response.included || []
      };

    } catch (error) {
      console.error('Flight pricing error:', error);
      return {
        success: false,
        error: 'Flight pricing failed',
        message: error.response?.data?.errors?.[0]?.detail || error.message
      };
    }
  }

  /**
   * Create flight booking
   */
  async createBooking(bookingData) {
    const {
      flightOffers,
      travelers,
      contacts,
      remarks = {}
    } = bookingData;

    try {
      const requestBody = {
        data: {
          type: 'flight-order',
          flightOffers: Array.isArray(flightOffers) ? flightOffers : [flightOffers],
          travelers,
          contacts,
          remarks
        }
      };

      const response = await this.amadeus.makeRequest(
        this.amadeus.endpoints.flightCreate,
        'POST',
        requestBody
      );

      return {
        success: true,
        booking: response.data,
        pnr: response.data?.associatedRecords?.[0]?.reference,
        ticketingAgreement: response.data?.ticketingAgreement
      };

    } catch (error) {
      console.error('Flight booking error:', error);
      return {
        success: false,
        error: 'Flight booking failed',
        message: error.response?.data?.errors?.[0]?.detail || error.message,
        errors: error.response?.data?.errors || []
      };
    }
  }

  /**
   * Get flight status
   */
  async getFlightStatus(carrierCode, flightNumber, scheduledDepartureDate) {
    const params = {
      carrierCode,
      flightNumber,
      scheduledDepartureDate
    };

    try {
      const response = await this.amadeus.makeRequest(
        this.amadeus.endpoints.flightStatus,
        'GET',
        null,
        params
      );

      if (response.data && response.data.length > 0) {
        const flight = response.data[0];
        return {
          success: true,
          flight: {
            carrierCode: flight.carrierCode,
            flightNumber: flight.flightNumber,
            scheduledDepartureDate: flight.scheduledDepartureDate,
            flightStatus: flight.flightStatus,
            departure: flight.flightPoints?.[0],
            arrival: flight.flightPoints?.[1],
            aircraft: flight.aircraft,
            operating: flight.operating
          }
        };
      }

      return {
        success: false,
        error: 'Flight not found',
        message: 'No flight found with the provided details'
      };

    } catch (error) {
      console.error('Flight status error:', error);
      return {
        success: false,
        error: 'Flight status lookup failed',
        message: error.response?.data?.errors?.[0]?.detail || error.message
      };
    }
  }

  /**
   * Get airport information
   */
  async getAirportInfo(keyword, subType = 'AIRPORT,CITY') {
    // Türkçe karakterleri İngilizce'ye çevir (Amadeus API Türkçe karakterleri desteklemiyor)
    const normalizeKeyword = (str) => {
      return str
        .replace(/İ/g, 'I')
        .replace(/ı/g, 'i')
        .replace(/Ğ/g, 'G')
        .replace(/ğ/g, 'g')
        .replace(/Ü/g, 'U')
        .replace(/ü/g, 'u')
        .replace(/Ş/g, 'S')
        .replace(/ş/g, 's')
        .replace(/Ö/g, 'O')
        .replace(/ö/g, 'o')
        .replace(/Ç/g, 'C')
        .replace(/ç/g, 'c');
    };

    const normalizedKeyword = normalizeKeyword(keyword);

    const params = {
      keyword: normalizedKeyword,
      subType,
      'page[limit]': 10
    };

    try {
      const response = await this.amadeus.makeRequest(
        this.amadeus.endpoints.airportLocation,
        'GET',
        null,
        params
      );

      return {
        success: true,
        airports: response.data || [],
        count: response.data?.length || 0
      };

    } catch (error) {
      console.error('Airport search error:', error);
      return {
        success: false,
        error: 'Airport search failed',
        message: error.response?.data?.errors?.[0]?.detail || error.message
      };
    }
  }

  /**
   * Get airline information
   */
  async getAirlineInfo(airlineCodes) {
    const params = {
      airlineCodes: Array.isArray(airlineCodes) ? airlineCodes.join(',') : airlineCodes
    };

    try {
      const response = await this.amadeus.makeRequest(
        this.amadeus.endpoints.airlines,
        'GET',
        null,
        params
      );

      return {
        success: true,
        airlines: response.data || [],
        count: response.data?.length || 0
      };

    } catch (error) {
      console.error('Airline info error:', error);
      return {
        success: false,
        error: 'Airline info lookup failed',
        message: error.response?.data?.errors?.[0]?.detail || error.message
      };
    }
  }

  /**
   * Get check-in links
   */
  async getCheckinLinks(airlineCode, language = 'en-GB') {
    const params = {
      airlineCode,
      language
    };

    try {
      const response = await this.amadeus.makeRequest(
        this.amadeus.endpoints.checkinLinks,
        'GET',
        null,
        params
      );

      return {
        success: true,
        checkinLinks: response.data || [],
        count: response.data?.length || 0
      };

    } catch (error) {
      console.error('Checkin links error:', error);
      return {
        success: false,
        error: 'Checkin links lookup failed',
        message: error.response?.data?.errors?.[0]?.detail || error.message
      };
    }
  }

  /**
   * Test service connection
   */
  async testConnection() {
    return await this.amadeus.testConnection();
  }

  /**
   * Format flight offer for frontend display
   */
  formatFlightOffer(offer) {
    const {
      id,
      source,
      instantTicketingRequired,
      nonHomogeneous,
      oneWay,
      lastTicketingDate,
      numberOfBookableSeats,
      itineraries,
      price,
      pricingOptions,
      validatingAirlineCodes,
      travelerPricings
    } = offer;

    // Extract first itinerary details
    const outbound = itineraries[0];
    const return_ = itineraries[1] || null;

    return {
      id,
      source,
      oneWay,
      price: {
        currency: price.currency,
        total: price.total,
        base: price.base,
        fees: price.fees || [],
        taxes: price.taxes || []
      },
      outbound: {
        duration: outbound.duration,
        segments: outbound.segments.map(segment => ({
          departure: {
            iataCode: segment.departure.iataCode,
            terminal: segment.departure.terminal,
            at: segment.departure.at
          },
          arrival: {
            iataCode: segment.arrival.iataCode,
            terminal: segment.arrival.terminal,
            at: segment.arrival.at
          },
          carrierCode: segment.carrierCode,
          number: segment.number,
          aircraft: segment.aircraft,
          operating: segment.operating,
          duration: segment.duration,
          stops: segment.numberOfStops
        }))
      },
      return: return_ ? {
        duration: return_.duration,
        segments: return_.segments.map(segment => ({
          departure: {
            iataCode: segment.departure.iataCode,
            terminal: segment.departure.terminal,
            at: segment.departure.at
          },
          arrival: {
            iataCode: segment.arrival.iataCode,
            terminal: segment.arrival.terminal,
            at: segment.arrival.at
          },
          carrierCode: segment.carrierCode,
          number: segment.number,
          aircraft: segment.aircraft,
          operating: segment.operating,
          duration: segment.duration,
          stops: segment.numberOfStops
        }))
      } : null,
      bookableSeats: numberOfBookableSeats,
      lastTicketingDate,
      validatingAirlines: validatingAirlineCodes,
      travelerPricings: travelerPricings.map(tp => ({
        travelerId: tp.travelerId,
        fareOption: tp.fareOption,
        travelerType: tp.travelerType,
        price: tp.price,
        fareDetailsBySegment: tp.fareDetailsBySegment
      }))
    };
  }
}

module.exports = AmadeusService;