const mysql = require('mysql2/promise');

async function createAdminAndMigrateRoles() {
  const connection = await mysql.createConnection({
    host: 'localhost',
    user: 'root',
    password: '',
    database: 'vtravel_local'
  });

  try {
    console.log('🔄 Starting admin creation and role migration...\n');

    // 1. Check if Vamos admin already exists
    const [existing] = await connection.execute(
      'SELECT id, email FROM agency_users WHERE email = ?',
      ['yetkili@vamos.ist']
    );

    let adminId;
    if (existing.length > 0) {
      console.log('⚠️  Vamos admin account already exists, updating...');
      adminId = existing[0].id;
      
      await connection.execute(`
        UPDATE agency_users 
        SET 
          first_name = 'Vamos',
          last_name = 'Dijital Ofis',
          phone = '05364376679',
          role = 'system_admin',
          is_active = 1,
          permissions = JSON_ARRAY('*')
        WHERE id = ?
      `, [adminId]);
      
      console.log('✅ Vamos admin account updated\n');
    } else {
      console.log('📝 Creating Vamos admin account...');
      
      // Get Yönetim department ID
      const [depts] = await connection.execute(
        'SELECT id FROM departments WHERE name LIKE "%Yönetim%" OR name LIKE "%Management%" LIMIT 1'
      );
      
      let deptId = null;
      if (depts.length > 0) {
        deptId = depts[0].id;
      } else {
        // Create Yönetim department if doesn't exist
        const [result] = await connection.execute(`
          INSERT INTO departments (id, name, description, color, is_active, permissions)
          VALUES (UUID(), 'Yönetim', 'Sistem Yönetimi ve Operasyon', 'bg-red-500', 1, JSON_ARRAY('*'))
        `);
        
        const [newDept] = await connection.execute(
          'SELECT id FROM departments ORDER BY created_at DESC LIMIT 1'
        );
        deptId = newDept[0].id;
        console.log('✅ Yönetim department created');
      }

      // Create admin user
      await connection.execute(`
        INSERT INTO agency_users (
          id, first_name, last_name, email, phone, 
          role, department_id, is_active, permissions
        ) VALUES (
          UUID(), 'Vamos', 'Dijital Ofis', 'yetkili@vamos.ist', '05364376679',
          'system_admin', ?, 1, JSON_ARRAY('*')
        )
      `, [deptId]);
      
      console.log('✅ Vamos admin account created\n');
    }

    // 2. Migrate existing roles
    console.log('🔄 Migrating existing user roles...\n');

    // Update super_admin -> system_admin
    const [superAdminResult] = await connection.execute(`
      UPDATE agency_users 
      SET role = 'system_admin' 
      WHERE role = 'super_admin'
    `);
    if (superAdminResult.affectedRows > 0) {
      console.log(`✅ Updated ${superAdminResult.affectedRows} super_admin(s) to system_admin`);
    }

    // Update admin -> agency_manager
    const [adminResult] = await connection.execute(`
      UPDATE agency_users 
      SET role = 'agency_manager' 
      WHERE role = 'admin'
    `);
    if (adminResult.affectedRows > 0) {
      console.log(`✅ Updated ${adminResult.affectedRows} admin(s) to agency_manager`);
    }

    // Update manager -> department_manager
    const [managerResult] = await connection.execute(`
      UPDATE agency_users 
      SET role = 'department_manager' 
      WHERE role = 'manager'
    `);
    if (managerResult.affectedRows > 0) {
      console.log(`✅ Updated ${managerResult.affectedRows} manager(s) to department_manager`);
    }

    // Update senior_agent -> senior_specialist
    const [seniorResult] = await connection.execute(`
      UPDATE agency_users 
      SET role = 'senior_specialist' 
      WHERE role = 'senior_agent'
    `);
    if (seniorResult.affectedRows > 0) {
      console.log(`✅ Updated ${seniorResult.affectedRows} senior_agent(s) to senior_specialist`);
    }

    // Update agent -> specialist
    const [agentResult] = await connection.execute(`
      UPDATE agency_users 
      SET role = 'specialist' 
      WHERE role = 'agent'
    `);
    if (agentResult.affectedRows > 0) {
      console.log(`✅ Updated ${agentResult.affectedRows} agent(s) to specialist`);
    }

    console.log('\n✅ Role migration completed!\n');

    // 3. Verify results
    console.log('📊 Current user roles:\n');
    const [users] = await connection.execute(`
      SELECT 
        CONCAT(first_name, ' ', last_name) as name,
        email,
        role,
        is_active
      FROM agency_users
      ORDER BY 
        CASE role
          WHEN 'system_admin' THEN 1
          WHEN 'agency_manager' THEN 2
          WHEN 'department_manager' THEN 3
          WHEN 'senior_specialist' THEN 4
          WHEN 'specialist' THEN 5
          WHEN 'assistant' THEN 6
          WHEN 'intern' THEN 7
          ELSE 99
        END
    `);

    users.forEach(user => {
      const status = user.is_active ? '🟢' : '🔴';
      console.log(`${status} ${user.name.padEnd(25)} | ${user.email.padEnd(30)} | ${user.role}`);
    });

    console.log('\n✅ All operations completed successfully!');

  } catch (error) {
    console.error('❌ Error:', error.message);
    throw error;
  } finally {
    await connection.end();
  }
}

createAdminAndMigrateRoles();
