const express = require('express');
const router = express.Router();
const db = require('../services/database');
const { v4: uuidv4 } = require('uuid');

// ============================================
// GET /api/visa-requests
// Get all visa requests with filters
// ============================================
router.get('/', async (req, res) => {
  try {
    const { status, priority, country, search } = req.query;

    let query = `
      SELECT vr.*,
             c.first_name, c.last_name, c.company_name, c.email as customer_email, c.phone as customer_phone,
             au.first_name as assigned_first_name, au.last_name as assigned_last_name
      FROM visa_requests vr
      LEFT JOIN customers c ON vr.customer_id = c.id
      LEFT JOIN agency_users au ON vr.assigned_to = au.id
      WHERE 1=1
    `;

    const params = [];

    if (status) {
      query += ' AND vr.status = ?';
      params.push(status);
    }

    if (priority) {
      query += ' AND vr.priority = ?';
      params.push(priority);
    }

    if (country) {
      query += ' AND vr.destination_country LIKE ?';
      params.push(`%${country}%`);
    }

    if (search) {
      query += ` AND (
        vr.request_number LIKE ? OR
        vr.customer_name LIKE ? OR
        vr.applicant_first_name LIKE ? OR
        vr.applicant_last_name LIKE ? OR
        vr.applicant_passport_number LIKE ?
      )`;
      const searchPattern = `%${search}%`;
      params.push(searchPattern, searchPattern, searchPattern, searchPattern, searchPattern);
    }

    query += ' ORDER BY vr.created_at DESC';

    const [requests] = await db.query(query, params);

    // Parse JSON fields
    const formattedRequests = requests.map(req => ({
      ...req,
      documents_required: req.documents_required ? JSON.parse(req.documents_required) : [],
      documents_received: req.documents_received ? JSON.parse(req.documents_received) : []
    }));

    res.json({
      success: true,
      data: formattedRequests,
      count: formattedRequests.length
    });
  } catch (error) {
    console.error('Error fetching visa requests:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch visa requests',
      message: error.message
    });
  }
});

// ============================================
// GET /api/visa-requests/:id
// Get single visa request by ID
// ============================================
router.get('/:id', async (req, res) => {
  try {
    const { id } = req.params;

    const [requests] = await db.query(`
      SELECT vr.*,
             c.first_name, c.last_name, c.company_name, c.email as customer_email, c.phone as customer_phone, c.type as customer_type,
             au.first_name as assigned_first_name, au.last_name as assigned_last_name, au.email as assigned_email
      FROM visa_requests vr
      LEFT JOIN customers c ON vr.customer_id = c.id
      LEFT JOIN agency_users au ON vr.assigned_to = au.id
      WHERE vr.id = ?
    `, [id]);

    if (requests.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'Visa request not found'
      });
    }

    const request = requests[0];

    // Parse JSON fields
    request.documents_required = request.documents_required ? JSON.parse(request.documents_required) : [];
    request.documents_received = request.documents_received ? JSON.parse(request.documents_received) : [];

    res.json({
      success: true,
      data: request
    });
  } catch (error) {
    console.error('Error fetching visa request:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch visa request',
      message: error.message
    });
  }
});

// ============================================
// POST /api/visa-requests
// Create new visa request
// ============================================
router.post('/', async (req, res) => {
  try {
    const {
      customer_id,
      customer_type,
      customer_name,
      status = 'pending',
      priority = 'normal',
      destination_country,
      visa_type,
      purpose_of_visit,
      planned_entry_date,
      planned_exit_date,
      applicant_first_name,
      applicant_last_name,
      applicant_passport_number,
      applicant_passport_expiry,
      applicant_nationality,
      applicant_date_of_birth,
      applicant_phone,
      applicant_email,
      documents_required,
      service_fee,
      visa_fee,
      currency = 'TRY',
      assigned_to,
      internal_notes,
      customer_notes
    } = req.body;

    // Validation
    if (!customer_name || !destination_country || !visa_type) {
      return res.status(400).json({
        success: false,
        error: 'Customer name, destination country, and visa type are required'
      });
    }

    if (!applicant_first_name || !applicant_last_name || !applicant_passport_number || !applicant_passport_expiry || !applicant_nationality || !applicant_date_of_birth) {
      return res.status(400).json({
        success: false,
        error: 'All applicant information is required'
      });
    }

    const id = uuidv4();

    // Generate request number
    const [[{ count }]] = await db.query('SELECT COUNT(*) as count FROM visa_requests');
    const request_number = `VISA-${new Date().getFullYear()}-${String(count + 1).padStart(4, '0')}`;

    // Calculate duration if dates provided
    let duration_of_stay = null;
    if (planned_entry_date && planned_exit_date) {
      const entry = new Date(planned_entry_date);
      const exit = new Date(planned_exit_date);
      duration_of_stay = Math.ceil((exit - entry) / (1000 * 60 * 60 * 24));
    }

    // Calculate total amount
    const total_amount = (parseFloat(service_fee) || 0) + (parseFloat(visa_fee) || 0);

    await db.query(`
      INSERT INTO visa_requests (
        id, customer_id, customer_type, customer_name, request_number, status, priority,
        destination_country, visa_type, purpose_of_visit, planned_entry_date, planned_exit_date, duration_of_stay,
        applicant_first_name, applicant_last_name, applicant_passport_number, applicant_passport_expiry,
        applicant_nationality, applicant_date_of_birth, applicant_phone, applicant_email,
        documents_required, service_fee, visa_fee, total_amount, currency,
        assigned_to, internal_notes, customer_notes
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `, [
      id, customer_id, customer_type, customer_name, request_number, status, priority,
      destination_country, visa_type, purpose_of_visit, planned_entry_date, planned_exit_date, duration_of_stay,
      applicant_first_name, applicant_last_name, applicant_passport_number, applicant_passport_expiry,
      applicant_nationality, applicant_date_of_birth, applicant_phone, applicant_email,
      JSON.stringify(documents_required || []), service_fee, visa_fee, total_amount, currency,
      assigned_to, internal_notes, customer_notes
    ]);

    const [newRequest] = await db.query('SELECT * FROM visa_requests WHERE id = ?', [id]);

    res.status(201).json({
      success: true,
      message: 'Visa request created successfully',
      data: newRequest[0]
    });
  } catch (error) {
    console.error('Error creating visa request:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to create visa request',
      message: error.message
    });
  }
});

// ============================================
// PUT /api/visa-requests/:id
// Update visa request
// ============================================
router.put('/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const updates = req.body;

    // Check if request exists
    const [existing] = await db.query('SELECT id FROM visa_requests WHERE id = ?', [id]);
    if (existing.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'Visa request not found'
      });
    }

    // Build update query dynamically
    const allowedFields = [
      'status', 'priority', 'destination_country', 'visa_type', 'purpose_of_visit',
      'planned_entry_date', 'planned_exit_date', 'duration_of_stay',
      'applicant_first_name', 'applicant_last_name', 'applicant_passport_number',
      'applicant_passport_expiry', 'applicant_nationality', 'applicant_date_of_birth',
      'applicant_phone', 'applicant_email', 'application_number', 'appointment_date',
      'appointment_location', 'submitted_date', 'decision_date', 'visa_validity_start',
      'visa_validity_end', 'number_of_entries', 'documents_required', 'documents_received',
      'documents_complete', 'service_fee', 'visa_fee', 'total_amount', 'payment_status',
      'assigned_to', 'processed_by', 'approved_by', 'internal_notes', 'customer_notes',
      'rejection_reason'
    ];

    const updateFields = [];
    const values = [];

    Object.keys(updates).forEach(key => {
      if (allowedFields.includes(key) && updates[key] !== undefined) {
        updateFields.push(`${key} = ?`);
        // Stringify JSON fields
        if (key === 'documents_required' || key === 'documents_received') {
          values.push(JSON.stringify(updates[key]));
        } else {
          values.push(updates[key]);
        }
      }
    });

    if (updateFields.length === 0) {
      return res.status(400).json({
        success: false,
        error: 'No valid fields to update'
      });
    }

    values.push(id);

    await db.query(
      `UPDATE visa_requests SET ${updateFields.join(', ')} WHERE id = ?`,
      values
    );

    const [updated] = await db.query('SELECT * FROM visa_requests WHERE id = ?', [id]);

    res.json({
      success: true,
      message: 'Visa request updated successfully',
      data: updated[0]
    });
  } catch (error) {
    console.error('Error updating visa request:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to update visa request',
      message: error.message
    });
  }
});

// ============================================
// DELETE /api/visa-requests/:id
// Delete visa request
// ============================================
router.delete('/:id', async (req, res) => {
  try {
    const { id } = req.params;

    const [result] = await db.query('DELETE FROM visa_requests WHERE id = ?', [id]);

    if (result.affectedRows === 0) {
      return res.status(404).json({
        success: false,
        error: 'Visa request not found'
      });
    }

    res.json({
      success: true,
      message: 'Visa request deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting visa request:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to delete visa request',
      message: error.message
    });
  }
});

// ============================================
// GET /api/visa-requests/stats/overview
// Get visa request statistics
// ============================================
router.get('/stats/overview', async (req, res) => {
  try {
    const [[stats]] = await db.query(`
      SELECT
        COUNT(*) as total,
        SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
        SUM(CASE WHEN status = 'in_process' THEN 1 ELSE 0 END) as in_process,
        SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
        SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected,
        SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled,
        SUM(total_amount) as total_revenue,
        AVG(total_amount) as average_amount
      FROM visa_requests
    `);

    res.json({
      success: true,
      data: stats
    });
  } catch (error) {
    console.error('Error fetching visa stats:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch visa statistics',
      message: error.message
    });
  }
});

module.exports = router;
