const express = require('express');
const router = express.Router();
const db = require('../services/database');
const { v4: uuidv4 } = require('uuid');

// ============================================
// GET /api/transfer-requests
// Get all transfer requests with filters
// ============================================
router.get('/', async (req, res) => {
  try {
    const { status, transfer_type, date, search } = req.query;

    let query = `
      SELECT tr.*,
             c.first_name, c.last_name, c.company_name, c.email as customer_email, c.phone as customer_phone,
             au.first_name as assigned_first_name, au.last_name as assigned_last_name
      FROM transfer_requests tr
      LEFT JOIN customers c ON tr.customer_id = c.id
      LEFT JOIN agency_users au ON tr.assigned_to = au.id
      WHERE 1=1
    `;

    const params = [];

    if (status) {
      query += ' AND tr.status = ?';
      params.push(status);
    }

    if (transfer_type) {
      query += ' AND tr.transfer_type = ?';
      params.push(transfer_type);
    }

    if (date) {
      query += ' AND DATE(tr.pickup_date) = DATE(?)';
      params.push(date);
    }

    if (search) {
      query += ` AND (
        tr.request_number LIKE ? OR
        tr.customer_name LIKE ? OR
        tr.passenger_name LIKE ? OR
        tr.flight_number LIKE ? OR
        tr.driver_name LIKE ?
      )`;
      const searchPattern = `%${search}%`;
      params.push(searchPattern, searchPattern, searchPattern, searchPattern, searchPattern);
    }

    query += ' ORDER BY tr.pickup_date DESC';

    const [requests] = await db.query(query, params);

    // Parse JSON fields
    const formattedRequests = requests.map(req => ({
      ...req,
      special_equipment: req.special_equipment ? JSON.parse(req.special_equipment) : []
    }));

    res.json({
      success: true,
      data: formattedRequests,
      count: formattedRequests.length
    });
  } catch (error) {
    console.error('Error fetching transfer requests:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch transfer requests',
      message: error.message
    });
  }
});

// ============================================
// GET /api/transfer-requests/:id
// Get single transfer request by ID
// ============================================
router.get('/:id', async (req, res) => {
  try {
    const { id } = req.params;

    const [requests] = await db.query(`
      SELECT tr.*,
             c.first_name, c.last_name, c.company_name, c.email as customer_email, c.phone as customer_phone, c.type as customer_type,
             au.first_name as assigned_first_name, au.last_name as assigned_last_name, au.email as assigned_email
      FROM transfer_requests tr
      LEFT JOIN customers c ON tr.customer_id = c.id
      LEFT JOIN agency_users au ON tr.assigned_to = au.id
      WHERE tr.id = ?
    `, [id]);

    if (requests.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'Transfer request not found'
      });
    }

    const request = requests[0];

    // Parse JSON fields
    request.special_equipment = request.special_equipment ? JSON.parse(request.special_equipment) : [];

    res.json({
      success: true,
      data: request
    });
  } catch (error) {
    console.error('Error fetching transfer request:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch transfer request',
      message: error.message
    });
  }
});

// ============================================
// POST /api/transfer-requests
// Create new transfer request
// ============================================
router.post('/', async (req, res) => {
  try {
    const {
      customer_id,
      customer_type,
      customer_name,
      status = 'pending',
      priority = 'normal',
      transfer_type,
      pickup_date,
      pickup_location,
      pickup_address,
      dropoff_location,
      dropoff_address,
      flight_number,
      airline,
      terminal,
      arrival_time,
      departure_time,
      number_of_passengers,
      number_of_luggage = 0,
      passenger_name,
      passenger_phone,
      passenger_email,
      vehicle_type,
      vehicle_preferences,
      special_equipment,
      quoted_price,
      currency = 'TRY',
      assigned_to,
      internal_notes,
      customer_notes,
      special_requests
    } = req.body;

    // Validation
    if (!customer_name || !transfer_type || !pickup_date || !pickup_location || !dropoff_location) {
      return res.status(400).json({
        success: false,
        error: 'Customer name, transfer type, pickup date, pickup location, and dropoff location are required'
      });
    }

    if (!passenger_name || !passenger_phone) {
      return res.status(400).json({
        success: false,
        error: 'Passenger name and phone are required'
      });
    }

    if (!vehicle_type) {
      return res.status(400).json({
        success: false,
        error: 'Vehicle type is required'
      });
    }

    const id = uuidv4();

    // Generate request number
    const [[{ count }]] = await db.query('SELECT COUNT(*) as count FROM transfer_requests');
    const request_number = `TRF-${new Date().getFullYear()}-${String(count + 1).padStart(4, '0')}`;

    await db.query(`
      INSERT INTO transfer_requests (
        id, customer_id, customer_type, customer_name, request_number, status, priority,
        transfer_type, pickup_date, pickup_location, pickup_address, dropoff_location, dropoff_address,
        flight_number, airline, terminal, arrival_time, departure_time,
        number_of_passengers, number_of_luggage, passenger_name, passenger_phone, passenger_email,
        vehicle_type, vehicle_preferences, special_equipment,
        quoted_price, currency, assigned_to, internal_notes, customer_notes, special_requests
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `, [
      id, customer_id, customer_type, customer_name, request_number, status, priority,
      transfer_type, pickup_date, pickup_location, pickup_address, dropoff_location, dropoff_address,
      flight_number, airline, terminal, arrival_time, departure_time,
      number_of_passengers, number_of_luggage, passenger_name, passenger_phone, passenger_email,
      vehicle_type, vehicle_preferences, JSON.stringify(special_equipment || []),
      quoted_price, currency, assigned_to, internal_notes, customer_notes, special_requests
    ]);

    const [newRequest] = await db.query('SELECT * FROM transfer_requests WHERE id = ?', [id]);

    res.status(201).json({
      success: true,
      message: 'Transfer request created successfully',
      data: newRequest[0]
    });
  } catch (error) {
    console.error('Error creating transfer request:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to create transfer request',
      message: error.message
    });
  }
});

// ============================================
// PUT /api/transfer-requests/:id
// Update transfer request
// ============================================
router.put('/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const updates = req.body;

    // Check if request exists
    const [existing] = await db.query('SELECT id FROM transfer_requests WHERE id = ?', [id]);
    if (existing.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'Transfer request not found'
      });
    }

    // Build update query dynamically
    const allowedFields = [
      'status', 'priority', 'transfer_type', 'pickup_date', 'pickup_location', 'pickup_address',
      'dropoff_location', 'dropoff_address', 'flight_number', 'airline', 'terminal',
      'arrival_time', 'departure_time', 'number_of_passengers', 'number_of_luggage',
      'passenger_name', 'passenger_phone', 'passenger_email', 'vehicle_type',
      'vehicle_preferences', 'special_equipment', 'driver_name', 'driver_phone',
      'driver_vehicle_plate', 'quoted_price', 'final_price', 'payment_status',
      'driver_en_route_at', 'pickup_completed_at', 'dropoff_completed_at',
      'actual_pickup_time', 'actual_dropoff_time', 'assigned_to', 'processed_by',
      'internal_notes', 'customer_notes', 'driver_notes', 'special_requests'
    ];

    const updateFields = [];
    const values = [];

    Object.keys(updates).forEach(key => {
      if (allowedFields.includes(key) && updates[key] !== undefined) {
        updateFields.push(`${key} = ?`);
        // Stringify JSON fields
        if (key === 'special_equipment') {
          values.push(JSON.stringify(updates[key]));
        } else {
          values.push(updates[key]);
        }
      }
    });

    if (updateFields.length === 0) {
      return res.status(400).json({
        success: false,
        error: 'No valid fields to update'
      });
    }

    values.push(id);

    await db.query(
      `UPDATE transfer_requests SET ${updateFields.join(', ')} WHERE id = ?`,
      values
    );

    const [updated] = await db.query('SELECT * FROM transfer_requests WHERE id = ?', [id]);

    res.json({
      success: true,
      message: 'Transfer request updated successfully',
      data: updated[0]
    });
  } catch (error) {
    console.error('Error updating transfer request:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to update transfer request',
      message: error.message
    });
  }
});

// ============================================
// DELETE /api/transfer-requests/:id
// Delete transfer request
// ============================================
router.delete('/:id', async (req, res) => {
  try {
    const { id } = req.params;

    const [result] = await db.query('DELETE FROM transfer_requests WHERE id = ?', [id]);

    if (result.affectedRows === 0) {
      return res.status(404).json({
        success: false,
        error: 'Transfer request not found'
      });
    }

    res.json({
      success: true,
      message: 'Transfer request deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting transfer request:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to delete transfer request',
      message: error.message
    });
  }
});

// ============================================
// PATCH /api/transfer-requests/:id/assign-driver
// Assign driver to transfer
// ============================================
router.patch('/:id/assign-driver', async (req, res) => {
  try {
    const { id } = req.params;
    const { driver_name, driver_phone, driver_vehicle_plate } = req.body;

    if (!driver_name || !driver_phone) {
      return res.status(400).json({
        success: false,
        error: 'Driver name and phone are required'
      });
    }

    await db.query(`
      UPDATE transfer_requests
      SET driver_name = ?, driver_phone = ?, driver_vehicle_plate = ?, assigned_at = NOW(), status = 'assigned'
      WHERE id = ?
    `, [driver_name, driver_phone, driver_vehicle_plate, id]);

    const [updated] = await db.query('SELECT * FROM transfer_requests WHERE id = ?', [id]);

    res.json({
      success: true,
      message: 'Driver assigned successfully',
      data: updated[0]
    });
  } catch (error) {
    console.error('Error assigning driver:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to assign driver',
      message: error.message
    });
  }
});

// ============================================
// GET /api/transfer-requests/stats/overview
// Get transfer request statistics
// ============================================
router.get('/stats/overview', async (req, res) => {
  try {
    const [[stats]] = await db.query(`
      SELECT
        COUNT(*) as total,
        SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
        SUM(CASE WHEN status = 'assigned' THEN 1 ELSE 0 END) as assigned,
        SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as in_progress,
        SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
        SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled,
        SUM(final_price) as total_revenue,
        AVG(final_price) as average_price
      FROM transfer_requests
    `);

    res.json({
      success: true,
      data: stats
    });
  } catch (error) {
    console.error('Error fetching transfer stats:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to fetch transfer statistics',
      message: error.message
    });
  }
});

module.exports = router;
