const express = require('express');
const router = express.Router();
const SeleniumService = require('../services/SeleniumService');
const PegasusSeleniumService = require('../services/PegasusSeleniumService');
const PegasusNDCService = require('../services/PegasusNDCService');
const FlightStatusService = require('../services/FlightStatusService');
const AmadeusService = require('../services/AmadeusService');

// Initialize services
const flightStatusService = new FlightStatusService();
const amadeusService = new AmadeusService();

// In-memory storage for demo (replace with database)
let tickets = [];
let customers = [];
let checkInJobs = [];

// Initialize scheduler data sources when available
const initializeScheduler = () => {
  if (global.scheduler) {
    global.scheduler.setDataSources(tickets, customers, checkInJobs);
  }
};

// Helper functions
const generateId = () => Date.now().toString() + Math.random().toString(36).substr(2, 9);

const calculateCheckinOpenTime = (departureDate, departureTime) => {
  const departure = new Date(`${departureDate}T${departureTime}`);
  const checkinOpen = new Date(departure.getTime() - 24 * 60 * 60 * 1000); // 24 hours before
  return checkinOpen;
};

const determineCheckinStatus = (checkinOpenTime, departureDate) => {
  const now = new Date();
  const departure = new Date(departureDate);
  
  if (now < checkinOpenTime) {
    return 'waiting'; // Check-in henüz açık değil
  }
  
  if (now >= checkinOpenTime && now < departure) {
    return 'ready'; // Check-in açık, hazır
  }
  
  return 'expired'; // Uçuş geçti
};

// GET /api/tickets - Tüm biletleri listele (Amadeus flight info ile enriched)
router.get('/', async (req, res) => {
  const { status, airline, pnr, enrichWithFlightInfo = 'false' } = req.query;
  let filteredTickets = [...tickets];

  if (status) {
    filteredTickets = filteredTickets.filter(t => t.checkinStatus === status);
  }

  if (airline) {
    filteredTickets = filteredTickets.filter(t => t.airline === airline);
  }

  if (pnr) {
    filteredTickets = filteredTickets.filter(t =>
      t.pnr.toLowerCase().includes(pnr.toLowerCase())
    );
  }

  // Enrichment with Amadeus flight info if requested
  if (enrichWithFlightInfo === 'true') {
    try {
      const enrichedTickets = await Promise.all(
        filteredTickets.map(async (ticket) => {
          try {
            let enrichedTicket = { ...ticket };

            // Skip flight info if already has recent data (less than 30 minutes old)
            if (!ticket.lastFlightUpdate ||
                Date.now() - new Date(ticket.lastFlightUpdate).getTime() >= 30 * 60 * 1000) {

              try {
                // Get flight status from Amadeus
                const flightStatusResult = await amadeusService.getFlightStatus(
                  ticket.airline,
                  ticket.flightNumber.replace(ticket.airline, ''), // Remove airline code from flight number
                  ticket.departureDate.toISOString().split('T')[0]
                );

                if (flightStatusResult.success) {
                  const flightData = flightStatusResult.flight;

                  enrichedTicket = {
                    ...enrichedTicket,
                    amadeusFlightInfo: {
                      status: flightData.flightStatus,
                      departure: flightData.departure,
                      arrival: flightData.arrival,
                      aircraft: flightData.aircraft,
                      operating: flightData.operating,
                      lastUpdated: new Date().toISOString()
                    },
                    lastFlightUpdate: new Date(),
                    updatedAt: new Date()
                  };
                } else {
                  // Flight status not found, but don't fail the enrichment
                  console.log(`⚠️ Flight status not found for ${ticket.pnr} (${ticket.airline}${ticket.flightNumber}), skipping status enrichment`);
                  enrichedTicket.lastFlightUpdate = new Date();
                }
              } catch (statusError) {
                // Log the error but don't fail the entire enrichment process
                console.log(`⚠️ Flight status lookup failed for ${ticket.pnr}: ${statusError.message}`);
                enrichedTicket.lastFlightUpdate = new Date();
              }
            }

            // Get airport information if not present or old (24 hours)
            if (!ticket.airportInfo?.lastUpdated ||
                Date.now() - new Date(ticket.airportInfo.lastUpdated).getTime() >= 24 * 60 * 60 * 1000) {

              try {
                // Get departure airport info
                let departureAirportInfo = null;
                if (ticket.departureAirport) {
                  try {
                    const depResult = await amadeusService.getAirportInfo(ticket.departureAirport);
                    if (depResult.success && depResult.airports.length > 0) {
                      departureAirportInfo = depResult.airports[0];
                    }
                  } catch (depError) {
                    console.log(`⚠️ Departure airport lookup failed for ${ticket.departureAirport}: ${depError.message}`);
                  }
                }

                // Get arrival airport info
                let arrivalAirportInfo = null;
                if (ticket.arrivalAirport) {
                  try {
                    const arrResult = await amadeusService.getAirportInfo(ticket.arrivalAirport);
                    if (arrResult.success && arrResult.airports.length > 0) {
                      arrivalAirportInfo = arrResult.airports[0];
                    }
                  } catch (arrError) {
                    console.log(`⚠️ Arrival airport lookup failed for ${ticket.arrivalAirport}: ${arrError.message}`);
                  }
                }

                // Add airport info to enriched ticket
                enrichedTicket.airportInfo = {
                  departure: departureAirportInfo ? {
                    iataCode: departureAirportInfo.iataCode,
                    name: departureAirportInfo.name,
                    city: departureAirportInfo.address?.cityName,
                    country: departureAirportInfo.address?.countryName,
                    timezone: departureAirportInfo.timeZoneOffset,
                    location: departureAirportInfo.geoCode
                  } : null,
                  arrival: arrivalAirportInfo ? {
                    iataCode: arrivalAirportInfo.iataCode,
                    name: arrivalAirportInfo.name,
                    city: arrivalAirportInfo.address?.cityName,
                    country: arrivalAirportInfo.address?.countryName,
                    timezone: arrivalAirportInfo.timeZoneOffset,
                    location: arrivalAirportInfo.geoCode
                  } : null,
                  lastUpdated: new Date().toISOString()
                };
              } catch (airportError) {
                console.log(`⚠️ Airport enrichment failed for ${ticket.pnr}: ${airportError.message}`);
                // Set lastUpdated to prevent retrying too frequently
                enrichedTicket.airportInfo = {
                  departure: null,
                  arrival: null,
                  lastUpdated: new Date().toISOString()
                };
              }
            }

            return enrichedTicket;
          } catch (error) {
            console.error(`Error enriching ticket ${ticket.pnr}:`, error.message);
            return ticket;
          }
        })
      );

      res.json({
        success: true,
        tickets: enrichedTickets,
        total: enrichedTickets.length,
        enriched: true
      });
    } catch (error) {
      console.error('Error during flight info enrichment:', error);
      // Fall back to basic response
      res.json({
        success: true,
        tickets: filteredTickets,
        total: filteredTickets.length,
        enriched: false,
        enrichmentError: error.message
      });
    }
  } else {
    res.json({
      success: true,
      tickets: filteredTickets,
      total: filteredTickets.length
    });
  }
});

// POST /api/tickets - Yeni bilet ekle
router.post('/', (req, res) => {
  const {
    pnr,
    customerInfo,
    airline,
    flightType,
    flightNumber,
    departureAirport,
    arrivalAirport,
    departureDate,
    departureTime,
    passportNumber,
    passportExpiry,
    visaRequired,
    visaNumber,
    isAutoCheckinEnabled = true,
    notes
  } = req.body;

  // Validation - temel alanlar
  if (!pnr || !customerInfo || !departureDate || !departureTime || !flightType) {
    return res.status(400).json({
      error: 'PNR, müşteri bilgileri, uçuş türü ve uçuş tarih-saati zorunludur'
    });
  }

  if (!customerInfo.lastName || !customerInfo.phone) {
    return res.status(400).json({
      error: 'Müşteri soyadı ve telefonu zorunludur'
    });
  }

  // Dış hat uçuşlar için pasaport kontrolü
  if (flightType === 'international') {
    if (!passportNumber || !passportExpiry) {
      return res.status(400).json({
        error: 'Dış hat uçuşları için pasaport numarası ve son geçerlilik tarihi zorunludur'
      });
    }

    // Vize gerekli ise vize numarası kontrolü
    if (visaRequired && !visaNumber) {
      return res.status(400).json({
        error: 'Vize gerekli seçildi ancak vize numarası girilmedi'
      });
    }
  }

  // Check if PNR already exists
  const existingTicket = tickets.find(t => t.pnr.toUpperCase() === pnr.toUpperCase());
  if (existingTicket) {
    return res.status(409).json({
      error: 'Bu PNR zaten sisteme kayıtlı'
    });
  }

  // Find or create customer
  let customer = customers.find(c => 
    c.firstName === customerInfo.firstName && 
    c.lastName === customerInfo.lastName &&
    c.phone === customerInfo.phone
  );

  if (!customer) {
    customer = {
      id: generateId(),
      firstName: customerInfo.firstName,
      lastName: customerInfo.lastName,
      email: customerInfo.email,
      phone: customerInfo.phone,
      notes: customerInfo.notes || '',
      isVip: customerInfo.isVip || false,
      createdAt: new Date(),
      updatedAt: new Date()
    };
    customers.push(customer);
  }

  // Calculate check-in timing
  const checkinOpenTime = calculateCheckinOpenTime(departureDate, departureTime);
  const checkinStatus = determineCheckinStatus(checkinOpenTime, departureDate);

  // Create ticket
  const ticket = {
    id: generateId(),
    pnr: pnr.toUpperCase(),
    customerId: customer.id,
    customer,
    
    // Flight Details
    airline: airline || 'THY',
    flightType: flightType || 'domestic',
    flightNumber: flightNumber ? flightNumber.toUpperCase() : '',
    departureAirport: departureAirport || '',
    arrivalAirport: arrivalAirport || '',
    departureDate: new Date(departureDate),
    departureTime,
    
    // International flight documents
    passportNumber: flightType === 'international' ? passportNumber : null,
    passportExpiry: flightType === 'international' ? passportExpiry : null,
    visaRequired: flightType === 'international' ? (visaRequired || false) : false,
    visaNumber: flightType === 'international' && visaRequired ? visaNumber : null,
    
    // Check-in Details
    checkinStatus,
    checkinOpenTime,
    checkinCompletedAt: null,
    checkinAttempts: 0,
    lastCheckinAttempt: null,
    
    // Results
    seat: null,
    boardingPassUrl: null,
    smsStatus: 'pending',
    errorMessage: null,
    
    // Meta
    createdAt: new Date(),
    updatedAt: new Date(),
    isAutoCheckinEnabled,
    notes: notes || ''
  };

  tickets.push(ticket);
  
  // Update scheduler data sources
  initializeScheduler();

  // Schedule auto check-in using the scheduler service
  if (isAutoCheckinEnabled && (checkinStatus === 'waiting' || checkinStatus === 'ready')) {
    if (global.scheduler) {
      global.scheduler.scheduleCheckIn(ticket);
    } else {
      // Fallback to old scheduling method
      scheduleAutoCheckin(ticket);
    }
  }

  // If checkin status is 'ready', start immediate check-in
  if (checkinStatus === 'ready' && isAutoCheckinEnabled) {
    console.log(`🚀 Starting immediate check-in for ${pnr}`);
    // Start check-in process in background
    setTimeout(() => {
      performCheckinProcess(ticket, `immediate_${ticket.id}_${Date.now()}`, req.io);
    }, 2000); // Small delay to allow response to be sent first
  }

  console.log(`🎫 New ticket added: ${pnr} for ${customer.firstName} ${customer.lastName}`);

  res.status(201).json({
    success: true,
    message: 'Bilet başarıyla eklendi',
    ticket
  });
});

// POST /api/tickets/bulk - Toplu bilet ekleme
router.post('/bulk', (req, res) => {
  const { tickets: ticketList } = req.body;

  if (!Array.isArray(ticketList) || ticketList.length === 0) {
    return res.status(400).json({
      error: 'Bilet listesi gerekli'
    });
  }

  const results = [];
  const errors = [];

  ticketList.forEach((ticketData, index) => {
    try {
      // Validate each ticket
      const { pnr, customerInfo, flightNumber, departureDate, departureTime } = ticketData;
      
      if (!pnr || !customerInfo || !flightNumber || !departureDate || !departureTime) {
        errors.push({
          index,
          pnr: pnr || 'N/A',
          error: 'Eksik zorunlu alanlar'
        });
        return;
      }

      // Check for duplicates
      const existingTicket = tickets.find(t => t.pnr.toUpperCase() === pnr.toUpperCase());
      if (existingTicket) {
        errors.push({
          index,
          pnr,
          error: 'PNR zaten mevcut'
        });
        return;
      }

      // Process similar to single ticket creation
      let customer = customers.find(c => 
        c.firstName === customerInfo.firstName && 
        c.lastName === customerInfo.lastName &&
        c.phone === customerInfo.phone
      );

      if (!customer) {
        customer = {
          id: generateId(),
          ...customerInfo,
          isVip: customerInfo.isVip || false,
          createdAt: new Date(),
          updatedAt: new Date()
        };
        customers.push(customer);
      }

      const checkinOpenTime = calculateCheckinOpenTime(departureDate, departureTime);
      const checkinStatus = determineCheckinStatus(checkinOpenTime, departureDate);

      const ticket = {
        id: generateId(),
        pnr: pnr.toUpperCase(),
        customerId: customer.id,
        customer,
        airline: ticketData.airline || 'THY',
        flightNumber: flightNumber.toUpperCase(),
        departureAirport: ticketData.departureAirport || '',
        arrivalAirport: ticketData.arrivalAirport || '',
        departureDate: new Date(departureDate),
        departureTime,
        checkinStatus,
        checkinOpenTime,
        checkinCompletedAt: null,
        checkinAttempts: 0,
        lastCheckinAttempt: null,
        seat: null,
        boardingPassUrl: null,
        smsStatus: 'pending',
        errorMessage: null,
        createdAt: new Date(),
        updatedAt: new Date(),
        isAutoCheckinEnabled: ticketData.isAutoCheckinEnabled !== false,
        notes: ticketData.notes || ''
      };

      tickets.push(ticket);
      results.push(ticket);

      // Schedule auto check-in using scheduler service
      if (ticket.isAutoCheckinEnabled && (checkinStatus === 'waiting' || checkinStatus === 'ready')) {
        if (global.scheduler) {
          global.scheduler.scheduleCheckIn(ticket);
        } else {
          scheduleAutoCheckin(ticket);
        }
      }

      // If checkin status is 'ready', start immediate check-in
      if (checkinStatus === 'ready' && ticket.isAutoCheckinEnabled) {
        console.log(`🚀 Starting immediate check-in for ${ticket.pnr}`);
        // Start check-in process in background
        setTimeout(() => {
          performCheckinProcess(ticket, `bulk_immediate_${ticket.id}_${Date.now()}`);
        }, 2000); // Small delay
      }

    } catch (error) {
      errors.push({
        index,
        pnr: ticketData.pnr || 'N/A',
        error: error.message
      });
    }
  });

  // Update scheduler data sources after bulk insert
  initializeScheduler();

  res.json({
    success: true,
    message: `${results.length} bilet başarıyla eklendi`,
    results: {
      successful: results.length,
      failed: errors.length,
      tickets: results,
      errors
    }
  });
});

// PUT /api/tickets/:id - Bilet güncelle
router.put('/:id', (req, res) => {
  const { id } = req.params;
  const updateData = req.body;

  const ticketIndex = tickets.findIndex(t => t.id === id);
  if (ticketIndex === -1) {
    return res.status(404).json({
      error: 'Bilet bulunamadı'
    });
  }

  const ticket = tickets[ticketIndex];
  
  // Update allowed fields
  const allowedUpdates = ['notes', 'isAutoCheckinEnabled', 'departureDate', 'departureTime'];
  allowedUpdates.forEach(field => {
    if (updateData[field] !== undefined) {
      ticket[field] = updateData[field];
    }
  });

  // Recalculate check-in status if date/time changed
  if (updateData.departureDate || updateData.departureTime) {
    ticket.checkinOpenTime = calculateCheckinOpenTime(ticket.departureDate, ticket.departureTime);
    ticket.checkinStatus = determineCheckinStatus(ticket.checkinOpenTime, ticket.departureDate);
  }

  ticket.updatedAt = new Date();
  tickets[ticketIndex] = ticket;

  res.json({
    success: true,
    message: 'Bilet güncellendi',
    ticket
  });
});

// DELETE /api/tickets/:id - Bilet sil
router.delete('/:id', (req, res) => {
  const { id } = req.params;
  
  const ticketIndex = tickets.findIndex(t => t.id === id);
  if (ticketIndex === -1) {
    return res.status(404).json({
      error: 'Bilet bulunamadı'
    });
  }

  const removedTicket = tickets.splice(ticketIndex, 1)[0];
  
  // Cancel any scheduled check-in jobs
  checkInJobs = checkInJobs.filter(job => job.ticketId !== id);

  res.json({
    success: true,
    message: 'Bilet silindi',
    ticket: removedTicket
  });
});

// POST /api/tickets/test-checkin - Test check-in with visible browser
router.post('/test-checkin', async (req, res) => {
  const { pnr, lastName, visible = true } = req.body;
  
  if (!pnr || !lastName) {
    return res.status(400).json({
      error: 'PNR ve soyisim gerekli'
    });
  }

  const jobId = `test_${pnr}_${Date.now()}`;
  
  res.json({
    success: true,
    message: 'Test check-in başlatıldı',
    jobId,
    visible
  });

  // Perform test check-in in background with visible browser
  performTestCheckin({ pnr, lastName, visible }, jobId, req.io);
});

// POST /api/tickets/:id/checkin - Manuel check-in başlat
router.post('/:id/checkin', async (req, res) => {
  const { id } = req.params;
  
  const ticket = tickets.find(t => t.id === id);
  if (!ticket) {
    return res.status(404).json({
      error: 'Bilet bulunamadı'
    });
  }

  if (ticket.checkinStatus === 'completed') {
    return res.status(400).json({
      error: 'Check-in zaten tamamlanmış'
    });
  }

  if (ticket.checkinStatus === 'expired') {
    return res.status(400).json({
      error: 'Check-in süresi geçmiş'
    });
  }

  const jobId = `manual_${id}_${Date.now()}`;
  
  // Update ticket status
  ticket.checkinStatus = 'processing';
  ticket.checkinAttempts += 1;
  ticket.lastCheckinAttempt = new Date();
  ticket.updatedAt = new Date();

  res.json({
    success: true,
    message: 'Manuel check-in başlatıldı',
    jobId,
    ticket
  });

  // Perform check-in in background
  performCheckinProcess(ticket, jobId, req.io);
});

// Helper function to schedule auto check-in
function scheduleAutoCheckin(ticket) {
  const scheduledFor = new Date(ticket.checkinOpenTime.getTime() + 5000); // 5 seconds after check-in opens
  
  const job = {
    id: generateId(),
    ticketId: ticket.id,
    scheduledFor,
    status: 'scheduled',
    attempts: 0,
    createdAt: new Date(),
    updatedAt: new Date()
  };

  checkInJobs.push(job);
  
  const delay = scheduledFor.getTime() - Date.now();
  if (delay > 0) {
    setTimeout(() => {
      executeScheduledCheckin(job);
    }, delay);
  }

  console.log(`⏰ Auto check-in scheduled for ticket ${ticket.pnr} at ${scheduledFor.toISOString()}`);
}

// Execute scheduled check-in
async function executeScheduledCheckin(job) {
  const ticket = tickets.find(t => t.id === job.ticketId);
  if (!ticket) {
    console.log(`❌ Ticket not found for scheduled job: ${job.id}`);
    return;
  }

  console.log(`🤖 Executing auto check-in for ${ticket.pnr}`);
  
  job.status = 'processing';
  job.attempts += 1;
  job.updatedAt = new Date();

  ticket.checkinStatus = 'processing';
  ticket.checkinAttempts += 1;
  ticket.lastCheckinAttempt = new Date();
  ticket.updatedAt = new Date();

  await performCheckinProcess(ticket, job.id);
  
  job.status = ticket.checkinStatus === 'completed' ? 'completed' : 'failed';
  job.updatedAt = new Date();
}

// Test check-in process with visible browser
async function performTestCheckin({ pnr, lastName, visible }, jobId, io = null) {
  try {
    const checkinService = new SeleniumService();
    
    if (io) {
      io.emit('checkin_status', {
        jobId,
        status: 'processing',
        message: `${pnr} için test check-in başlatılıyor (visible: ${visible})...`,
        progress: 25
      });
    }

    // Initialize driver with visible browser
    await checkinService.initDriver(!visible); // headless = !visible
    
    const result = await checkinService.performCheckin({
      pnr,
      lastName: lastName.toUpperCase(),
      firstName: '',
      phone: ''
    });

    if (io) {
      io.emit('checkin_result', {
        jobId,
        success: result.success,
        message: result.message,
        result,
        screenshots: result.screenshots
      });
    }

    // Close service after test
    await checkinService.close();

    console.log(`${result.success ? '✅' : '❌'} Test check-in ${result.success ? 'completed' : 'failed'} for ${pnr}: ${result.message}`);

  } catch (error) {
    console.error(`❌ Test check-in error for ${pnr}:`, error);
    
    if (io) {
      io.emit('checkin_result', {
        jobId,
        success: false,
        message: `Test check-in hatası: ${error.message}`
      });
    }
  }
}

// Common check-in process
async function performCheckinProcess(ticket, jobId, io = null) {
  try {
    let checkinService;
    let result;

    // Havayoluna göre servis seçimi
    if (ticket.airline === 'PC') {
      // Pegasus için web scraping kullan (Selenium)
      checkinService = new PegasusSeleniumService();
      
      if (io) {
        io.emit('checkin_status', {
          jobId,
          ticketId: ticket.id,
          status: 'processing',
          message: `${ticket.pnr} için Pegasus web check-in başlatılıyor...`,
          progress: 25
        });
      }

      result = await checkinService.performCheckin({
        pnr: ticket.pnr,
        lastName: ticket.customer.lastName.toUpperCase(),
        firstName: ticket.customer.firstName,
        phone: ticket.customer.phone,
        flightType: ticket.flightType,
        passportNumber: ticket.passportNumber,
        visaNumber: ticket.visaNumber
      });
      
    } else if (ticket.airline === 'THY') {
      // THY için Selenium kullan
      checkinService = new SeleniumService();
      
      if (io) {
        io.emit('checkin_status', {
          jobId,
          ticketId: ticket.id,
          status: 'processing',
          message: `${ticket.pnr} için THY Selenium check-in başlatılıyor...`,
          progress: 25
        });
      }

      result = await checkinService.performCheckin({
        pnr: ticket.pnr,
        lastName: ticket.customer.lastName.toUpperCase(),
        firstName: ticket.customer.firstName,
        phone: ticket.customer.phone
      });
      
    } else {
      // Diğer havayolları için henüz destek yok
      result = {
        success: false,
        message: `${ticket.airline} havayolu için otomatik check-in henüz desteklenmiyor`
      };
    }

    // Update ticket based on result
    if (result.success) {
      ticket.checkinStatus = 'completed';
      ticket.checkinCompletedAt = new Date();
      ticket.seat = result.seat;
      ticket.boardingPassUrl = result.boardingPassUrl;
      ticket.errorMessage = null;
    } else {
      ticket.checkinStatus = 'failed';
      ticket.errorMessage = result.message;
    }

    ticket.updatedAt = new Date();

    if (io) {
      io.emit('checkin_result', {
        jobId,
        ticketId: ticket.id,
        success: result.success,
        message: result.message,
        ticket
      });
    }

    // Selenium servisini kapat (sadece THY için)
    if (checkinService && typeof checkinService.close === 'function') {
      await checkinService.close();
    }

    console.log(`${result.success ? '✅' : '❌'} Check-in ${result.success ? 'completed' : 'failed'} for ${ticket.pnr}: ${result.message}`);

  } catch (error) {
    console.error(`❌ Check-in error for ${ticket.pnr}:`, error);
    
    ticket.checkinStatus = 'failed';
    ticket.errorMessage = error.message;
    ticket.updatedAt = new Date();

    if (io) {
      io.emit('checkin_result', {
        jobId,
        ticketId: ticket.id,
        success: false,
        message: `Check-in hatası: ${error.message}`,
        ticket
      });
    }
  }
}

// GET /api/tickets/dashboard - Dashboard istatistikleri
router.get('/dashboard', (req, res) => {
  const now = new Date();
  const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
  const tomorrow = new Date(today.getTime() + 24 * 60 * 60 * 1000);

  // Today's check-ins (completed today)
  const todayCheckIns = tickets.filter(t => 
    t.checkinCompletedAt && 
    t.checkinCompletedAt >= today && 
    t.checkinCompletedAt < tomorrow
  ).length;

  // Today's scheduled check-ins
  const todayScheduled = tickets.filter(t => 
    t.checkinOpenTime && 
    t.checkinOpenTime >= today && 
    t.checkinOpenTime < tomorrow &&
    t.isAutoCheckinEnabled
  ).length;

  // Success rate (last 7 days)
  const weekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
  const recentTickets = tickets.filter(t => t.createdAt >= weekAgo);
  const recentCompleted = recentTickets.filter(t => t.checkinStatus === 'completed').length;
  const successRate = recentTickets.length > 0 ? Math.round((recentCompleted / recentTickets.length) * 100) : 0;

  // Upcoming flights (next 7 days)
  const nextWeek = new Date(now.getTime() + 7 * 24 * 60 * 60 * 1000);
  const upcomingFlights = tickets.filter(t => 
    t.departureDate >= now && 
    t.departureDate <= nextWeek
  ).length;

  // Active tickets
  const totalActiveTickets = tickets.filter(t => 
    t.checkinStatus !== 'expired' && 
    t.checkinStatus !== 'cancelled'
  ).length;

  // Failed check-ins
  const failedCheckIns = tickets.filter(t => t.checkinStatus === 'failed').length;

  const stats = {
    todayCheckIns,
    todayScheduled,
    successRate,
    upcomingFlights,
    totalActiveTickets,
    failedCheckIns
  };

  res.json({
    success: true,
    stats
  });
});

// GET /api/tickets/jobs - Scheduled jobs status
router.get('/jobs', (req, res) => {
  res.json({
    success: true,
    jobs: checkInJobs,
    total: checkInJobs.length
  });
});

// GET /api/tickets/scheduler/status - Get scheduler status
router.get('/scheduler/status', (req, res) => {
  if (!global.scheduler) {
    return res.status(503).json({
      error: 'Scheduler service not available'
    });
  }

  const status = global.scheduler.getStatus();
  res.json({
    success: true,
    scheduler: status
  });
});

// POST /api/tickets/scheduler/force/:jobId - Force execute a scheduled job
router.post('/scheduler/force/:jobId', async (req, res) => {
  const { jobId } = req.params;
  
  if (!global.scheduler) {
    return res.status(503).json({
      error: 'Scheduler service not available'
    });
  }

  try {
    const result = await global.scheduler.forceExecuteJob(jobId);
    
    if (result) {
      res.json({
        success: true,
        message: 'Job force executed successfully'
      });
    } else {
      res.status(404).json({
        error: 'Job not found or not schedulable'
      });
    }
  } catch (error) {
    res.status(500).json({
      error: 'Failed to force execute job',
      message: error.message
    });
  }
});

// DELETE /api/tickets/scheduler/cancel/:jobId - Cancel a scheduled job
router.delete('/scheduler/cancel/:jobId', (req, res) => {
  const { jobId } = req.params;
  
  if (!global.scheduler) {
    return res.status(503).json({
      error: 'Scheduler service not available'
    });
  }

  const result = global.scheduler.cancelJob(jobId);
  
  if (result) {
    res.json({
      success: true,
      message: 'Job cancelled successfully'
    });
  } else {
    res.status(404).json({
      error: 'Job not found or cannot be cancelled'
    });
  }
});

// GET /api/tickets/:id/jobs - Get jobs for a specific ticket
router.get('/:id/jobs', (req, res) => {
  const { id } = req.params;
  
  if (!global.scheduler) {
    return res.status(503).json({
      error: 'Scheduler service not available'
    });
  }

  const jobs = global.scheduler.getJobsForTicket(id);
  
  res.json({
    success: true,
    jobs,
    total: jobs.length
  });
});

// GET /api/tickets/:id/flight-status - Get flight status for a ticket (with Amadeus)
router.get('/:id/flight-status', async (req, res) => {
  const { id } = req.params;
  const { useAmadeus = 'true' } = req.query;

  const ticket = tickets.find(t => t.id === id);
  if (!ticket) {
    return res.status(404).json({
      error: 'Bilet bulunamadı'
    });
  }

  try {
    let flightStatus;
    let dataSource = 'mock';

    // Try Amadeus first if enabled
    if (useAmadeus === 'true') {
      try {
        const amadeusResult = await amadeusService.getFlightStatus(
          ticket.airline,
          ticket.flightNumber.replace(ticket.airline, ''),
          ticket.departureDate.toISOString().split('T')[0]
        );

        if (amadeusResult.success) {
          flightStatus = amadeusResult.flight;
          dataSource = 'amadeus';
          console.log(`✅ Flight status from Amadeus for ${ticket.pnr}`);
        } else {
          console.log(`⚠️ Amadeus failed for ${ticket.pnr}, falling back to mock`);
          flightStatus = await flightStatusService.getFlightStatus(
            ticket.airline,
            ticket.flightNumber,
            ticket.departureDate.toISOString().split('T')[0]
          );
        }
      } catch (error) {
        console.error(`❌ Amadeus error for ${ticket.pnr}:`, error.message);
        flightStatus = await flightStatusService.getFlightStatus(
          ticket.airline,
          ticket.flightNumber,
          ticket.departureDate.toISOString().split('T')[0]
        );
      }
    } else {
      flightStatus = await flightStatusService.getFlightStatus(
        ticket.airline,
        ticket.flightNumber,
        ticket.departureDate.toISOString().split('T')[0]
      );
    }

    const timingInfo = flightStatusService.getTicketTimingInfo(ticket);

    res.json({
      success: true,
      ticket: {
        id: ticket.id,
        pnr: ticket.pnr,
        airline: ticket.airline,
        flightNumber: ticket.flightNumber
      },
      flightStatus,
      timing: timingInfo,
      dataSource
    });

  } catch (error) {
    console.error('Flight status error:', error);
    res.status(500).json({
      error: 'Flight status bilgisi alınamadı',
      message: error.message
    });
  }
});

// POST /api/tickets/:id/update-flight-status - Update ticket with latest flight status
router.post('/:id/update-flight-status', async (req, res) => {
  const { id } = req.params;
  
  const ticketIndex = tickets.findIndex(t => t.id === id);
  if (ticketIndex === -1) {
    return res.status(404).json({
      error: 'Bilet bulunamadı'
    });
  }

  try {
    const ticket = tickets[ticketIndex];
    const updatedTicket = await flightStatusService.updateTicketWithFlightStatus(ticket);
    
    // Update in storage
    tickets[ticketIndex] = updatedTicket;
    
    // Update scheduler data
    initializeScheduler();

    // Emit real-time update
    if (global.io) {
      global.io.emit('ticket_status_update', {
        ticketId: id,
        status: updatedTicket.checkinStatus,
        message: 'Flight status updated',
        ticket: updatedTicket
      });
    }

    res.json({
      success: true,
      message: 'Flight status güncellendi',
      ticket: updatedTicket
    });

  } catch (error) {
    console.error('Update flight status error:', error);
    res.status(500).json({
      error: 'Flight status güncellenemedi',
      message: error.message
    });
  }
});

// GET /api/tickets/:id/timing - Get detailed timing information for a ticket
router.get('/:id/timing', (req, res) => {
  const { id } = req.params;
  
  const ticket = tickets.find(t => t.id === id);
  if (!ticket) {
    return res.status(404).json({
      error: 'Bilet bulunamadı'
    });
  }

  const timingInfo = flightStatusService.getTicketTimingInfo(ticket);

  res.json({
    success: true,
    timing: timingInfo
  });
});

// POST /api/tickets/update-all-flight-status - Bulk update flight status for all active tickets
router.post('/update-all-flight-status', async (req, res) => {
  const activeTickets = tickets.filter(t => 
    t.checkinStatus !== 'completed' && 
    t.checkinStatus !== 'expired' && 
    t.checkinStatus !== 'cancelled'
  );

  if (activeTickets.length === 0) {
    return res.json({
      success: true,
      message: 'Güncellenecek aktif bilet bulunamadı',
      updated: 0
    });
  }

  let updated = 0;
  const errors = [];

  for (const ticket of activeTickets) {
    try {
      const updatedTicket = await flightStatusService.updateTicketWithFlightStatus(ticket);
      
      // Find and update in storage
      const ticketIndex = tickets.findIndex(t => t.id === ticket.id);
      if (ticketIndex !== -1) {
        tickets[ticketIndex] = updatedTicket;
        updated++;
      }

      // Small delay to avoid overwhelming APIs
      await new Promise(resolve => setTimeout(resolve, 500));

    } catch (error) {
      console.error(`Error updating ${ticket.pnr}:`, error);
      errors.push({
        ticketId: ticket.id,
        pnr: ticket.pnr,
        error: error.message
      });
    }
  }

  // Update scheduler with new data
  initializeScheduler();

  res.json({
    success: true,
    message: `${updated} bilet güncellendi`,
    updated,
    errors: errors.length > 0 ? errors : undefined
  });
});

// POST /api/tickets/search-flights - Search flights for booking
router.post('/search-flights', async (req, res) => {
  const {
    origin,
    destination,
    departureDate,
    returnDate,
    passengers = { adults: 1, children: 0, infants: 0 },
    travelClass = 'ECONOMY',
    nonStop = false,
    currency = 'TRY'
  } = req.body;

  if (!origin || !destination || !departureDate) {
    return res.status(400).json({
      error: 'Origin, destination and departure date are required'
    });
  }

  try {
    console.log(`🔍 Flight search for booking: ${origin} → ${destination}`);

    const searchParams = {
      originLocationCode: origin.toUpperCase(),
      destinationLocationCode: destination.toUpperCase(),
      departureDate,
      returnDate,
      adults: passengers.adults,
      children: passengers.children || 0,
      infants: passengers.infants || 0,
      travelClass,
      nonStop,
      currencyCode: currency,
      max: 25
    };

    const searchResult = await amadeusService.searchFlights(searchParams);

    if (searchResult.success) {
      const formattedOffers = searchResult.data.map(offer =>
        amadeusService.formatFlightOffer(offer)
      );

      res.json({
        success: true,
        offers: formattedOffers,
        count: formattedOffers.length,
        searchParams: { origin, destination, departureDate, returnDate, passengers, travelClass }
      });
    } else {
      res.status(400).json({
        success: false,
        error: searchResult.error,
        message: searchResult.message
      });
    }

  } catch (error) {
    console.error('Flight search error:', error);
    res.status(500).json({
      success: false,
      error: 'Flight search failed',
      message: error.message
    });
  }
});

// POST /api/tickets/create-from-amadeus - Create ticket from Amadeus booking
router.post('/create-from-amadeus', async (req, res) => {
  const {
    flightOffer,
    travelers,
    contacts,
    customerInfo,
    isAutoCheckinEnabled = true,
    notes
  } = req.body;

  if (!flightOffer || !travelers || !customerInfo) {
    return res.status(400).json({
      error: 'Flight offer, travelers and customer info are required'
    });
  }

  try {
    console.log(`🎫 Creating Amadeus booking for ${travelers.length} traveler(s)`);

    // Create booking with Amadeus
    const bookingResult = await amadeusService.createBooking({
      flightOffers: [flightOffer],
      travelers,
      contacts
    });

    if (!bookingResult.success) {
      return res.status(400).json({
        success: false,
        error: bookingResult.error,
        message: bookingResult.message
      });
    }

    // Extract flight information from booking
    const booking = bookingResult.booking;
    const pnr = bookingResult.pnr;
    const firstFlight = booking.flightOffers[0];
    const outbound = firstFlight.itineraries[0];
    const firstSegment = outbound.segments[0];

    // Determine flight type
    const isDomestic = firstSegment.departure.iataCode.startsWith('TR') &&
                      firstSegment.arrival.iataCode.startsWith('TR');

    // Find or create customer
    let customer = customers.find(c =>
      c.firstName === customerInfo.firstName &&
      c.lastName === customerInfo.lastName &&
      c.phone === customerInfo.phone
    );

    if (!customer) {
      customer = {
        id: generateId(),
        ...customerInfo,
        isVip: customerInfo.isVip || false,
        createdAt: new Date(),
        updatedAt: new Date()
      };
      customers.push(customer);
    }

    // Calculate check-in timing
    const departureDateTime = new Date(firstSegment.departure.at);
    const checkinOpenTime = calculateCheckinOpenTime(
      departureDateTime.toISOString().split('T')[0],
      departureDateTime.toTimeString().slice(0, 5)
    );
    const checkinStatus = determineCheckinStatus(checkinOpenTime, departureDateTime);

    // Create ticket from Amadeus booking
    const ticket = {
      id: generateId(),
      pnr: pnr,
      amadeusBookingId: booking.id,
      customerId: customer.id,
      customer,

      // Flight Details from Amadeus
      airline: firstSegment.carrierCode,
      flightType: isDomestic ? 'domestic' : 'international',
      flightNumber: `${firstSegment.carrierCode}${firstSegment.number}`,
      departureAirport: firstSegment.departure.iataCode,
      arrivalAirport: firstSegment.arrival.iataCode,
      departureDate: departureDateTime,
      departureTime: departureDateTime.toTimeString().slice(0, 5),

      // Amadeus booking data
      amadeusData: {
        booking: booking,
        travelers: travelers,
        totalPrice: firstFlight.price,
        segments: outbound.segments
      },

      // Check-in Details
      checkinStatus,
      checkinOpenTime,
      checkinCompletedAt: null,
      checkinAttempts: 0,
      lastCheckinAttempt: null,

      // Results
      seat: null,
      boardingPassUrl: null,
      smsStatus: 'pending',
      errorMessage: null,

      // Meta
      createdAt: new Date(),
      updatedAt: new Date(),
      isAutoCheckinEnabled,
      notes: notes || '',
      source: 'amadeus'
    };

    tickets.push(ticket);

    // Update scheduler data sources
    initializeScheduler();

    // Schedule auto check-in if enabled
    if (isAutoCheckinEnabled && (checkinStatus === 'waiting' || checkinStatus === 'ready')) {
      if (global.scheduler) {
        global.scheduler.scheduleCheckIn(ticket);
      }
    }

    console.log(`✅ Ticket created from Amadeus booking: ${pnr}`);

    res.status(201).json({
      success: true,
      message: 'Ticket created from Amadeus booking',
      ticket,
      amadeusBooking: booking
    });

  } catch (error) {
    console.error('Amadeus ticket creation error:', error);
    res.status(500).json({
      success: false,
      error: 'Failed to create ticket from Amadeus booking',
      message: error.message
    });
  }
});

// GET /api/tickets/amadeus/test - Test Amadeus connection
router.get('/amadeus/test', async (req, res) => {
  try {
    const isConnected = await amadeusService.testConnection();

    res.json({
      success: isConnected,
      message: isConnected ? 'Amadeus connection successful' : 'Amadeus connection failed',
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: 'Amadeus connection test failed',
      message: error.message
    });
  }
});

// GET /api/tickets/amadeus/airports/:keyword - Search airports
router.get('/amadeus/airports/:keyword', async (req, res) => {
  const { keyword } = req.params;

  if (!keyword || keyword.length < 2) {
    return res.status(400).json({
      error: 'Keyword must be at least 2 characters'
    });
  }

  try {
    const result = await flightStatusService.searchAirports(keyword);

    if (result.success) {
      res.json({
        success: true,
        airports: result.airports
      });
    } else {
      res.status(400).json({
        success: false,
        error: result.message
      });
    }
  } catch (error) {
    res.status(500).json({
      success: false,
      error: 'Airport search failed',
      message: error.message
    });
  }
});

// GET /api/tickets/amadeus/checkin-links/:airlineCode - Get check-in links
router.get('/amadeus/checkin-links/:airlineCode', async (req, res) => {
  const { airlineCode } = req.params;
  const { language = 'en-GB' } = req.query;

  try {
    const result = await flightStatusService.getCheckinLinks(airlineCode, language);

    if (result.success) {
      res.json({
        success: true,
        links: result.links
      });
    } else {
      res.status(400).json({
        success: false,
        error: result.message
      });
    }
  } catch (error) {
    res.status(500).json({
      success: false,
      error: 'Check-in links lookup failed',
      message: error.message
    });
  }
});

// GET /api/tickets/:id/enrich-airports - Enrich ticket with airport information
router.get('/:id/enrich-airports', async (req, res) => {
  const { id } = req.params;

  const ticketIndex = tickets.findIndex(t => t.id === id);
  if (ticketIndex === -1) {
    return res.status(404).json({
      error: 'Bilet bulunamadı'
    });
  }

  const ticket = tickets[ticketIndex];

  try {
    console.log(`🏢 Enriching airports for ticket ${ticket.pnr}: ${ticket.departureAirport} → ${ticket.arrivalAirport}`);

    // Get departure airport info
    let departureAirportInfo = null;
    if (ticket.departureAirport) {
      const depResult = await amadeusService.getAirportInfo(ticket.departureAirport);
      if (depResult.success && depResult.airports.length > 0) {
        departureAirportInfo = depResult.airports[0];
      }
    }

    // Get arrival airport info
    let arrivalAirportInfo = null;
    if (ticket.arrivalAirport) {
      const arrResult = await amadeusService.getAirportInfo(ticket.arrivalAirport);
      if (arrResult.success && arrResult.airports.length > 0) {
        arrivalAirportInfo = arrResult.airports[0];
      }
    }

    // Update ticket with airport information
    const enrichedTicket = {
      ...ticket,
      airportInfo: {
        departure: departureAirportInfo ? {
          iataCode: departureAirportInfo.iataCode,
          name: departureAirportInfo.name,
          city: departureAirportInfo.address?.cityName,
          country: departureAirportInfo.address?.countryName,
          timezone: departureAirportInfo.timeZoneOffset,
          location: departureAirportInfo.geoCode
        } : null,
        arrival: arrivalAirportInfo ? {
          iataCode: arrivalAirportInfo.iataCode,
          name: arrivalAirportInfo.name,
          city: arrivalAirportInfo.address?.cityName,
          country: arrivalAirportInfo.address?.countryName,
          timezone: arrivalAirportInfo.timeZoneOffset,
          location: arrivalAirportInfo.geoCode
        } : null,
        lastUpdated: new Date().toISOString()
      },
      updatedAt: new Date()
    };

    // Update in storage
    tickets[ticketIndex] = enrichedTicket;

    console.log(`✅ Airport info enriched for ${ticket.pnr}`);

    res.json({
      success: true,
      message: 'Airport information updated',
      ticket: enrichedTicket,
      airportInfo: enrichedTicket.airportInfo
    });

  } catch (error) {
    console.error(`❌ Airport enrichment error for ${ticket.pnr}:`, error);
    res.status(500).json({
      success: false,
      error: 'Airport information lookup failed',
      message: error.message
    });
  }
});

// POST /api/tickets/enrich-all-airports - Bulk enrich airports for all tickets
router.post('/enrich-all-airports', async (req, res) => {
  const activeTickets = tickets.filter(t =>
    t.checkinStatus !== 'completed' &&
    t.checkinStatus !== 'expired' &&
    t.checkinStatus !== 'cancelled' &&
    (t.departureAirport || t.arrivalAirport)
  );

  if (activeTickets.length === 0) {
    return res.json({
      success: true,
      message: 'No active tickets found for airport enrichment',
      enriched: 0
    });
  }

  let enriched = 0;
  const errors = [];

  console.log(`🏢 Starting bulk airport enrichment for ${activeTickets.length} tickets`);

  for (const ticket of activeTickets) {
    try {
      // Skip if already has recent airport info (less than 24 hours old)
      if (ticket.airportInfo?.lastUpdated &&
          Date.now() - new Date(ticket.airportInfo.lastUpdated).getTime() < 24 * 60 * 60 * 1000) {
        continue;
      }

      // Get departure airport info
      let departureAirportInfo = null;
      if (ticket.departureAirport) {
        const depResult = await amadeusService.getAirportInfo(ticket.departureAirport);
        if (depResult.success && depResult.airports.length > 0) {
          departureAirportInfo = depResult.airports[0];
        }
      }

      // Get arrival airport info
      let arrivalAirportInfo = null;
      if (ticket.arrivalAirport) {
        const arrResult = await amadeusService.getAirportInfo(ticket.arrivalAirport);
        if (arrResult.success && arrResult.airports.length > 0) {
          arrivalAirportInfo = arrResult.airports[0];
        }
      }

      // Update ticket
      const ticketIndex = tickets.findIndex(t => t.id === ticket.id);
      if (ticketIndex !== -1) {
        tickets[ticketIndex] = {
          ...tickets[ticketIndex],
          airportInfo: {
            departure: departureAirportInfo ? {
              iataCode: departureAirportInfo.iataCode,
              name: departureAirportInfo.name,
              city: departureAirportInfo.address?.cityName,
              country: departureAirportInfo.address?.countryName,
              timezone: departureAirportInfo.timeZoneOffset,
              location: departureAirportInfo.geoCode
            } : null,
            arrival: arrivalAirportInfo ? {
              iataCode: arrivalAirportInfo.iataCode,
              name: arrivalAirportInfo.name,
              city: arrivalAirportInfo.address?.cityName,
              country: arrivalAirportInfo.address?.countryName,
              timezone: arrivalAirportInfo.timeZoneOffset,
              location: arrivalAirportInfo.geoCode
            } : null,
            lastUpdated: new Date().toISOString()
          },
          updatedAt: new Date()
        };
        enriched++;
      }

      // Rate limiting to avoid overwhelming Amadeus API
      await new Promise(resolve => setTimeout(resolve, 200));

    } catch (error) {
      console.error(`Error enriching airports for ${ticket.pnr}:`, error);
      errors.push({
        ticketId: ticket.id,
        pnr: ticket.pnr,
        error: error.message
      });
    }
  }

  console.log(`✅ Bulk airport enrichment completed: ${enriched} tickets enriched`);

  res.json({
    success: true,
    message: `${enriched} tickets enriched with airport information`,
    enriched,
    total: activeTickets.length,
    errors: errors.length > 0 ? errors : undefined
  });
});

// Clean up flight status cache periodically
setInterval(() => {
  flightStatusService.cleanCache();
}, 15 * 60 * 1000); // Clean every 15 minutes

module.exports = router;