const express = require('express');
const router = express.Router();
const {
  listTicketRequests,
  getTicketRequest,
  createTicketRequest,
  updateTicketRequest,
  deleteTicketRequest,
  assignPnrToTicket,
  getTicketWithPassengers,
  getUpcomingTickets,
  addPassengerToTicket,
  removePassengerFromTicket,
  createPassenger,
  getPassenger,
  listCheckinTickets,
  createCheckinTicket
} = require('../services/database');

// GET /api/tickets - List all tickets (ticket_requests from database)
router.get('/', async (req, res) => {
  try {
    const {
      customerId,
      status,
      airline,
      departureDate,
      search,
      limit,
      offset
    } = req.query;

    const filters = {};
    if (customerId) filters.customerId = customerId;
    if (status) filters.status = status;
    if (airline) filters.airline = airline;
    if (departureDate) filters.departureDate = departureDate;
    if (search) filters.search = search;

    const pagination = {
      limit: Math.min(parseInt(limit) || 20, 100),
      offset: parseInt(offset) || 0
    };

    const result = await listTicketRequests(filters, pagination);

    res.json({
      success: true,
      data: result.data,
      total: result.total,
      limit: pagination.limit,
      offset: pagination.offset
    });
  } catch (error) {
    console.error('Error listing tickets:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// GET /api/tickets/upcoming - Get upcoming tickets (next 24 hours)
router.get('/upcoming', async (req, res) => {
  try {
    const tickets = await getUpcomingTickets();
    res.json({
      success: true,
      data: tickets
    });
  } catch (error) {
    console.error('Error getting upcoming tickets:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// GET /api/tickets/:id - Get ticket details with passengers
router.get('/:id', async (req, res) => {
  try {
    const ticket = await getTicketWithPassengers(req.params.id);

    if (!ticket) {
      return res.status(404).json({
        success: false,
        error: 'Ticket not found'
      });
    }

    res.json({
      success: true,
      data: ticket
    });
  } catch (error) {
    console.error('Error getting ticket:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// POST /api/tickets - Create new ticket request
router.post('/', async (req, res) => {
  try {
    const {
      customerId,
      airline,
      departureAirport,
      arrivalAirport,
      departureDate,
      passengers
    } = req.body;

    // Validation
    if (!customerId || !airline || !departureAirport || !arrivalAirport || !departureDate) {
      return res.status(400).json({
        success: false,
        error: 'Missing required fields'
      });
    }

    if (!passengers || passengers.length === 0) {
      return res.status(400).json({
        success: false,
        error: 'At least one passenger is required'
      });
    }

    // Create ticket
    const ticketData = {
      ...req.body,
      status: req.body.status || 'pending',
      requestDate: new Date().toISOString()
    };

    const ticket = await createTicketRequest(ticketData);

    // Emit real-time update
    if (global.io) {
      global.io.emit('ticket_created', { ticket });
    }

    res.status(201).json({
      success: true,
      data: ticket
    });
  } catch (error) {
    console.error('Error creating ticket:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// PUT /api/tickets/:id - Update ticket
router.put('/:id', async (req, res) => {
  try {
    const ticket = await updateTicketRequest(req.params.id, req.body);

    if (!ticket) {
      return res.status(404).json({
        success: false,
        error: 'Ticket not found'
      });
    }

    // Emit real-time update
    if (global.io) {
      global.io.emit('ticket_updated', { ticket });
    }

    res.json({
      success: true,
      data: ticket
    });
  } catch (error) {
    console.error('Error updating ticket:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// DELETE /api/tickets/:id - Cancel ticket
router.delete('/:id', async (req, res) => {
  try {
    const result = await deleteTicketRequest(req.params.id);

    if (!result) {
      return res.status(404).json({
        success: false,
        error: 'Ticket not found'
      });
    }

    // Emit real-time update
    if (global.io) {
      global.io.emit('ticket_cancelled', { ticketId: req.params.id });
    }

    res.json({
      success: true,
      message: 'Ticket cancelled successfully'
    });
  } catch (error) {
    console.error('Error deleting ticket:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// POST /api/tickets/:id/assign-pnr - Assign PNR to ticket
router.post('/:id/assign-pnr', async (req, res) => {
  try {
    const { pnr } = req.body;

    if (!pnr) {
      return res.status(400).json({
        success: false,
        error: 'PNR is required'
      });
    }

    // Validate PNR format (6 alphanumeric)
    if (!/^[A-Z0-9]{6}$/.test(pnr.toUpperCase())) {
      return res.status(400).json({
        success: false,
        error: 'Invalid PNR format (must be 6 alphanumeric characters)'
      });
    }

    const ticket = await assignPnrToTicket(req.params.id, pnr.toUpperCase());

    if (!ticket) {
      return res.status(404).json({
        success: false,
        error: 'Ticket not found'
      });
    }

    // Emit real-time update
    if (global.io) {
      global.io.emit('pnr_assigned', { ticket });
    }

    res.json({
      success: true,
      data: ticket,
      message: 'PNR assigned successfully'
    });
  } catch (error) {
    console.error('Error assigning PNR:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// POST /api/tickets/:ticketId/passengers - Add passenger to ticket
router.post('/:ticketId/passengers', async (req, res) => {
  try {
    const { ticketId } = req.params;
    const passengerData = req.body;

    // Validate passenger data
    if (!passengerData.firstName || !passengerData.lastName) {
      return res.status(400).json({
        success: false,
        error: 'Passenger firstName and lastName are required'
      });
    }

    // Check if passenger exists or create new
    let passengerId;
    if (passengerData.id) {
      // Use existing passenger
      const existingPassenger = await getPassenger(passengerData.id);
      if (!existingPassenger) {
        return res.status(404).json({
          success: false,
          error: 'Passenger not found'
        });
      }
      passengerId = passengerData.id;
    } else {
      // Create new passenger
      const newPassenger = await createPassenger(passengerData);
      passengerId = newPassenger.id;
    }

    // Link passenger to ticket
    await addPassengerToTicket(ticketId, passengerId);

    // Get updated ticket with passengers
    const ticket = await getTicketWithPassengers(ticketId);

    // Emit real-time update
    if (global.io) {
      global.io.emit('passenger_added', { ticket });
    }

    res.status(201).json({
      success: true,
      data: ticket,
      message: 'Passenger added successfully'
    });
  } catch (error) {
    console.error('Error adding passenger:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// DELETE /api/tickets/:ticketId/passengers/:passengerId - Remove passenger from ticket
router.delete('/:ticketId/passengers/:passengerId', async (req, res) => {
  try {
    const { ticketId, passengerId } = req.params;

    await removePassengerFromTicket(ticketId, passengerId);

    // Get updated ticket
    const ticket = await getTicketWithPassengers(ticketId);

    // Emit real-time update
    if (global.io) {
      global.io.emit('passenger_removed', { ticket });
    }

    res.json({
      success: true,
      data: ticket,
      message: 'Passenger removed successfully'
    });
  } catch (error) {
    console.error('Error removing passenger:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

module.exports = router;
