const express = require('express');
const router = express.Router();

// Import database functions
const {
  getDashboardStats,
  getAccountingStats,
  listCustomers,
  listTicketRequests,
  listCheckinTickets,
  listAccommodationRequests,
  listVisaRequests,
  listTransferRequests,
  listInvoices,
  listRevenueRecords
} = require('../services/database');

// Cache configuration
const CACHE_TTL = 5 * 60 * 1000; // 5 minutes in milliseconds
const cache = new Map();

// Cache middleware
function cacheMiddleware(key) {
  return (req, res, next) => {
    const cacheKey = `${key}-${JSON.stringify(req.query)}`;
    const cached = cache.get(cacheKey);

    if (cached && Date.now() - cached.timestamp < CACHE_TTL) {
      return res.json(cached.data);
    }

    // Store original json method
    const originalJson = res.json.bind(res);

    // Override json method to cache response
    res.json = (data) => {
      if (res.statusCode === 200) {
        cache.set(cacheKey, {
          data,
          timestamp: Date.now()
        });
      }
      return originalJson(data);
    };

    next();
  };
}

// Utility functions
function parseDate(dateString) {
  if (!dateString) return null;
  const date = new Date(dateString);
  return isNaN(date.getTime()) ? null : date;
}

function formatCurrency(amount, currency = 'TRY') {
  if (amount === null || amount === undefined) return '0.00';
  return Number(amount).toFixed(2);
}

function calculatePercentage(value, total) {
  if (!total || total === 0) return 0;
  return ((value / total) * 100).toFixed(2);
}

function getDateFilter(period, startDate, endDate) {
  const now = new Date();
  let start, end;

  switch (period) {
    case 'today':
      start = new Date(now.setHours(0, 0, 0, 0));
      end = new Date(now.setHours(23, 59, 59, 999));
      break;
    case 'week':
      start = new Date(now.setDate(now.getDate() - now.getDay()));
      start.setHours(0, 0, 0, 0);
      end = new Date();
      break;
    case 'month':
      start = new Date(now.getFullYear(), now.getMonth(), 1);
      end = new Date();
      break;
    case 'year':
      start = new Date(now.getFullYear(), 0, 1);
      end = new Date();
      break;
    case 'custom':
      start = parseDate(startDate);
      end = parseDate(endDate);
      if (!start || !end) {
        throw new Error('Invalid custom date range');
      }
      break;
    default:
      start = new Date(now.getFullYear(), now.getMonth(), 1);
      end = new Date();
  }

  return { start, end };
}

// ============================================
// DASHBOARD OVERVIEW STATISTICS
// ============================================

/**
 * GET /api/reports/dashboard
 * Dashboard overview statistics
 *
 * Response:
 * - todayCheckIns: Number of check-ins completed today
 * - todayScheduled: Number of scheduled check-ins today
 * - successRate: Check-in success rate percentage
 * - totalTicketRequests: Total ticket requests
 * - pendingRequests: Pending ticket requests
 * - ticketedToday: Tickets issued today
 * - upcomingFlights: Upcoming flights count
 * - failedCheckIns: Failed check-ins today
 * - customers: Customer statistics
 * - revenue: Revenue statistics
 */
router.get('/dashboard', cacheMiddleware('dashboard'), async (req, res) => {
  try {
    // Get base dashboard stats from database
    const baseStats = await getDashboardStats();

    // Get customer statistics
    const allCustomers = await listCustomers({}, { limit: 10000, offset: 0 });
    const activeCustomers = allCustomers.customers.filter(c => c.isActive);
    const vipCustomers = allCustomers.customers.filter(c => c.isVip);

    // Get revenue statistics
    const todayRevenue = await getAccountingStats('today');
    const weekRevenue = await getAccountingStats('week');
    const monthRevenue = await getAccountingStats('month');

    // Build comprehensive dashboard response
    const dashboardData = {
      todayCheckIns: baseStats.todayCheckins || 0,
      todayScheduled: baseStats.todayScheduled || 0,
      successRate: Number(baseStats.successRate || 0).toFixed(2),
      totalTicketRequests: baseStats.totalTicketRequests || 0,
      pendingRequests: baseStats.pendingRequests || 0,
      ticketedToday: baseStats.ticketedToday || 0,
      upcomingFlights: baseStats.upcomingFlights || 0,
      failedCheckIns: baseStats.failedCheckins || 0,
      customers: {
        total: allCustomers.total,
        active: activeCustomers.length,
        vip: vipCustomers.length
      },
      revenue: {
        today: Number(todayRevenue.totalRevenue || 0),
        thisWeek: Number(weekRevenue.totalRevenue || 0),
        thisMonth: Number(monthRevenue.totalRevenue || 0)
      }
    };

    res.json({
      success: true,
      data: dashboardData
    });
  } catch (error) {
    console.error('Error fetching dashboard stats:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// ============================================
// ACCOUNTING STATISTICS
// ============================================

/**
 * GET /api/reports/accounting
 * Accounting and revenue statistics
 *
 * Query params:
 * - period: 'today' | 'week' | 'month' | 'year' | 'custom' (default: 'month')
 * - startDate: Custom period start (ISO date)
 * - endDate: Custom period end (ISO date)
 *
 * Response:
 * - totalRevenue: Total revenue for period
 * - netRevenue: Net revenue after costs
 * - pendingPayments: Unpaid invoice amounts
 * - completedTransactions: Transaction count
 * - ticketRevenue: Revenue from tickets
 * - accommodationRevenue: Revenue from accommodations
 * - visaRevenue: Revenue from visa services
 * - transferRevenue: Revenue from transfers
 * - revenueByMonth: Monthly breakdown (if applicable)
 */
router.get('/accounting', cacheMiddleware('accounting'), async (req, res) => {
  try {
    const { period = 'month', startDate, endDate } = req.query;

    // Get accounting stats
    const stats = await getAccountingStats(period);

    // Get revenue records for time-series breakdown
    const revenueRecords = await listRevenueRecords(
      { status: 'completed' },
      { limit: 10000, offset: 0 }
    );

    // Calculate revenue by month for the last 3 months
    const now = new Date();
    const revenueByMonth = [];
    for (let i = 2; i >= 0; i--) {
      const monthDate = new Date(now.getFullYear(), now.getMonth() - i, 1);
      const monthStr = monthDate.toISOString().substring(0, 7); // YYYY-MM format
      const monthEnd = new Date(monthDate.getFullYear(), monthDate.getMonth() + 1, 0);

      const monthRevenue = revenueRecords.records
        .filter(r => {
          const serviceDate = new Date(r.serviceDate);
          return serviceDate >= monthDate && serviceDate <= monthEnd;
        })
        .reduce((sum, r) => sum + (r.amount || 0), 0);

      revenueByMonth.push({
        month: monthStr,
        revenue: Number(monthRevenue).toFixed(2)
      });
    }

    res.json({
      success: true,
      data: {
        totalRevenue: Number(stats.totalRevenue || 0),
        netRevenue: Number(stats.netRevenue || 0),
        pendingPayments: Number(stats.pendingPayments || 0),
        completedTransactions: stats.completedTransactions || 0,
        ticketRevenue: Number(stats.ticketRevenue || 0),
        accommodationRevenue: Number(stats.accommodationRevenue || 0),
        visaRevenue: Number(stats.visaRevenue || 0),
        transferRevenue: Number(stats.transferRevenue || 0),
        revenueByMonth
      }
    });
  } catch (error) {
    console.error('Error fetching accounting stats:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// ============================================
// CUSTOMER ANALYTICS
// ============================================

/**
 * GET /api/reports/customers
 * Customer analytics and statistics
 *
 * Query params:
 * - period: 'today' | 'week' | 'month' | 'year' | 'custom'
 * - startDate: Custom period start
 * - endDate: Custom period end
 * - type: 'individual' | 'corporate' | 'agency'
 *
 * Response:
 * - totalCustomers: Total customer count
 * - activeCustomers: Active customers
 * - vipCustomers: VIP customers
 * - byType: Breakdown by customer type
 * - newCustomers: New customers in period
 * - topCustomers: Top customers by revenue
 */
router.get('/customers', cacheMiddleware('customers'), async (req, res) => {
  try {
    const { period = 'month', startDate, endDate, type } = req.query;
    const dateFilter = getDateFilter(period, startDate, endDate);

    // Get all customers
    const filters = type ? { type } : {};
    const allCustomers = await listCustomers(filters, { limit: 10000, offset: 0 });

    // Calculate statistics
    const activeCustomers = allCustomers.customers.filter(c => c.isActive);
    const vipCustomers = allCustomers.customers.filter(c => c.isVip);

    // New customers in period
    const newCustomers = allCustomers.customers.filter(c => {
      const createdDate = new Date(c.createdAt);
      return createdDate >= dateFilter.start && createdDate <= dateFilter.end;
    });

    // Breakdown by type
    const byType = {
      individual: allCustomers.customers.filter(c => c.type === 'individual').length,
      corporate: allCustomers.customers.filter(c => c.type === 'corporate').length,
      agency: allCustomers.customers.filter(c => c.type === 'agency').length
    };

    // Get revenue records to calculate top customers
    const revenueRecords = await listRevenueRecords({}, { limit: 10000, offset: 0 });
    const customerRevenue = {};

    revenueRecords.records.forEach(record => {
      if (record.customerId) {
        if (!customerRevenue[record.customerId]) {
          customerRevenue[record.customerId] = 0;
        }
        customerRevenue[record.customerId] += Number(record.amount || 0);
      }
    });

    // Get top 10 customers by revenue
    const topCustomers = Object.entries(customerRevenue)
      .sort((a, b) => b[1] - a[1])
      .slice(0, 10)
      .map(([customerId, revenue]) => {
        const customer = allCustomers.customers.find(c => c.id === customerId);
        return {
          customerId,
          customerName: customer ?
            (customer.companyName || `${customer.firstName} ${customer.lastName}`) :
            'Unknown',
          revenue: Number(revenue).toFixed(2)
        };
      });

    res.json({
      success: true,
      data: {
        totalCustomers: allCustomers.total,
        activeCustomers: activeCustomers.length,
        vipCustomers: vipCustomers.length,
        byType,
        newCustomers: newCustomers.length,
        topCustomers
      }
    });
  } catch (error) {
    console.error('Error fetching customer analytics:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// ============================================
// TICKET REPORTS
// ============================================

/**
 * GET /api/reports/tickets
 * Ticket request reports and statistics
 *
 * Query params:
 * - period: 'today' | 'week' | 'month' | 'year' | 'custom'
 * - startDate: Custom period start
 * - endDate: Custom period end
 * - status: Ticket status filter
 *
 * Response:
 * - totalRequests: Total ticket requests
 * - byStatus: Breakdown by status
 * - completionRate: Percentage of completed requests
 * - averageProcessingTime: Average time to process
 * - requestsByDay: Daily breakdown
 */
router.get('/tickets', cacheMiddleware('tickets'), async (req, res) => {
  try {
    const { period = 'month', startDate, endDate, status } = req.query;
    const dateFilter = getDateFilter(period, startDate, endDate);

    // Get ticket requests
    const filters = status ? { status } : {};
    const tickets = await listTicketRequests(filters, { limit: 10000, offset: 0 });

    // Filter by date
    const filteredTickets = tickets.requests.filter(t => {
      const createdDate = new Date(t.createdAt);
      return createdDate >= dateFilter.start && createdDate <= dateFilter.end;
    });

    // Calculate statistics by status
    const byStatus = {
      pending: filteredTickets.filter(t => t.status === 'pending').length,
      confirmed: filteredTickets.filter(t => t.status === 'confirmed').length,
      ticketed: filteredTickets.filter(t => t.status === 'ticketed').length,
      cancelled: filteredTickets.filter(t => t.status === 'cancelled').length
    };

    // Calculate completion rate
    const completedTickets = byStatus.ticketed;
    const completionRate = calculatePercentage(completedTickets, filteredTickets.length);

    // Calculate average processing time
    const completedWithTime = filteredTickets.filter(t =>
      t.status === 'ticketed' && t.createdAt && t.updatedAt
    );
    const avgProcessingTime = completedWithTime.length > 0
      ? completedWithTime.reduce((sum, t) => {
          const created = new Date(t.createdAt);
          const updated = new Date(t.updatedAt);
          return sum + (updated - created);
        }, 0) / completedWithTime.length
      : 0;

    // Calculate average in hours
    const avgHours = (avgProcessingTime / (1000 * 60 * 60)).toFixed(2);

    // Requests by day
    const requestsByDay = {};
    filteredTickets.forEach(t => {
      const day = new Date(t.createdAt).toISOString().split('T')[0];
      requestsByDay[day] = (requestsByDay[day] || 0) + 1;
    });

    const requestsByDayArray = Object.entries(requestsByDay)
      .map(([date, count]) => ({ date, count }))
      .sort((a, b) => a.date.localeCompare(b.date));

    res.json({
      success: true,
      data: {
        totalRequests: filteredTickets.length,
        byStatus,
        completionRate: Number(completionRate),
        averageProcessingTimeHours: Number(avgHours),
        requestsByDay: requestsByDayArray
      }
    });
  } catch (error) {
    console.error('Error fetching ticket reports:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// ============================================
// CHECK-IN REPORTS
// ============================================

/**
 * GET /api/reports/checkins
 * Check-in reports and statistics
 *
 * Query params:
 * - period: 'today' | 'week' | 'month' | 'year' | 'custom'
 * - startDate: Custom period start
 * - endDate: Custom period end
 * - status: Check-in status filter
 *
 * Response:
 * - totalCheckins: Total check-in attempts
 * - byStatus: Breakdown by status
 * - successRate: Check-in success rate
 * - byAirline: Breakdown by airline
 * - upcomingCheckins: Scheduled check-ins
 */
router.get('/checkins', cacheMiddleware('checkins'), async (req, res) => {
  try {
    const { period = 'month', startDate, endDate, status } = req.query;
    const dateFilter = getDateFilter(period, startDate, endDate);

    // Get check-in tickets
    const filters = status ? { checkinStatus: status } : {};
    const checkins = await listCheckinTickets(filters, { limit: 10000, offset: 0 });

    // Filter by date
    const filteredCheckins = checkins.tickets.filter(t => {
      const departureDate = new Date(t.departureDate);
      return departureDate >= dateFilter.start && departureDate <= dateFilter.end;
    });

    // Calculate statistics by status
    const byStatus = {
      waiting: filteredCheckins.filter(t => t.checkinStatus === 'waiting').length,
      ready: filteredCheckins.filter(t => t.checkinStatus === 'ready').length,
      processing: filteredCheckins.filter(t => t.checkinStatus === 'processing').length,
      completed: filteredCheckins.filter(t => t.checkinStatus === 'completed').length,
      failed: filteredCheckins.filter(t => t.checkinStatus === 'failed').length
    };

    // Calculate success rate
    const totalAttempts = byStatus.completed + byStatus.failed;
    const successRate = totalAttempts > 0
      ? calculatePercentage(byStatus.completed, totalAttempts)
      : 0;

    // Breakdown by airline
    const byAirline = {};
    filteredCheckins.forEach(t => {
      const airline = t.airline || 'Unknown';
      byAirline[airline] = (byAirline[airline] || 0) + 1;
    });

    const byAirlineArray = Object.entries(byAirline)
      .map(([airline, count]) => ({ airline, count }))
      .sort((a, b) => b.count - a.count);

    // Get upcoming check-ins (next 7 days)
    const now = new Date();
    const nextWeek = new Date(now.getTime() + 7 * 24 * 60 * 60 * 1000);
    const upcomingCheckins = checkins.tickets.filter(t => {
      const departureDate = new Date(t.departureDate);
      return departureDate > now && departureDate <= nextWeek &&
        ['waiting', 'ready'].includes(t.checkinStatus);
    }).length;

    res.json({
      success: true,
      data: {
        totalCheckins: filteredCheckins.length,
        byStatus,
        successRate: Number(successRate),
        byAirline: byAirlineArray,
        upcomingCheckins
      }
    });
  } catch (error) {
    console.error('Error fetching check-in reports:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// ============================================
// SERVICES SUMMARY (All services)
// ============================================

/**
 * GET /api/reports/services
 * Summary report for all services
 *
 * Query params:
 * - period: 'today' | 'week' | 'month' | 'year' | 'custom'
 * - startDate: Custom period start
 * - endDate: Custom period end
 *
 * Response:
 * - tickets: Ticket service statistics
 * - accommodations: Accommodation statistics
 * - visas: Visa service statistics
 * - transfers: Transfer service statistics
 */
router.get('/services', cacheMiddleware('services'), async (req, res) => {
  try {
    const { period = 'month', startDate, endDate } = req.query;
    const dateFilter = getDateFilter(period, startDate, endDate);

    // Get all service data
    const [tickets, accommodations, visas, transfers] = await Promise.all([
      listTicketRequests({}, { limit: 10000, offset: 0 }),
      listAccommodationRequests({}, { limit: 10000, offset: 0 }),
      listVisaRequests({}, { limit: 10000, offset: 0 }),
      listTransferRequests({}, { limit: 10000, offset: 0 })
    ]);

    // Filter and calculate ticket statistics
    const filteredTickets = tickets.requests.filter(t => {
      const createdDate = new Date(t.createdAt);
      return createdDate >= dateFilter.start && createdDate <= dateFilter.end;
    });

    const ticketStats = {
      total: filteredTickets.length,
      pending: filteredTickets.filter(t => t.status === 'pending').length,
      completed: filteredTickets.filter(t => t.status === 'ticketed').length,
      cancelled: filteredTickets.filter(t => t.status === 'cancelled').length
    };

    // Filter and calculate accommodation statistics
    const filteredAccommodations = accommodations.requests.filter(a => {
      const createdDate = new Date(a.createdAt);
      return createdDate >= dateFilter.start && createdDate <= dateFilter.end;
    });

    const accommodationStats = {
      total: filteredAccommodations.length,
      pending: filteredAccommodations.filter(a => a.status === 'pending').length,
      confirmed: filteredAccommodations.filter(a => a.status === 'confirmed').length,
      completed: filteredAccommodations.filter(a => a.status === 'completed').length,
      cancelled: filteredAccommodations.filter(a => a.status === 'cancelled').length
    };

    // Filter and calculate visa statistics
    const filteredVisas = visas.requests.filter(v => {
      const createdDate = new Date(v.createdAt);
      return createdDate >= dateFilter.start && createdDate <= dateFilter.end;
    });

    const visaStats = {
      total: filteredVisas.length,
      pending: filteredVisas.filter(v => v.status === 'pending').length,
      approved: filteredVisas.filter(v => v.status === 'approved').length,
      rejected: filteredVisas.filter(v => v.status === 'rejected').length,
      processing: filteredVisas.filter(v => v.status === 'processing').length
    };

    // Filter and calculate transfer statistics
    const filteredTransfers = transfers.requests.filter(t => {
      const createdDate = new Date(t.createdAt);
      return createdDate >= dateFilter.start && createdDate <= dateFilter.end;
    });

    const transferStats = {
      total: filteredTransfers.length,
      pending: filteredTransfers.filter(t => t.status === 'pending').length,
      confirmed: filteredTransfers.filter(t => t.status === 'confirmed').length,
      completed: filteredTransfers.filter(t => t.status === 'completed').length,
      cancelled: filteredTransfers.filter(t => t.status === 'cancelled').length
    };

    res.json({
      success: true,
      data: {
        tickets: ticketStats,
        accommodations: accommodationStats,
        visas: visaStats,
        transfers: transferStats
      }
    });
  } catch (error) {
    console.error('Error fetching services summary:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// ============================================
// REVENUE ANALYTICS
// ============================================

/**
 * GET /api/reports/revenue
 * Revenue analytics and financial reports
 *
 * Query params:
 * - period: 'today' | 'week' | 'month' | 'year' | 'custom'
 * - startDate: Custom period start
 * - endDate: Custom period end
 * - groupBy: 'day' | 'week' | 'month' | 'year'
 * - serviceType: 'ticket' | 'accommodation' | 'visa' | 'transfer'
 *
 * Response:
 * - totalRevenue: Total revenue for period
 * - byServiceType: Revenue breakdown by service
 * - revenueTimeSeries: Time-series revenue data
 * - topRevenueSources: Top revenue generating services
 * - averageTransactionValue: Average transaction amount
 */
router.get('/revenue', cacheMiddleware('revenue'), async (req, res) => {
  try {
    const {
      period = 'month',
      startDate,
      endDate,
      groupBy = 'day',
      serviceType
    } = req.query;

    const dateFilter = getDateFilter(period, startDate, endDate);

    // Get revenue records
    const filters = serviceType ? { type: serviceType } : {};
    const revenue = await listRevenueRecords(filters, { limit: 10000, offset: 0 });

    // Filter by date
    const filteredRevenue = revenue.records.filter(r => {
      const serviceDate = new Date(r.serviceDate);
      return serviceDate >= dateFilter.start && serviceDate <= dateFilter.end;
    });

    // Calculate total revenue
    const totalRevenue = filteredRevenue.reduce((sum, r) => sum + (r.amount || 0), 0);

    // Breakdown by service type
    const byServiceType = {
      ticket: 0,
      accommodation: 0,
      visa: 0,
      transfer: 0
    };

    filteredRevenue.forEach(r => {
      if (r.type && byServiceType[r.type] !== undefined) {
        byServiceType[r.type] += Number(r.amount || 0);
      }
    });

    // Format service type breakdown
    const byServiceTypeArray = Object.entries(byServiceType).map(([type, amount]) => ({
      serviceType: type,
      revenue: Number(amount).toFixed(2),
      percentage: calculatePercentage(amount, totalRevenue)
    }));

    // Calculate average transaction value
    const averageTransactionValue = filteredRevenue.length > 0
      ? totalRevenue / filteredRevenue.length
      : 0;

    // Generate time series data based on groupBy parameter
    const revenueTimeSeries = [];
    const groupedData = {};

    filteredRevenue.forEach(r => {
      const date = new Date(r.serviceDate);
      let key;

      switch (groupBy) {
        case 'day':
          key = date.toISOString().split('T')[0];
          break;
        case 'week':
          const weekStart = new Date(date);
          weekStart.setDate(date.getDate() - date.getDay());
          key = weekStart.toISOString().split('T')[0];
          break;
        case 'month':
          key = date.toISOString().substring(0, 7); // YYYY-MM
          break;
        case 'year':
          key = date.getFullYear().toString();
          break;
        default:
          key = date.toISOString().split('T')[0];
      }

      if (!groupedData[key]) {
        groupedData[key] = 0;
      }
      groupedData[key] += Number(r.amount || 0);
    });

    // Convert to array and sort
    Object.entries(groupedData).forEach(([period, amount]) => {
      revenueTimeSeries.push({
        period,
        revenue: Number(amount).toFixed(2)
      });
    });

    revenueTimeSeries.sort((a, b) => a.period.localeCompare(b.period));

    // Get top revenue sources (individual records)
    const topRevenueSources = filteredRevenue
      .sort((a, b) => (b.amount || 0) - (a.amount || 0))
      .slice(0, 10)
      .map(r => ({
        id: r.id,
        type: r.type,
        description: r.description || `${r.type} service`,
        amount: Number(r.amount || 0).toFixed(2),
        date: r.serviceDate
      }));

    res.json({
      success: true,
      data: {
        totalRevenue: Number(totalRevenue).toFixed(2),
        byServiceType: byServiceTypeArray,
        revenueTimeSeries,
        topRevenueSources,
        averageTransactionValue: Number(averageTransactionValue).toFixed(2),
        transactionCount: filteredRevenue.length
      }
    });
  } catch (error) {
    console.error('Error fetching revenue analytics:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// ============================================
// CACHE MANAGEMENT
// ============================================

/**
 * DELETE /api/reports/cache
 * Clear all report caches
 */
router.delete('/cache', (req, res) => {
  try {
    cache.clear();
    res.json({
      success: true,
      message: 'All report caches cleared'
    });
  } catch (error) {
    console.error('Error clearing cache:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

module.exports = router;
