const express = require('express');
const router = express.Router();
const AmadeusService = require('../services/AmadeusService');

// Initialize Amadeus service
const amadeusService = new AmadeusService();

// In-memory storage for demo (replace with database)
let flightSearches = [];
let bookings = [];

// Helper functions
const generateId = () => Date.now().toString() + Math.random().toString(36).substr(2, 9);

// GET /api/flights/test - Test Amadeus API connection
router.get('/test', async (req, res) => {
  try {
    const isConnected = await amadeusService.testConnection();

    res.json({
      success: isConnected,
      message: isConnected ? 'Amadeus API connection successful' : 'Amadeus API connection failed',
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: 'Connection test failed',
      message: error.message
    });
  }
});

// POST /api/flights/search - Search flights
router.post('/search', async (req, res) => {
  const {
    origin,
    destination,
    departureDate,
    returnDate,
    passengers = { adults: 1, children: 0, infants: 0 },
    travelClass = 'ECONOMY',
    nonStop = false,
    currency = 'TRY'
  } = req.body;

  // Validation
  if (!origin || !destination || !departureDate) {
    return res.status(400).json({
      error: 'Origin, destination and departure date are required'
    });
  }

  const searchId = generateId();

  try {
    console.log(`🔍 Flight search started: ${origin} → ${destination} on ${departureDate}`);

    const searchParams = {
      originLocationCode: origin.toUpperCase(),
      destinationLocationCode: destination.toUpperCase(),
      departureDate,
      returnDate,
      adults: passengers.adults,
      children: passengers.children || 0,
      infants: passengers.infants || 0,
      travelClass,
      nonStop,
      currencyCode: currency,
      max: 50 // Limit results for better performance
    };

    const searchResult = await amadeusService.searchFlights(searchParams);

    if (searchResult.success) {
      // Format results for frontend
      const formattedOffers = searchResult.data.map(offer =>
        amadeusService.formatFlightOffer(offer)
      );

      // Store search for reference
      const searchRecord = {
        id: searchId,
        searchParams,
        results: {
          offers: formattedOffers,
          count: formattedOffers.length,
          currency: currency
        },
        timestamp: new Date(),
        meta: searchResult.meta
      };

      flightSearches.push(searchRecord);

      console.log(`✅ Flight search completed: ${formattedOffers.length} offers found`);

      res.json({
        success: true,
        searchId,
        offers: formattedOffers,
        count: formattedOffers.length,
        currency,
        searchParams: {
          origin,
          destination,
          departureDate,
          returnDate,
          passengers,
          travelClass
        },
        meta: searchResult.meta
      });

    } else {
      console.log(`❌ Flight search failed: ${searchResult.message}`);

      res.status(400).json({
        success: false,
        error: searchResult.error,
        message: searchResult.message,
        searchId
      });
    }

  } catch (error) {
    console.error('Flight search error:', error);

    res.status(500).json({
      success: false,
      error: 'Flight search failed',
      message: error.message,
      searchId
    });
  }
});

// POST /api/flights/price - Get pricing for specific flight offer
router.post('/price', async (req, res) => {
  const { flightOffer } = req.body;

  if (!flightOffer) {
    return res.status(400).json({
      error: 'Flight offer is required'
    });
  }

  try {
    console.log(`💰 Getting pricing for flight offer: ${flightOffer.id}`);

    const pricingResult = await amadeusService.getFlightPricing(flightOffer);

    if (pricingResult.success) {
      console.log(`✅ Pricing retrieved successfully`);

      res.json({
        success: true,
        pricing: pricingResult.data,
        included: pricingResult.included
      });

    } else {
      console.log(`❌ Pricing failed: ${pricingResult.message}`);

      res.status(400).json({
        success: false,
        error: pricingResult.error,
        message: pricingResult.message
      });
    }

  } catch (error) {
    console.error('Flight pricing error:', error);

    res.status(500).json({
      success: false,
      error: 'Flight pricing failed',
      message: error.message
    });
  }
});

// POST /api/flights/book - Create flight booking
router.post('/book', async (req, res) => {
  const {
    flightOffers,
    travelers,
    contacts,
    remarks
  } = req.body;

  // Validation
  if (!flightOffers || !travelers || !contacts) {
    return res.status(400).json({
      error: 'Flight offers, travelers and contacts are required'
    });
  }

  if (!Array.isArray(travelers) || travelers.length === 0) {
    return res.status(400).json({
      error: 'At least one traveler is required'
    });
  }

  const bookingId = generateId();

  try {
    console.log(`🎫 Creating flight booking for ${travelers.length} traveler(s)`);

    const bookingData = {
      flightOffers: Array.isArray(flightOffers) ? flightOffers : [flightOffers],
      travelers,
      contacts,
      remarks: remarks || {}
    };

    const bookingResult = await amadeusService.createBooking(bookingData);

    if (bookingResult.success) {
      // Store booking locally
      const bookingRecord = {
        id: bookingId,
        amadeusBookingId: bookingResult.booking.id,
        pnr: bookingResult.pnr,
        status: 'confirmed',
        flightOffers: bookingData.flightOffers,
        travelers,
        contacts,
        totalPrice: bookingResult.booking.flightOffers?.[0]?.price,
        ticketingAgreement: bookingResult.ticketingAgreement,
        createdAt: new Date(),
        updatedAt: new Date()
      };

      bookings.push(bookingRecord);

      console.log(`✅ Flight booking created successfully: PNR ${bookingResult.pnr}`);

      res.status(201).json({
        success: true,
        bookingId,
        pnr: bookingResult.pnr,
        booking: bookingRecord,
        amadeusBooking: bookingResult.booking
      });

    } else {
      console.log(`❌ Flight booking failed: ${bookingResult.message}`);

      res.status(400).json({
        success: false,
        error: bookingResult.error,
        message: bookingResult.message,
        errors: bookingResult.errors,
        bookingId
      });
    }

  } catch (error) {
    console.error('Flight booking error:', error);

    res.status(500).json({
      success: false,
      error: 'Flight booking failed',
      message: error.message,
      bookingId
    });
  }
});

// GET /api/flights/bookings - Get all bookings
router.get('/bookings', (req, res) => {
  const { status, pnr } = req.query;
  let filteredBookings = [...bookings];

  if (status) {
    filteredBookings = filteredBookings.filter(b => b.status === status);
  }

  if (pnr) {
    filteredBookings = filteredBookings.filter(b =>
      b.pnr && b.pnr.toLowerCase().includes(pnr.toLowerCase())
    );
  }

  res.json({
    success: true,
    bookings: filteredBookings,
    total: filteredBookings.length
  });
});

// GET /api/flights/bookings/:id - Get specific booking
router.get('/bookings/:id', (req, res) => {
  const { id } = req.params;

  const booking = bookings.find(b => b.id === id || b.pnr === id.toUpperCase());

  if (!booking) {
    return res.status(404).json({
      error: 'Booking not found'
    });
  }

  res.json({
    success: true,
    booking
  });
});

// GET /api/flights/searches - Get search history
router.get('/searches', (req, res) => {
  const { limit = 20 } = req.query;

  const recentSearches = flightSearches
    .sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp))
    .slice(0, parseInt(limit));

  res.json({
    success: true,
    searches: recentSearches,
    total: recentSearches.length
  });
});

// GET /api/flights/searches/:id - Get specific search results
router.get('/searches/:id', (req, res) => {
  const { id } = req.params;

  const search = flightSearches.find(s => s.id === id);

  if (!search) {
    return res.status(404).json({
      error: 'Search not found'
    });
  }

  res.json({
    success: true,
    search
  });
});

// POST /api/flights/status - Get flight status
router.post('/status', async (req, res) => {
  const { carrierCode, flightNumber, departureDate } = req.body;

  if (!carrierCode || !flightNumber || !departureDate) {
    return res.status(400).json({
      error: 'Carrier code, flight number and departure date are required'
    });
  }

  try {
    console.log(`📡 Getting flight status: ${carrierCode}${flightNumber} on ${departureDate}`);

    const statusResult = await amadeusService.getFlightStatus(
      carrierCode.toUpperCase(),
      flightNumber,
      departureDate
    );

    if (statusResult.success) {
      console.log(`✅ Flight status retrieved`);

      res.json({
        success: true,
        flight: statusResult.flight
      });

    } else {
      console.log(`❌ Flight status lookup failed: ${statusResult.message}`);

      res.status(404).json({
        success: false,
        error: statusResult.error,
        message: statusResult.message
      });
    }

  } catch (error) {
    console.error('Flight status error:', error);

    res.status(500).json({
      success: false,
      error: 'Flight status lookup failed',
      message: error.message
    });
  }
});

// GET /api/flights/airports/:keyword - Search airports
router.get('/airports/:keyword', async (req, res) => {
  const { keyword } = req.params;
  const { subType = 'AIRPORT' } = req.query;

  if (!keyword || keyword.length < 2) {
    return res.status(400).json({
      error: 'Keyword must be at least 2 characters'
    });
  }

  try {
    const airportResult = await amadeusService.getAirportInfo(keyword, subType);

    if (airportResult.success) {
      res.json({
        success: true,
        airports: airportResult.airports,
        count: airportResult.count
      });

    } else {
      res.status(400).json({
        success: false,
        error: airportResult.error,
        message: airportResult.message
      });
    }

  } catch (error) {
    console.error('Airport search error:', error);

    res.status(500).json({
      success: false,
      error: 'Airport search failed',
      message: error.message
    });
  }
});

// GET /api/flights/airlines/:codes - Get airline information
router.get('/airlines/:codes', async (req, res) => {
  const { codes } = req.params;

  if (!codes) {
    return res.status(400).json({
      error: 'Airline codes are required'
    });
  }

  try {
    const airlineResult = await amadeusService.getAirlineInfo(codes);

    if (airlineResult.success) {
      res.json({
        success: true,
        airlines: airlineResult.airlines,
        count: airlineResult.count
      });

    } else {
      res.status(400).json({
        success: false,
        error: airlineResult.error,
        message: airlineResult.message
      });
    }

  } catch (error) {
    console.error('Airline info error:', error);

    res.status(500).json({
      success: false,
      error: 'Airline info lookup failed',
      message: error.message
    });
  }
});

// GET /api/flights/checkin-links/:airlineCode - Get check-in links
router.get('/checkin-links/:airlineCode', async (req, res) => {
  const { airlineCode } = req.params;
  const { language = 'en-GB' } = req.query;

  try {
    const checkinResult = await amadeusService.getCheckinLinks(airlineCode, language);

    if (checkinResult.success) {
      res.json({
        success: true,
        checkinLinks: checkinResult.checkinLinks,
        count: checkinResult.count
      });

    } else {
      res.status(400).json({
        success: false,
        error: checkinResult.error,
        message: checkinResult.message
      });
    }

  } catch (error) {
    console.error('Checkin links error:', error);

    res.status(500).json({
      success: false,
      error: 'Checkin links lookup failed',
      message: error.message
    });
  }
});

// DELETE /api/flights/searches/:id - Delete search record
router.delete('/searches/:id', (req, res) => {
  const { id } = req.params;

  const searchIndex = flightSearches.findIndex(s => s.id === id);

  if (searchIndex === -1) {
    return res.status(404).json({
      error: 'Search not found'
    });
  }

  const deletedSearch = flightSearches.splice(searchIndex, 1)[0];

  res.json({
    success: true,
    message: 'Search deleted',
    search: deletedSearch
  });
});

// GET /api/flights/dashboard - Flight service dashboard stats
router.get('/dashboard', (req, res) => {
  const now = new Date();
  const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
  const weekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);

  // Today's searches
  const todaySearches = flightSearches.filter(s =>
    new Date(s.timestamp) >= today
  ).length;

  // This week's bookings
  const weekBookings = bookings.filter(b =>
    new Date(b.createdAt) >= weekAgo
  ).length;

  // Recent searches
  const recentSearchCount = flightSearches.filter(s =>
    new Date(s.timestamp) >= weekAgo
  ).length;

  // Confirmed bookings
  const confirmedBookings = bookings.filter(b =>
    b.status === 'confirmed'
  ).length;

  // Popular routes (from recent searches)
  const routes = {};
  flightSearches.forEach(search => {
    const route = `${search.searchParams.originLocationCode}-${search.searchParams.destinationLocationCode}`;
    routes[route] = (routes[route] || 0) + 1;
  });

  const popularRoutes = Object.entries(routes)
    .sort(([,a], [,b]) => b - a)
    .slice(0, 5)
    .map(([route, count]) => ({ route, count }));

  const stats = {
    todaySearches,
    weekBookings,
    recentSearchCount,
    confirmedBookings,
    totalSearches: flightSearches.length,
    totalBookings: bookings.length,
    popularRoutes
  };

  res.json({
    success: true,
    stats
  });
});

module.exports = router;