const express = require('express');
const router = express.Router();
const SeleniumService = require('../services/SeleniumService');
const { spawn } = require('child_process');
const path = require('path');
const fs = require('fs').promises;
const fsSync = require('fs');
const { saveLog, getLogs, createCheckinTicket, createPassenger, addPassengerToCheckin, updateCheckinTicket } = require('../services/database-mysql');

// Store active selenium instances
const activeServices = new Map();

// Python check-in function (THY or Pegasus)
function runPythonCheckin(data, progressCallback) {
  return new Promise((resolve, reject) => {
    try {
      const airline = data.airline || 'THY';
      console.log(`🛫 Check-in başlatılıyor - Havayolu: ${airline}, PNR: ${data.pnr}`);

      let pythonPath, pythonVenv, configPath, configDir;

      // Havayoluna göre script ve path seçimi
      if (airline === 'PC' || airline === 'PEGASUS') {
        // Pegasus için
        configDir = '/Users/sametburaksari/CascadeProjects/vTravel/vtravel-backend/main_scripts/pegasus';
        configPath = `${configDir}/advanced_config.json`;
        pythonPath = `${configDir}/pegasus_direct_checkin.py`;
        pythonVenv = `${configDir}/venv/bin/python`;
        progressCallback(60, 'Pegasus check-in script başlatılıyor...');
      } else if (airline === 'AJET' || airline === 'AJet') {
        // AJET için
        configDir = '/Users/sametburaksari/CascadeProjects/vTravel/vtravel-backend/main_scripts/ajet';
        pythonPath = `${configDir}/ajet_final.py`;
        pythonVenv = `${configDir}/../venv/bin/python`;
        progressCallback(60, 'AJET check-in script başlatılıyor...');
      } else {
        // THY için (default)
        configDir = '/Users/sametburaksari/CascadeProjects/vTravel/vtravel-backend/main_scripts/thy';
        configPath = `${configDir}/advanced_config.json`;
        pythonPath = `${configDir}/thy_hero_checkin.py`;
        pythonVenv = `${configDir}/venv/bin/python`;
        progressCallback(60, 'THY Hero check-in script başlatılıyor...');
      }

      const config = {
        bookings: [{
          pnr: data.pnr,
          passengers: [{
            lastname: data.lastName,
            phone: data.phone
          }]
        }],
        settings: {}
      };

      try {
        // Config dosyası sadece THY ve Pegasus için gerekli (AJET için değil)
        if (configPath) {
          // Directory kontrolü
          if (!fsSync.existsSync(configDir)) {
            fsSync.mkdirSync(configDir, { recursive: true });
          }
          fsSync.writeFileSync(configPath, JSON.stringify(config, null, 2));
        }

        let pythonArgs;
        if (airline === 'PC' || airline === 'PEGASUS') {
          // Pegasus için argument formatı
          pythonArgs = [pythonPath, data.pnr, data.lastName];

          // Koltuk seçimi parametresi ekle
          if (data.seatSelection) {
            pythonArgs.push('--seat-selection', data.seatSelection);
          }

          if (data.debug) pythonArgs.push('--debug');
          pythonArgs.push('--headless'); // Pegasus için headless mode - tarayıcı açmasın
        } else if (airline === 'AJET' || airline === 'AJet') {
          // AJET için argument formatı
          pythonArgs = [pythonPath, data.pnr, data.lastName];

          // Email ve phone parametrelerini ekle
          if (data.email) {
            pythonArgs.push('--email', data.email);
          }
          if (data.phone) {
            pythonArgs.push('--phone', data.phone);
          }

          // AJET scripti headless mode desteklemiyor, normal mode'da çalışacak

          if (data.debug) pythonArgs.push('--debug');
        } else {
          // THY için argument formatı
          pythonArgs = [
            pythonPath,
            '--pnr', data.pnr,
            '--lastname', data.lastName,
            '--phone', data.phone || '',
            '--headless' // THY için headless mode - tarayıcı açmasın
          ];
        }

        console.log(`🐍 Python command: ${pythonVenv} ${pythonArgs.join(' ')}`);

        const pythonProcess = spawn(pythonVenv, pythonArgs, {
          cwd: configDir,
          env: { ...process.env }
        });

          let output = '';
          let errorOutput = '';

          pythonProcess.stdout.on('data', (data) => {
            const msg = data.toString();
            output += msg;
            console.log('Python output:', msg);

            // THY Hero check-in patterns
            if (msg.includes('🚀 Chrome driver kuruluyor')) {
              progressCallback(65, 'THY - Tarayıcı başlatılıyor...');
            }
            if (msg.includes('✅ Chrome driver hazır')) {
              progressCallback(68, 'THY - Tarayıcı hazır...');
            }
            if (msg.includes('🌐 THY ana sayfasına gidiliyor')) {
              progressCallback(70, 'THY - Ana sayfaya bağlanılıyor...');
            }
            if (msg.includes('📍 Ana sayfa yüklendi')) {
              progressCallback(72, 'THY - Ana sayfa yüklendi...');
            }
            if (msg.includes('🎯 Hero bölümünde') || msg.includes('Hero section')) {
              progressCallback(75, 'THY - Hero bölümü aranıyor...');
            }
            if (msg.includes('✅ Hero check-in tab\'ına tıklandı') || msg.includes('Check-in tab\'a tıklandı')) {
              progressCallback(78, 'THY - Check-in sekmesine geçildi...');
            }
            if (msg.includes('📝 Hero check-in formu dolduruluyor') || msg.includes('PNR') || msg.includes('Lastname')) {
              progressCallback(80, 'THY - Rezervasyon bilgileri giriliyor...');
            }
            if (msg.includes('Check-in butonu tıklandı') || msg.includes('Yolcu sayfasına gidiliyor')) {
              progressCallback(83, 'THY - Check-in işlemi başlatılıyor...');
            }
            if (msg.includes('Yolcu seçimi') || msg.includes('seçildi') || msg.includes('passengers selected')) {
              progressCallback(85, 'THY - Yolcu seçimi yapılıyor...');
            }
            if (msg.includes('Bagaj') || msg.includes('bagaj onayı') || msg.includes('Devam butonu')) {
              progressCallback(87, 'THY - Bagaj sayfası işleniyor...');
            }
            if (msg.includes('🪑') || msg.includes('Koltuk') || msg.includes('seat')) {
              progressCallback(90, 'THY - Koltuk seçimi yapılıyor...');
            }
            if (msg.includes('✅ Otomatik ücretsiz koltuk seçimi başarılı')) {
              progressCallback(92, 'THY - Koltuk seçildi...');
            }
            if (msg.includes('Ek Hizmetler') || msg.includes('Extra Services')) {
              progressCallback(94, 'THY - Ek hizmetler sayfası...');
            }
            if (msg.includes('✅ Biniş kartı başarıyla indirildi')) {
              progressCallback(97, 'THY - Biniş kartı indirildi...');
            }
            if (msg.includes('✅ SMS ile biniş kartı başarıyla gönderildi')) {
              progressCallback(98, 'THY - SMS gönderildi...');
            }
            if (msg.includes('✅ Biniş kartı işlemleri tamamlandı') || msg.includes('SMS ve/veya biniş kartı')) {
              progressCallback(100, '✅ THY Check-in tamamlandı!');
            }

            // Pegasus check-in patterns
            // AJET check-in patterns
            if (msg.includes('AJET Check-in v3')) {
              progressCallback(60, 'AJET - Script başlatılıyor...');
            }
            if (msg.includes('PNR ve soyisim girildi') || msg.includes('✓ PNR ve soyisim girildi')) {
              progressCallback(68, 'AJET - Bilgiler girildi...');
            }
            if (msg.includes('Submit button clicked')) {
              progressCallback(70, 'AJET - PNR bilgileri gönderiliyor...');
            }
            if (msg.includes('After submit') && msg.includes('flightselection')) {
              progressCallback(75, 'AJET - Uçuş seçimi...');
            }
            if (msg.includes('Flight check-in button clicked')) {
              progressCallback(78, 'AJET - Check-in başlatılıyor...');
            }
            if (msg.includes('Passenger popup clicked')) {
              progressCallback(80, 'AJET - Yolcu onayı...');
            }
            if (msg.includes('After APIS') && msg.includes('seat-selection')) {
              progressCallback(83, 'AJET - APIS formu tamamlandı...');
            }
            if (msg.includes('[Seat] Processing')) {
              progressCallback(85, 'AJET - Koltuk seçimi...');
            }
            if (msg.includes('Seat popup Devam Et clicked')) {
              progressCallback(88, 'AJET - Koltuk onaylandı...');
            }
            if (msg.includes('[Baggage] Processing')) {
              progressCallback(90, 'AJET - Bagaj sayfası...');
            }
            if (msg.includes('Step 2: Checkbox clicked')) {
              progressCallback(92, 'AJET - Bagaj kuralları kabul edildi...');
            }
            if (msg.includes('After baggage') && msg.includes('summary')) {
              progressCallback(95, 'AJET - Özet sayfası...');
            }
            if (msg.includes('✓ Summary confirmed')) {
              progressCallback(95.5, 'AJET - Özet onaylandı...');
            }
            if (msg.includes('CHECK-IN TAMAMLANDI') || msg.includes('thankyou')) {
              progressCallback(96, '✅ AJET Check-in tamamlandı!');
            }
            if (msg.includes('⚠️  Check-in zaten yapılmış')) {
              progressCallback(75, 'AJET - Check-in zaten yapılmış, boarding pass işlemine geçiliyor...');
            }
            if (msg.includes('⚠️  Sayfa ilerlemedi')) {
              progressCallback(72, 'AJET - Sayfa yüklenemiyor, yeniden deneniyor...');
            }
            if (msg.includes('[Boarding Pass] Biniş kartı gönderme')) {
              progressCallback(97, 'AJET - Biniş kartı sayfasına gidiliyor...');
            }
            if (msg.includes('✓ Yolcu seçildi')) {
              progressCallback(97.5, 'AJET - Yolcu seçildi...');
            }
            if (msg.includes('✓ E-posta girildi')) {
              progressCallback(98, 'AJET - E-posta bilgisi girildi...');
            }
            if (msg.includes('[Email] E-posta gönderiliyor')) {
              progressCallback(98.2, 'AJET - E-posta gönderiliyor...');
            }
            if (msg.includes('✓ E-posta gönder butonuna tıklandı')) {
              progressCallback(98.5, 'AJET - E-posta gönderildi...');
            }
            if (msg.includes('✓ SMS/Telefon tabına geçildi')) {
              progressCallback(99, 'AJET - SMS sekmesine geçiliyor...');
            }
            if (msg.includes('✓ Telefon girildi')) {
              progressCallback(99.2, 'AJET - Telefon bilgisi girildi...');
            }
            if (msg.includes('[SMS] SMS gönderiliyor')) {
              progressCallback(99.4, 'AJET - SMS gönderiliyor...');
            }
            if (msg.includes('✓ SMS gönder butonuna tıklandı')) {
              progressCallback(99.7, 'AJET - SMS gönderildi...');
            }
            if (msg.includes('BİNİŞ KARTI GÖNDERME İŞLEMİ TAMAMLANDI')) {
              progressCallback(100, '✅ AJET Biniş kartları gönderildi!');
            }

            if (msg.includes('✅ Browser setup successful') || msg.includes('stealth mode')) {
              progressCallback(65, 'Pegasus - Tarayıcı başlatılıyor...');
            }
            if (msg.includes('🏠 First visiting main page') || msg.includes('Cloudflare')) {
              progressCallback(70, 'Pegasus - Cloudflare bypass...');
            }
            if (msg.includes('✅ Main page loaded') || msg.includes('Successfully accessed check-in page')) {
              progressCallback(73, 'Pegasus - Check-in sayfasına bağlanıldı');
            }
            if (msg.includes('🔍 Looking for CHECK IN') || msg.includes('BİNİŞ KARTLARI')) {
              progressCallback(75, 'Pegasus - Check-in butonu aranıyor...');
            }
            if (msg.includes('Found CHECK IN button') || msg.includes('Clicking CHECK-IN button')) {
              progressCallback(78, 'Pegasus - Check-in butonu tıklanıyor...');
            }
            if (msg.includes('👥 Looking for passenger') || msg.includes('Selected') && msg.includes('passengers')) {
              progressCallback(80, 'Pegasus - Yolcu seçiliyor...');
            }
            if (msg.includes('DEVAM button') || msg.includes('Clicking first DEVAM')) {
              progressCallback(83, 'Pegasus - DEVAM adımı...');
            }
            if (msg.includes('🪑 Reached seat selection') || msg.includes('BAGAJ')) {
              progressCallback(86, 'Pegasus - Koltuk sayfasında...');
            }
            if (msg.includes('pop-up') || msg.includes('DEVAM ET')) {
              progressCallback(88, 'Pegasus - Pop-up işleniyor...');
            }
            if (msg.includes('🧳 Looking for baggage') || msg.includes('Baggage')) {
              progressCallback(90, 'Pegasus - Bagaj sayfasında...');
            }
            if (msg.includes('ONAYLA VE CHECK-IN') || msg.includes('payment')) {
              progressCallback(93, 'Pegasus - Onay sayfasında...');
            }
            if (msg.includes('success') && msg.includes('URL')) {
              progressCallback(95, 'Pegasus - Success sayfasına ulaşıldı...');
            }
            if (msg.includes('BİNİŞ KARTLARI button') || msg.includes('Looking for email')) {
              progressCallback(97, 'Pegasus - Email gönderimi için hazırlanıyor...');
            }
            if (msg.includes('Email entered') || msg.includes('bilet@milaturizm.com')) {
              progressCallback(98, 'Pegasus - Email adresi girildi...');
            }
            if (msg.includes('E-Posta Gönderildi') || msg.includes('✅ CHECK-IN SUCCESSFUL')) {
              progressCallback(100, '✅ Pegasus check-in tamamlandı!');
            }

            // Genel hata pattern'leri
            if (msg.includes('❌') || msg.includes('ERROR') || msg.includes('hatası') || msg.includes('başarısız') || msg.includes('FAILED')) {
              progressCallback(50, '❌ Hata oluştu, lütfen logları kontrol edin...');
            }
          });

          pythonProcess.stderr.on('data', (data) => {
            const msg = data.toString();
            errorOutput += msg;
            console.error('Python error:', msg);

            // stderr'daki INFO logları da parse et (Python logger stderr'a yazıyor)
            // THY Hero check-in patterns
            if (msg.includes('🚀 Chrome driver kuruluyor')) {
              console.log('✅ PATTERN MATCH: Chrome driver kuruluyor');
              progressCallback(65, 'THY - Tarayıcı başlatılıyor...');
            }
            if (msg.includes('✅ Chrome driver hazır')) {
              progressCallback(68, 'THY - Tarayıcı hazır...');
            }
            if (msg.includes('🌐 THY ana sayfasına gidiliyor')) {
              progressCallback(70, 'THY - Ana sayfaya bağlanılıyor...');
            }
            if (msg.includes('📍 Ana sayfa yüklendi')) {
              progressCallback(72, 'THY - Ana sayfa yüklendi...');
            }
            if (msg.includes('🎯 Hero bölümünde') || msg.includes('Hero section')) {
              progressCallback(75, 'THY - Hero bölümü aranıyor...');
            }
            if (msg.includes('✅ Hero check-in tab\'ına tıklandı') || msg.includes('Check-in tab\'a tıklandı')) {
              progressCallback(78, 'THY - Check-in sekmesine geçildi...');
            }
            if (msg.includes('📝 Hero check-in formu dolduruluyor') || msg.includes('PNR') || msg.includes('Lastname')) {
              progressCallback(80, 'THY - Rezervasyon bilgileri giriliyor...');
            }
            if (msg.includes('Check-in butonu tıklandı') || msg.includes('Yolcu sayfasına gidiliyor')) {
              progressCallback(83, 'THY - Check-in işlemi başlatılıyor...');
            }
            if (msg.includes('Yolcu seçimi') || msg.includes('seçildi') || msg.includes('passengers selected')) {
              progressCallback(85, 'THY - Yolcu seçimi yapılıyor...');
            }
            if (msg.includes('Bagaj') || msg.includes('bagaj onayı') || msg.includes('Devam butonu')) {
              progressCallback(87, 'THY - Bagaj sayfası işleniyor...');
            }
            if (msg.includes('🪑') || msg.includes('Koltuk') || msg.includes('seat')) {
              progressCallback(90, 'THY - Koltuk seçimi yapılıyor...');
            }
            if (msg.includes('✅ Otomatik ücretsiz koltuk seçimi başarılı')) {
              progressCallback(92, 'THY - Koltuk seçildi...');
            }
            if (msg.includes('Ek Hizmetler') || msg.includes('Extra Services')) {
              progressCallback(94, 'THY - Ek hizmetler sayfası...');
            }
            if (msg.includes('✅ Biniş kartı başarıyla indirildi')) {
              progressCallback(97, 'THY - Biniş kartı indirildi...');
            }
            if (msg.includes('✅ SMS ile biniş kartı başarıyla gönderildi')) {
              progressCallback(98, 'THY - SMS gönderildi...');
            }
            if (msg.includes('✅ Biniş kartı işlemleri tamamlandı') || msg.includes('SMS ve/veya biniş kartı')) {
              progressCallback(100, '✅ THY Check-in tamamlandı!');
            }

            // Pegasus check-in patterns
            if (msg.includes('✅ Browser setup successful') || msg.includes('stealth mode')) {
              progressCallback(65, 'Pegasus - Tarayıcı başlatılıyor...');
            }
            if (msg.includes('🏠 First visiting main page') || msg.includes('Cloudflare')) {
              progressCallback(70, 'Pegasus - Cloudflare bypass...');
            }
            if (msg.includes('✅ Main page loaded') || msg.includes('Successfully accessed check-in page')) {
              progressCallback(73, 'Pegasus - Check-in sayfasına bağlanıldı');
            }
            if (msg.includes('🔍 Looking for CHECK IN') || msg.includes('BİNİŞ KARTLARI')) {
              progressCallback(75, 'Pegasus - Check-in butonu aranıyor...');
            }
            if (msg.includes('Found CHECK IN button') || msg.includes('Clicking CHECK-IN button')) {
              progressCallback(78, 'Pegasus - Check-in butonu tıklanıyor...');
            }
            if (msg.includes('👥 Looking for passenger') || msg.includes('Selected') && msg.includes('passengers')) {
              progressCallback(80, 'Pegasus - Yolcu seçiliyor...');
            }
            if (msg.includes('DEVAM button') || msg.includes('Clicking first DEVAM')) {
              progressCallback(83, 'Pegasus - DEVAM adımı...');
            }
            if (msg.includes('🪑 Reached seat selection') || msg.includes('BAGAJ')) {
              progressCallback(86, 'Pegasus - Koltuk sayfasında...');
            }
            if (msg.includes('pop-up') || msg.includes('DEVAM ET')) {
              progressCallback(88, 'Pegasus - Pop-up işleniyor...');
            }
            if (msg.includes('🧳 Looking for baggage') || msg.includes('Baggage')) {
              progressCallback(90, 'Pegasus - Bagaj sayfasında...');
            }
            if (msg.includes('ONAYLA VE CHECK-IN') || msg.includes('payment')) {
              progressCallback(93, 'Pegasus - Onay sayfasında...');
            }
            if (msg.includes('success') && msg.includes('URL')) {
              progressCallback(95, 'Pegasus - Success sayfasına ulaşıldı...');
            }
            if (msg.includes('BİNİŞ KARTLARI button') || msg.includes('Looking for email')) {
              progressCallback(97, 'Pegasus - Email gönderimi için hazırlanıyor...');
            }
            if (msg.includes('Email entered') || msg.includes('bilet@milaturizm.com')) {
              progressCallback(98, 'Pegasus - Email adresi girildi...');
            }
            if (msg.includes('E-Posta Gönderildi') || msg.includes('✅ CHECK-IN SUCCESSFUL')) {
              progressCallback(100, '✅ Pegasus check-in tamamlandı!');
            }

            // Hata mesajları - frontend'e göster
            if (msg.includes('❌') || msg.includes('ERROR') || msg.includes('hatası') || msg.includes('başarısız') || msg.includes('FAILED')) {
              // Hata detayını da gönder
              const errorLine = msg.split('\n').find(line =>
                line.includes('❌') || line.includes('ERROR') || line.includes('hatası')
              );
              if (errorLine) {
                progressCallback(50, `❌ Hata: ${errorLine.substring(0, 100)}`);
              } else {
                progressCallback(50, '❌ Hata oluştu, lütfen logları kontrol edin...');
              }
            }
          });

          pythonProcess.on('close', (code) => {
            // Try to parse JSON result from output
            let jsonResult = null;
            const jsonMatch = output.match(/JSON_RESULT_START:(.+):JSON_RESULT_END/);
            if (jsonMatch && jsonMatch[1]) {
              try {
                jsonResult = JSON.parse(jsonMatch[1]);
                console.log('📦 Parsed JSON result:', jsonResult);
              } catch (e) {
                console.error('❌ Failed to parse JSON result:', e);
              }
            }

            // Başarı kontrolü: exit code=0 VE başarı mesajları var mı?
            const hasSuccessMessage = output.includes('CHECK-IN TAMAMLANDI') ||
                                     output.includes('BİNİŞ KARTI GÖNDERME İŞLEMİ TAMAMLANDI') ||
                                     output.includes('✅ CHECK-IN SUCCESSFUL') ||
                                     output.includes('Biniş kartı işlemleri tamamlandı') ||
                                     (jsonResult && jsonResult.success === true);

            if (code === 0 && hasSuccessMessage) {
              if (jsonResult) {
                // Use JSON result if available
                resolve({
                  success: true,
                  message: jsonResult.message || 'Check-in başarılı!',
                  flightInfo: {
                    pnr: data.pnr,
                    airline: airline,
                    emailSent: jsonResult.email_sent || false,
                    emailAddress: jsonResult.email_address || null,
                    passengerCount: jsonResult.passenger_count || 1
                  },
                  output: output,
                  jsonResult: jsonResult
                });
              } else {
                // Fallback to old THY logic
                const boardingPassPath = path.join(__dirname, '../../main_scripts/thy/boarding_passes');
                const expectedFileName = `${data.pnr.toLowerCase()}-${data.lastName.toUpperCase()}.pdf`;
                const boardingPassFile = path.join(boardingPassPath, expectedFileName);

                const smsStatus = output.includes('SMS ile biniş kartı başarıyla gönderildi') ||
                                 output.includes('✅ SMS gönderildi');

                resolve({
                  success: true,
                  message: 'Check-in başarılı! Biniş kartı hazırlandı.',
                  flightInfo: {
                    pnr: data.pnr,
                    airline: 'THY',
                    boardingPassFile: boardingPassFile,
                    smsStatus: smsStatus ? 'Gönderildi' : 'SMS gönderilemedi'
                  },
                  output: output
                });
              }
            } else {
              // Hata - exit code != 0 VEYA başarı mesajı yok
              const errorMessage = jsonResult?.error ||
                                  (errorOutput.includes('no such window') ? 'Chrome tarayıcı hatası' : null) ||
                                  (errorOutput.includes('NoSuchWindowException') ? 'Tarayıcı penceresi kapandı' : null) ||
                                  'Check-in başarısız oldu.';

              resolve({
                success: false,
                message: errorMessage,
                error: errorOutput || output,
                output: output,
                jsonResult: jsonResult
              });
            }
          });

      } catch (configError) {
        reject(configError);
      }

    } catch (error) {
      reject(error);
    }
  });
}

// POST /api/checkin - Tek check-in işlemi
router.post('/', async (req, res) => {
  const { pnr, lastName, firstName, phone, airline, seatSelection, email, flightDate, flightTime } = req.body;

  // Validation
  if (!pnr || !lastName) {
    return res.status(400).json({
      error: 'PNR ve soyisim zorunludur'
    });
  }

  const jobId = `checkin_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

  try {
    console.log(`🎯 New check-in request: ${pnr}/${lastName} (Job: ${jobId})`);
    console.log(`📅 Flight Date: ${flightDate}, Flight Time: ${flightTime}`);

    // PRODUCTION MODE: Uçuş bilgisi sorgulamayı skip et (frontend'den gelecek)
    let flightInfo = null;
    console.log(`📋 Production mode: Uçuş bilgisi sorgulaması atlandı - frontend verisi kullanılacak`);

    /* DEVELOPMENT MODE: Uçuş bilgilerini getir (AJET/PC için) - ŞİMDİLİK KAPALI
    if (airline === 'AJET' || airline === 'AJet' || airline === 'PC' || airline === 'PEGASUS') {
      try {
        console.log(`🔍 Uçuş bilgileri sorgulanıyor: ${pnr}/${lastName}`);

        const flightInfoResponse = await fetch('http://localhost:3002/api/checkin/flight-info', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ pnr, lastName, airline })
        });

        if (flightInfoResponse.ok) {
          const flightData = await flightInfoResponse.json();
          if (flightData.success) {
            flightInfo = flightData.flightInfo;
            console.log(`✈️ Uçuş bilgileri alındı: ${flightInfo.origin} → ${flightInfo.destination}`);

            if (!flightInfo.checkinAvailable) {
              console.log(`⛔ Check-in henüz açık değil veya kapandı.`);
              return res.status(400).json({
                success: false,
                error: 'Check-in şu anda müsait değil',
                status: 'waiting'
              });
            }
          } else {
            return res.status(400).json({
              success: false,
              error: 'Uçuş bilgisi alınamadı',
              status: 'failed'
            });
          }
        }
      } catch (error) {
        console.error(`❌ Uçuş bilgisi sorgulama hatası: ${error.message}`);
      }
    }
    */

    // Job başlatıldığını bildiren response
    res.json({
      success: true,
      message: 'Check-in işlemi başlatıldı',
      jobId,
      status: 'processing',
      flightInfo: flightInfo // Frontend'e uçuş bilgilerini de gönder
    });

    // Real-time updates için WebSocket
    const io = req.io;
    
    // Client'a başlangıç mesajı gönder
    io.emit('checkin_status', {
      jobId,
      status: 'processing',
      message: 'Check-in işlemi başlatılıyor...',
      progress: 10
    });

    // Selenium service başlat
    const seleniumService = new SeleniumService();
    activeServices.set(jobId, seleniumService);

    // Progress updates - hem WebSocket hem de DB'ye yaz
    const sendProgress = async (progress, message, level = 'info') => {
      io.emit('checkin_status', {
        jobId,
        status: 'processing',
        message,
        progress
      });

      // Database'e kaydet
      try {
        await saveLog({
          jobId,
          pnr,
          airline: airline || 'THY',
          percentage: progress,
          level,
          message,
          details: null
        });
      } catch (err) {
        console.error('❌ Log kaydetme hatası:', err);
      }
    };

    // Database'e check-in ticket kaydı oluştur (frontend verisiyle)
    let ticketId = null;
    try {
      // Frontend'den gelen flightDate ve flightTime kullan (YYYY-MM-DD formatında)
      // Eğer tarih boşsa (hemen check-in yap seçilmişse), yarının tarihi kullan
      let departureDate = flightDate;
      if (!departureDate || departureDate.trim() === '') {
        const tomorrow = new Date();
        tomorrow.setDate(tomorrow.getDate() + 1); // Yarın
        departureDate = tomorrow.toISOString().split('T')[0]; // YYYY-MM-DD
      }

      let departureTime = flightTime || '14:00'; // Default öğleden sonra 14:00

      const ticketData = {
        ticketRequestId: null, // Manuel check-in - ticket request yok
        pnr: pnr.toUpperCase(),
        customerId: null, // Manuel check-in - customer yok
        requestedByUserId: null,
        airline: airline || 'THY',
        flightNumber: 'MANUAL', // Frontend'de flight number yok
        departureAirport: 'XXX', // Frontend'de airport bilgisi yok
        arrivalAirport: 'YYY', // Frontend'de airport bilgisi yok
        departureDate: departureDate,
        departureTime: departureTime,
        checkinStatus: 'waiting', // Python çalıştırılmayacak, waiting olarak kaydet
        checkinOpenTime: new Date().toISOString(),
        isAutoCheckinEnabled: false, // Manuel check-in
        notes: `Manuel check-in - ${lastName.toUpperCase()} - ${phone || email || 'No contact'}`
      };

      const ticket = await createCheckinTicket(ticketData);
      ticketId = ticket.id;
      console.log(`💾 Database kaydı oluşturuldu: ${ticketId}`);

      // Yolcu kaydı oluştur (basitleştirilmiş)
      try {
        const firstNameToUse = firstName ? firstName.toUpperCase() : 'YOLCU';
        const lastNameToUse = lastName.toUpperCase();

        // Direkt yeni passenger oluştur
        const passenger = await createPassenger({
          firstName: firstNameToUse,
          lastName: lastNameToUse,
          phone: phone || null,
          email: email || null
        });
        console.log(`👤 Yeni yolcu kaydı oluşturuldu: ${passenger.id} - ${passenger.firstName} ${passenger.lastName}`);

        // Check-in ticket'a bağla
        await addPassengerToCheckin(ticketId, passenger.id, {
          checkinStatus: 'waiting'
        });

        console.log(`🔗 Yolcu check-in ticket'a bağlandı: ${passenger.id}`);
      } catch (passengerError) {
        console.warn(`⚠️ Yolcu kaydı oluşturulamadı/bağlanamadı: ${passengerError.message}`);
        console.error(passengerError);
      }
    } catch (dbError) {
      console.warn(`⚠️ Database kaydı oluşturulamadı (devam ediliyor): ${dbError.message}`);
    }

    // DEVELOPMENT MODE: Python script çalıştır - AKTİF
    sendProgress(25, `${airline} web sitesine bağlanılıyor...`);
    await new Promise(resolve => setTimeout(resolve, 1000));

    sendProgress(50, 'Check-in formu dolduruluyor...');
    await new Promise(resolve => setTimeout(resolve, 1000));

    sendProgress(75, 'Check-in işlemi yapılıyor...');

    const result = await runPythonCheckin({
      pnr: pnr.toUpperCase(),
      lastName: lastName.toUpperCase(),
      phone: phone || '',
      email: email || '',
      airline: airline || 'THY',
      seatSelection: seatSelection || 'with_seat'
    }, sendProgress);

    sendProgress(100, 'İşlem tamamlandı');

    // Database'i güncelle - check-in sonucu kaydet
    if (ticketId) {
      try {
        const updateData = {
          checkinStatus: result.success ? 'completed' : 'failed',
          smsStatus: result.success ? 'sent' : 'failed',
          // emailStatus: result.success ? 'sent' : 'failed', // Database'de kolon yok
          errorMessage: result.success ? null : result.message
        };

        await updateCheckinTicket(ticketId, updateData);
        console.log(`💾 Database güncellendi: ${ticketId} - ${updateData.checkinStatus}`);
      } catch (dbUpdateError) {
        console.error(`❌ Database güncelleme hatası: ${dbUpdateError.message}`);
      }
    }

    // Sonuç mesajı gönder
    io.emit('checkin_result', {
      jobId,
      ticketId,  // Frontend için ticket ID ekle
      success: result.success,
      message: result.message,
      flightInfo: result.flightInfo,
      boardingPassUrl: result.boardingPassUrl,
      screenshots: result.screenshots
    });

    // Selenium service'i temizle
    await seleniumService.close();
    activeServices.delete(jobId);

    console.log(`✅ Check-in completed: ${jobId} - ${result.success ? 'SUCCESS' : 'FAILED'}`);

  } catch (error) {
    console.error(`❌ Check-in error (${jobId}):`, error);

    // Database'i güncelle - hata durumu
    if (ticketId) {
      try {
        await updateCheckinTicket(ticketId, {
          checkinStatus: 'failed',
          smsStatus: 'failed',
          // emailStatus: 'failed', // Database'de kolon yok
          errorMessage: error.message
        });
        console.log(`💾 Database güncellendi (hata): ${ticketId} - failed`);
      } catch (dbUpdateError) {
        console.error(`❌ Database güncelleme hatası: ${dbUpdateError.message}`);
      }
    }

    // Hata durumunda client'a bildir
    const io = req.io;
    io.emit('checkin_result', {
      jobId,
      ticketId,
      success: false,
      message: `Check-in sırasında hata oluştu: ${error.message}`,
      error: error.message
    });

    // Cleanup
    const service = activeServices.get(jobId);
    if (service) {
      await service.close();
      activeServices.delete(jobId);
    }
  }
});

// POST /api/checkin/bulk - Toplu check-in işlemi  
router.post('/bulk', async (req, res) => {
  const { checkinList } = req.body;
  
  if (!Array.isArray(checkinList) || checkinList.length === 0) {
    return res.status(400).json({
      error: 'Check-in listesi gerekli'
    });
  }

  const batchId = `batch_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
  
  res.json({
    success: true,
    message: 'Toplu check-in işlemi başlatıldı',
    batchId,
    totalCount: checkinList.length
  });

  // WebSocket ile progress takibi
  const io = req.io;
  let completedCount = 0;
  const results = [];

  for (const checkinData of checkinList) {
    try {
      const seleniumService = new SeleniumService();
      const result = await seleniumService.performCheckin(checkinData);
      
      completedCount++;
      results.push({ ...checkinData, result });

      // Progress update
      io.emit('bulk_checkin_progress', {
        batchId,
        completedCount,
        totalCount: checkinList.length,
        progress: (completedCount / checkinList.length) * 100,
        lastResult: result
      });

      await seleniumService.close();
      
      // Rate limiting - 2 saniye bekle
      if (completedCount < checkinList.length) {
        await new Promise(resolve => setTimeout(resolve, 2000));
      }

    } catch (error) {
      console.error('Bulk check-in item error:', error);
      completedCount++;
      results.push({ 
        ...checkinData, 
        result: { 
          success: false, 
          message: error.message 
        } 
      });
    }
  }

  // Final result
  io.emit('bulk_checkin_complete', {
    batchId,
    results,
    summary: {
      total: checkinList.length,
      successful: results.filter(r => r.result.success).length,
      failed: results.filter(r => !r.result.success).length
    }
  });
});

// GET /api/checkin/status/:jobId - Job durumu sorgulama
router.get('/status/:jobId', (req, res) => {
  const { jobId } = req.params;
  const service = activeServices.get(jobId);
  
  res.json({
    jobId,
    isRunning: !!service,
    status: service ? 'processing' : 'completed'
  });
});

// DELETE /api/checkin/:jobId - İşlemi iptal et
router.delete('/:jobId', async (req, res) => {
  const { jobId } = req.params;
  const service = activeServices.get(jobId);

  if (service) {
    await service.close();
    activeServices.delete(jobId);

    res.json({
      success: true,
      message: 'Check-in işlemi iptal edildi'
    });
  } else {
    // Aktif service bulunamadı ama yine de başarılı sayalım
    console.log(`⏹️ Check-in durdurma isteği: ${jobId} (aktif service yok)`);

    res.json({
      success: true,
      message: 'Check-in işlemi durduruldu'
    });
  }
});

// API endpoint: Check-in loglarını getir
router.get('/logs/:jobId', async (req, res) => {
  try {
    const { jobId } = req.params;
    const logs = await getLogs(jobId);

    res.json({
      success: true,
      logs
    });
  } catch (error) {
    console.error('❌ Error fetching logs:', error);
    res.status(500).json({
      success: false,
      error: 'Loglar getirilemedi'
    });
  }
});

// API endpoint: Uçuş bilgilerini getir (PNR ve soyisim ile)
router.post('/flight-info', async (req, res) => {
  try {
    const { pnr, lastName, airline } = req.body;

    if (!pnr || !lastName) {
      return res.status(400).json({
        success: false,
        error: 'PNR ve soyisim zorunludur'
      });
    }

    console.log(`✈️ Uçuş bilgileri sorgulanıyor - Havayolu: ${airline}, PNR: ${pnr}`);

    const airlineCode = airline || 'AJET';
    let pythonPath, pythonVenv, scriptDir;

    // Havayoluna göre script seçimi
    if (airlineCode === 'AJET' || airlineCode === 'AJet') {
      scriptDir = '/Users/sametburaksari/CascadeProjects/vTravel/vtravel-backend/main_scripts/ajet';
      pythonPath = `${scriptDir}/ajet_get_flight_info.py`;
      pythonVenv = `${scriptDir}/../venv/bin/python`;
    } else {
      // Şimdilik sadece AJET destekleniyor
      return res.status(400).json({
        success: false,
        error: `${airlineCode} için uçuş bilgisi sorgulama henüz desteklenmiyor`
      });
    }

    // Python scriptini çalıştır (flight-info script'i headless desteklemiyor)
    const pythonProcess = spawn(pythonVenv, [pythonPath, pnr, lastName, '--json'], {
      cwd: scriptDir
    });

    let output = '';
    let errorOutput = '';
    let responseSent = false; // Response kontrolü için flag

    pythonProcess.stdout.on('data', (data) => {
      output += data.toString();
    });

    pythonProcess.stderr.on('data', (data) => {
      errorOutput += data.toString();
      console.error('Python stderr:', data.toString());
    });

    pythonProcess.on('close', (code) => {
      if (responseSent) return; // Eğer response gönderildiyse çık

      if (code === 0) {
        try {
          // JSON output'u parse et
          const result = JSON.parse(output);

          if (result.success && result.flights && result.flights.length > 0) {
            const flight = result.flights[0]; // İlk uçuşu al

            // Check-in açık mı kontrolü - uçuş tarih ve saatine göre
            let checkinAvailable = false;
            let checkinMessage = '';
            let hoursUntilFlight = null;

            if (flight.date && flight.departureTime) {
              try {
                // Türkçe ay isimlerini İngilizce'ye çevir
                const monthMap = {
                  'Ocak': 'January', 'Oca': 'Jan',
                  'Şubat': 'February', 'Şub': 'Feb',
                  'Mart': 'March', 'Mar': 'Mar',
                  'Nisan': 'April', 'Nis': 'Apr',
                  'Mayıs': 'May', 'May': 'May',
                  'Haziran': 'June', 'Haz': 'Jun',
                  'Temmuz': 'July', 'Tem': 'Jul',
                  'Ağustos': 'August', 'Ağu': 'Aug',
                  'Eylül': 'September', 'Eyl': 'Sep',
                  'Ekim': 'October', 'Eki': 'Oct',
                  'Kasım': 'November', 'Kas': 'Nov',
                  'Aralık': 'December', 'Ara': 'Dec'
                };

                let dateStr = flight.date;
                for (const [tr, en] of Object.entries(monthMap)) {
                  dateStr = dateStr.replace(tr, en);
                }

                // "11 October 2025" + "10:00" → Date object
                const [day, month, year] = dateStr.split(' ');
                const [hours, minutes] = flight.departureTime.split(':');
                const flightDateTime = new Date(`${month} ${day}, ${year} ${hours}:${minutes}:00`);

                const now = new Date();
                const diffMs = flightDateTime - now;
                const diffHours = diffMs / (1000 * 60 * 60);
                hoursUntilFlight = Math.round(diffHours * 10) / 10; // 1 ondalık basamak

                // AJET check-in kuralları: 24 saat önce açılır, 2 saat önce kapanır
                if (diffHours <= 24 && diffHours >= 2) {
                  checkinAvailable = true;
                  checkinMessage = `Check-in açık! Uçuşa ${hoursUntilFlight} saat var.`;
                } else if (diffHours > 24) {
                  checkinAvailable = false;
                  checkinMessage = `Check-in henüz açılmadı. Uçuşa ${hoursUntilFlight} saat var. (24 saat öncesine kadar bekleyecek)`;
                } else if (diffHours < 2) {
                  checkinAvailable = false;
                  checkinMessage = `Check-in kapandı. Uçuşa ${hoursUntilFlight} saat kaldı.`;
                }
              } catch (dateError) {
                console.error('Tarih hesaplama hatası:', dateError);
                checkinMessage = 'Tarih bilgisi işlenemedi, manuel kontrol gerekli';
              }
            }

            responseSent = true;
            res.json({
              success: true,
              flightInfo: {
                pnr: result.pnr,
                airline: airlineCode,
                date: flight.date || null,
                departureTime: flight.departureTime || null,
                arrivalTime: flight.arrivalTime || null,
                origin: flight.origin || null,
                destination: flight.destination || null,
                originCity: flight.originCity || null,
                destinationCity: flight.destinationCity || null,
                flightNumber: flight.flightNumber || null,
                dayOfWeek: flight.dayOfWeek || null,
                checkinAvailable: checkinAvailable,
                checkinMessage: checkinMessage,
                hoursUntilFlight: hoursUntilFlight
              }
            });
            console.log(`✅ Uçuş bilgileri alındı: ${flight.origin} → ${flight.destination} | Check-in: ${checkinAvailable ? 'AÇIK' : 'KAPALI'}`);
          } else {
            responseSent = true;
            res.json({
              success: false,
              error: result.error || 'Uçuş bilgisi bulunamadı'
            });
          }
        } catch (parseError) {
          console.error('JSON parse hatası:', parseError);
          responseSent = true;
          res.status(500).json({
            success: false,
            error: 'Uçuş bilgisi işlenemedi',
            details: output
          });
        }
      } else {
        console.error(`Python script hata kodu: ${code}`);
        responseSent = true;
        res.status(500).json({
          success: false,
          error: 'Uçuş bilgisi alınamadı',
          details: errorOutput
        });
      }
    });

    pythonProcess.on('error', (error) => {
      if (responseSent) return; // Eğer response gönderildiyse çık

      console.error('Python process hatası:', error);
      responseSent = true;
      res.status(500).json({
        success: false,
        error: 'Script çalıştırılamadı'
      });
    });

  } catch (error) {
    console.error('❌ Flight info error:', error);
    res.status(500).json({
      success: false,
      error: 'Sunucu hatası'
    });
  }
});

// Cleanup on server shutdown
process.on('SIGTERM', async () => {
  console.log('🛑 Cleaning up active Selenium services...');
  for (const [jobId, service] of activeServices) {
    await service.close();
    console.log(`Closed service: ${jobId}`);
  }
});

module.exports = router;