const express = require('express');
const router = express.Router();
const {
  listCheckinTickets,
  getCheckinTicket,
  createCheckinTicket,
  updateCheckinTicket,
  deleteCheckinTicket,
  updateCheckinStatus,
  getUpcomingCheckins,
  listCheckinTicketsWithTicketInfo,
  getLogs
} = require('../services/database-mysql');

// GET /api/checkin-tickets - List all check-in tickets
router.get('/', async (req, res) => {
  try {
    const {
      ticketRequestId,
      status,
      airline,
      scheduledDate,
      search,
      limit,
      offset,
      withTicketInfo
    } = req.query;

    const filters = {};
    if (ticketRequestId) filters.ticketRequestId = ticketRequestId;
    if (status) filters.status = status;
    if (airline) filters.airline = airline;
    if (scheduledDate) filters.scheduledDate = scheduledDate;
    if (search) filters.search = search;

    const pagination = {
      limit: Math.min(parseInt(limit) || 20, 100),
      offset: parseInt(offset) || 0
    };

    // Ticket bilgileriyle birlikte getir (HER ZAMAN yolcu bilgisi dahil)
    const result = await listCheckinTicketsWithTicketInfo(filters, pagination);

    res.json({
      success: true,
      data: result.data,
      total: result.total,
      limit: pagination.limit,
      offset: pagination.offset
    });
  } catch (error) {
    console.error('Error listing check-in tickets:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// GET /api/checkin-tickets/upcoming - Get upcoming check-ins (next 24 hours)
router.get('/upcoming', async (req, res) => {
  try {
    const checkins = await getUpcomingCheckins();
    res.json({
      success: true,
      data: checkins
    });
  } catch (error) {
    console.error('Error getting upcoming check-ins:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// GET /api/checkin-tickets/:id - Get single check-in ticket
router.get('/:id', async (req, res) => {
  try {
    const checkin = await getCheckinTicket(req.params.id);

    if (!checkin) {
      return res.status(404).json({
        success: false,
        error: 'Check-in ticket not found'
      });
    }

    // Logları da getir (ticket_request_id ile)
    let logs = [];
    if (checkin.ticketRequestId) {
      try {
        logs = await getLogs(checkin.ticketRequestId);
      } catch (logError) {
        console.warn(`⚠️ Loglar alınamadı: ${logError.message}`);
      }
    }

    res.json({
      success: true,
      data: {
        ...checkin,
        logs: logs || []
      }
    });
  } catch (error) {
    console.error('Error getting check-in ticket:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// POST /api/checkin-tickets - Create new check-in ticket
router.post('/', async (req, res) => {
  try {
    const {
      ticketRequestId,
      pnr,
      lastName,
      phone,
      airline,
      scheduledTime,
      autoCheckin
    } = req.body;

    // Validation
    if (!ticketRequestId && !pnr) {
      return res.status(400).json({
        success: false,
        error: 'Either ticketRequestId or PNR is required'
      });
    }

    if (!lastName) {
      return res.status(400).json({
        success: false,
        error: 'lastName is required'
      });
    }

    const checkinData = {
      ticketRequestId,
      pnr: pnr?.toUpperCase(),
      lastName: lastName.toUpperCase(),
      phone,
      airline: airline || 'THY',
      scheduledTime,
      autoCheckin: autoCheckin !== false,
      status: 'scheduled',
      smsStatus: 'pending'
    };

    const checkin = await createCheckinTicket(checkinData);

    // Emit real-time update
    if (global.io) {
      global.io.emit('checkin_ticket_created', { checkin });
    }

    res.status(201).json({
      success: true,
      data: checkin
    });
  } catch (error) {
    console.error('Error creating check-in ticket:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// PUT /api/checkin-tickets/:id - Update check-in ticket
router.put('/:id', async (req, res) => {
  try {
    const checkin = await updateCheckinTicket(req.params.id, req.body);

    if (!checkin) {
      return res.status(404).json({
        success: false,
        error: 'Check-in ticket not found'
      });
    }

    // Emit real-time update
    if (global.io) {
      global.io.emit('checkin_ticket_updated', { checkin });
    }

    res.json({
      success: true,
      data: checkin
    });
  } catch (error) {
    console.error('Error updating check-in ticket:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// PUT /api/checkin-tickets/:id/status - Update check-in status
router.put('/:id/status', async (req, res) => {
  try {
    const { status, boardingPassUrl, smsStatus, error } = req.body;

    if (!status) {
      return res.status(400).json({
        success: false,
        error: 'status is required'
      });
    }

    const checkin = await updateCheckinStatus(req.params.id, {
      status,
      boardingPassUrl,
      smsStatus,
      error
    });

    if (!checkin) {
      return res.status(404).json({
        success: false,
        error: 'Check-in ticket not found'
      });
    }

    // Emit real-time update
    if (global.io) {
      global.io.emit('checkin_status_updated', { checkin });
    }

    res.json({
      success: true,
      data: checkin,
      message: 'Check-in status updated successfully'
    });
  } catch (error) {
    console.error('Error updating check-in status:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// DELETE /api/checkin-tickets/:id - Delete check-in ticket
router.delete('/:id', async (req, res) => {
  try {
    const result = await deleteCheckinTicket(req.params.id);

    if (!result) {
      return res.status(404).json({
        success: false,
        error: 'Check-in ticket not found'
      });
    }

    // Emit real-time update
    if (global.io) {
      global.io.emit('checkin_ticket_deleted', { id: req.params.id });
    }

    res.json({
      success: true,
      message: 'Check-in ticket deleted successfully'
    });
  } catch (error) {
    console.error('Error deleting check-in ticket:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

module.exports = router;
