-- Migration: Create accommodation, visa, and transfer request tables
-- Date: 2025-10-21
-- Description: Add support for accommodation, visa, and transfer service requests

-- ============================================
-- ACCOMMODATION REQUESTS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS accommodation_requests (
  id VARCHAR(36) PRIMARY KEY,

  -- Customer Information
  customer_id VARCHAR(36),
  customer_type ENUM('individual', 'corporate') NOT NULL,
  customer_name VARCHAR(255) NOT NULL,

  -- Request Details
  request_number VARCHAR(50) UNIQUE NOT NULL,
  status ENUM('pending', 'quoted', 'approved', 'booked', 'confirmed', 'cancelled', 'completed') DEFAULT 'pending',
  priority ENUM('normal', 'high', 'urgent') DEFAULT 'normal',

  -- Accommodation Details
  destination_city VARCHAR(255) NOT NULL,
  destination_country VARCHAR(255) NOT NULL,
  check_in_date DATE NOT NULL,
  check_out_date DATE NOT NULL,
  nights INT NOT NULL,

  -- Guest Information
  number_of_rooms INT NOT NULL DEFAULT 1,
  number_of_adults INT NOT NULL DEFAULT 1,
  number_of_children INT NOT NULL DEFAULT 0,
  room_type VARCHAR(255),
  board_type ENUM('room_only', 'breakfast', 'half_board', 'full_board', 'all_inclusive'),

  -- Hotel Information (filled after booking)
  hotel_name VARCHAR(255),
  hotel_address TEXT,
  hotel_phone VARCHAR(50),
  hotel_email VARCHAR(255),
  hotel_rating DECIMAL(2,1),
  hotel_confirmation_number VARCHAR(100),

  -- Pricing
  quoted_price DECIMAL(10,2),
  final_price DECIMAL(10,2),
  currency VARCHAR(10) DEFAULT 'TRY',

  -- Special Requests
  special_requests TEXT,
  meal_preferences TEXT,
  accessibility_needs TEXT,

  -- Assignment & Processing
  assigned_to VARCHAR(36),
  assigned_department VARCHAR(36),
  processed_by VARCHAR(36),
  processed_at TIMESTAMP NULL,

  -- Notes & Communication
  internal_notes TEXT,
  customer_notes TEXT,

  -- Meta
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

  INDEX idx_accommodation_customer (customer_id),
  INDEX idx_accommodation_status (status),
  INDEX idx_accommodation_dates (check_in_date, check_out_date),
  INDEX idx_accommodation_assigned (assigned_to),
  INDEX idx_accommodation_request_number (request_number),
  FOREIGN KEY (customer_id) REFERENCES customers(id) ON DELETE SET NULL,
  FOREIGN KEY (assigned_to) REFERENCES agency_users(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- VISA REQUESTS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS visa_requests (
  id VARCHAR(36) PRIMARY KEY,

  -- Customer Information
  customer_id VARCHAR(36),
  customer_type ENUM('individual', 'corporate') NOT NULL,
  customer_name VARCHAR(255) NOT NULL,

  -- Request Details
  request_number VARCHAR(50) UNIQUE NOT NULL,
  status ENUM('pending', 'documents_received', 'in_process', 'submitted', 'approved', 'rejected', 'cancelled', 'delivered') DEFAULT 'pending',
  priority ENUM('normal', 'high', 'urgent') DEFAULT 'normal',

  -- Visa Details
  destination_country VARCHAR(255) NOT NULL,
  visa_type ENUM('tourist', 'business', 'student', 'work', 'transit', 'other') NOT NULL,
  purpose_of_visit TEXT,
  planned_entry_date DATE,
  planned_exit_date DATE,
  duration_of_stay INT,

  -- Applicant Information
  applicant_first_name VARCHAR(255) NOT NULL,
  applicant_last_name VARCHAR(255) NOT NULL,
  applicant_passport_number VARCHAR(50) NOT NULL,
  applicant_passport_expiry DATE NOT NULL,
  applicant_nationality VARCHAR(100) NOT NULL,
  applicant_date_of_birth DATE NOT NULL,
  applicant_phone VARCHAR(50),
  applicant_email VARCHAR(255),

  -- Application Details
  application_number VARCHAR(100),
  appointment_date TIMESTAMP NULL,
  appointment_location VARCHAR(255),
  submitted_date DATE,
  decision_date DATE,
  visa_validity_start DATE,
  visa_validity_end DATE,
  number_of_entries ENUM('single', 'multiple'),

  -- Documents
  documents_required JSON,
  documents_received JSON,
  documents_complete TINYINT(1) DEFAULT 0,

  -- Pricing
  service_fee DECIMAL(10,2),
  visa_fee DECIMAL(10,2),
  total_amount DECIMAL(10,2),
  currency VARCHAR(10) DEFAULT 'TRY',
  payment_status ENUM('unpaid', 'partial', 'paid') DEFAULT 'unpaid',

  -- Assignment & Processing
  assigned_to VARCHAR(36),
  assigned_department VARCHAR(36),
  processed_by VARCHAR(36),
  approved_by VARCHAR(36),
  approved_at TIMESTAMP NULL,

  -- Notes & Communication
  internal_notes TEXT,
  customer_notes TEXT,
  rejection_reason TEXT,

  -- Meta
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

  INDEX idx_visa_customer (customer_id),
  INDEX idx_visa_status (status),
  INDEX idx_visa_passport (applicant_passport_number),
  INDEX idx_visa_assigned (assigned_to),
  INDEX idx_visa_request_number (request_number),
  INDEX idx_visa_appointment (appointment_date),
  FOREIGN KEY (customer_id) REFERENCES customers(id) ON DELETE SET NULL,
  FOREIGN KEY (assigned_to) REFERENCES agency_users(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TRANSFER REQUESTS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS transfer_requests (
  id VARCHAR(36) PRIMARY KEY,

  -- Customer Information
  customer_id VARCHAR(36),
  customer_type ENUM('individual', 'corporate') NOT NULL,
  customer_name VARCHAR(255) NOT NULL,

  -- Request Details
  request_number VARCHAR(50) UNIQUE NOT NULL,
  status ENUM('pending', 'assigned', 'confirmed', 'in_progress', 'completed', 'cancelled', 'no_show') DEFAULT 'pending',
  priority ENUM('normal', 'high', 'urgent') DEFAULT 'normal',
  transfer_type ENUM('airport_pickup', 'airport_dropoff', 'city_transfer', 'hourly', 'point_to_point') NOT NULL,

  -- Transfer Details
  pickup_date TIMESTAMP NOT NULL,
  pickup_location VARCHAR(255) NOT NULL,
  pickup_address TEXT,
  dropoff_location VARCHAR(255) NOT NULL,
  dropoff_address TEXT,

  -- Flight Information (for airport transfers)
  flight_number VARCHAR(50),
  airline VARCHAR(100),
  terminal VARCHAR(50),
  arrival_time TIMESTAMP NULL,
  departure_time TIMESTAMP NULL,

  -- Passenger Information
  number_of_passengers INT NOT NULL DEFAULT 1,
  number_of_luggage INT DEFAULT 0,
  passenger_name VARCHAR(255) NOT NULL,
  passenger_phone VARCHAR(50) NOT NULL,
  passenger_email VARCHAR(255),

  -- Vehicle Details
  vehicle_type ENUM('sedan', 'minivan', 'minibus', 'bus', 'vip', 'luxury') NOT NULL,
  vehicle_preferences TEXT,
  special_equipment JSON,

  -- Driver Assignment
  driver_name VARCHAR(255),
  driver_phone VARCHAR(50),
  driver_vehicle_plate VARCHAR(50),
  assigned_at TIMESTAMP NULL,

  -- Pricing
  quoted_price DECIMAL(10,2),
  final_price DECIMAL(10,2),
  currency VARCHAR(10) DEFAULT 'TRY',
  payment_status ENUM('unpaid', 'paid', 'refunded') DEFAULT 'unpaid',

  -- Status Tracking
  driver_en_route_at TIMESTAMP NULL,
  pickup_completed_at TIMESTAMP NULL,
  dropoff_completed_at TIMESTAMP NULL,
  actual_pickup_time TIMESTAMP NULL,
  actual_dropoff_time TIMESTAMP NULL,

  -- Assignment & Processing
  assigned_to VARCHAR(36),
  assigned_department VARCHAR(36),
  processed_by VARCHAR(36),

  -- Notes & Communication
  internal_notes TEXT,
  customer_notes TEXT,
  driver_notes TEXT,
  special_requests TEXT,

  -- Meta
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

  INDEX idx_transfer_customer (customer_id),
  INDEX idx_transfer_status (status),
  INDEX idx_transfer_pickup (pickup_date),
  INDEX idx_transfer_assigned (assigned_to),
  INDEX idx_transfer_request_number (request_number),
  INDEX idx_transfer_driver (driver_phone),
  FOREIGN KEY (customer_id) REFERENCES customers(id) ON DELETE SET NULL,
  FOREIGN KEY (assigned_to) REFERENCES agency_users(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- SAMPLE DATA FOR TESTING
-- ============================================

-- Sample Accommodation Request
INSERT INTO accommodation_requests (id, customer_id, customer_type, customer_name, request_number, status, destination_city, destination_country, check_in_date, check_out_date, nights, number_of_rooms, number_of_adults, number_of_children, special_requests)
VALUES (UUID(), NULL, 'individual', 'Ahmet Yılmaz', 'ACC-2025-001', 'pending', 'Antalya', 'Türkiye', '2025-11-01', '2025-11-04', 3, 1, 2, 1, 'Deniz manzaralı oda talep ediyorum');

-- Sample Visa Request
INSERT INTO visa_requests (id, customer_id, customer_type, customer_name, request_number, status, destination_country, visa_type, applicant_first_name, applicant_last_name, applicant_passport_number, applicant_passport_expiry, applicant_nationality, applicant_date_of_birth, purpose_of_visit)
VALUES (UUID(), NULL, 'individual', 'Ayşe Kaya', 'VISA-2025-001', 'pending', 'ABD', 'tourist', 'Ayşe', 'Kaya', 'U12345678', '2028-06-15', 'Türkiye', '1990-05-20', 'Tatil amaçlı ziyaret');

-- Sample Transfer Request
INSERT INTO transfer_requests (id, customer_id, customer_type, customer_name, request_number, status, transfer_type, pickup_date, pickup_location, dropoff_location, number_of_passengers, passenger_name, passenger_phone, vehicle_type)
VALUES (UUID(), NULL, 'individual', 'Mehmet Demir', 'TRF-2025-001', 'pending', 'airport_pickup', '2025-10-25 14:30:00', 'İstanbul Havalimanı', 'Taksim Meydanı', 2, 'Mehmet Demir', '+90 555 123 4567', 'sedan');
