// Load environment variables
require('dotenv').config();

const express = require('express');
const cors = require('cors');
const { createServer } = require('http');
const { Server } = require('socket.io');

// Import routes
const authRoutes = require('./routes/auth');
const checkinRoutes = require('./routes/checkin');
const checkinTicketsRoutes = require('./routes/checkin-tickets-db'); // Database-backed checkin tickets
const customerRoutes = require('./routes/customers');
const ticketRoutes = require('./routes/tickets-db'); // Database-backed tickets route
const ticketRequestRoutes = require('./routes/ticket-requests');
const flightRoutes = require('./routes/flights');
const accommodationRequestsRoutes = require('./routes/accommodation-requests');
const visaRequestsRoutes = require('./routes/visa-requests');
const transferRequestsRoutes = require('./routes/transfer-requests');
const reportsRoutes = require('./routes/reports');
const departmentsRoutes = require('./routes/departments');
const agencyUsersRoutes = require('./routes/agency-users');

// Import optimized services
const WorkerSchedulerService = require('./services/WorkerSchedulerService');
const { timeoutWrapper } = require('./services/TimeoutWrapper');

const app = express();
const server = createServer(app);
const io = new Server(server, {
  cors: {
    origin: ["http://localhost:3000", "http://localhost:3001", "http://localhost:3002"],
    methods: ["GET", "POST"]
  }
});

const PORT = process.env.PORT || 3002;

// Middleware
app.use(cors({
  origin: ["http://localhost:3000", "http://localhost:3001", "http://localhost:3002"],
  credentials: true
}));
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Request timeout middleware for API endpoints
app.use((req, res, next) => {
  // Set request timeout to prevent blocking
  req.setTimeout(30000); // 30 seconds max per request
  res.setTimeout(30000);

  next();
});

// Socket.io for real-time updates
io.on('connection', (socket) => {
  console.log('Client connected:', socket.id);

  socket.on('disconnect', () => {
    console.log('Client disconnected:', socket.id);
  });
});

// Initialize optimized scheduler with error handling
let scheduler;
try {
  scheduler = new WorkerSchedulerService();
  console.log('✅ Optimized worker scheduler initialized');
} catch (error) {
  console.error('❌ Failed to initialize scheduler:', error);
  process.exit(1);
}

// Make services available globally and to routes
global.io = io;
global.scheduler = scheduler;
global.timeoutWrapper = timeoutWrapper;

app.use((req, res, next) => {
  req.io = io;
  req.scheduler = scheduler;
  req.timeout = timeoutWrapper;
  next();
});

// API Routes with timeout protection
app.use('/api/auth', timeoutMiddleware(authRoutes));
app.use('/api/checkin', timeoutMiddleware(checkinRoutes));
app.use('/api/checkin-tickets', timeoutMiddleware(checkinTicketsRoutes));
app.use('/api/customers', timeoutMiddleware(customerRoutes));
app.use('/api/tickets', timeoutMiddleware(ticketRoutes));
app.use('/api/ticket-requests', timeoutMiddleware(ticketRequestRoutes));
app.use('/api/accommodation-requests', timeoutMiddleware(accommodationRequestsRoutes));
app.use('/api/visa-requests', timeoutMiddleware(visaRequestsRoutes));
app.use('/api/transfer-requests', timeoutMiddleware(transferRequestsRoutes));
app.use('/api/flights', timeoutMiddleware(flightRoutes));
app.use('/api/reports', timeoutMiddleware(reportsRoutes));
app.use('/api/departments', timeoutMiddleware(departmentsRoutes));
app.use('/api/agency-users', timeoutMiddleware(agencyUsersRoutes));

// Health check with scheduler status
app.get('/api/health', async (req, res) => {
  try {
    const schedulerStatus = scheduler.getStatus();
    const memoryUsage = process.memoryUsage();

    res.json({
      status: 'OK',
      message: 'vTravel API Server Running (Optimized)',
      timestamp: new Date().toISOString(),
      scheduler: {
        isRunning: schedulerStatus.isRunning,
        activeWorkers: schedulerStatus.stats.activeWorkers,
        totalJobs: schedulerStatus.stats.totalJobs,
        queuedJobs: schedulerStatus.stats.queuedJobs,
        processingJobs: schedulerStatus.stats.processingJobs
      },
      performance: {
        uptime: process.uptime(),
        memoryUsage: {
          heapUsed: Math.round(memoryUsage.heapUsed / 1024 / 1024) + 'MB',
          heapTotal: Math.round(memoryUsage.heapTotal / 1024 / 1024) + 'MB',
          external: Math.round(memoryUsage.external / 1024 / 1024) + 'MB'
        }
      }
    });
  } catch (error) {
    res.status(500).json({
      status: 'ERROR',
      message: error.message,
      timestamp: new Date().toISOString()
    });
  }
});

// Performance metrics endpoint
app.get('/api/metrics', async (req, res) => {
  try {
    const schedulerStatus = scheduler.getStatus();

    res.json({
      scheduler: schedulerStatus,
      system: {
        nodeVersion: process.version,
        platform: process.platform,
        arch: process.arch,
        uptime: process.uptime(),
        memoryUsage: process.memoryUsage(),
        cpuUsage: process.cpuUsage()
      },
      performance: {
        avgResponseTime: '< 100ms', // Would be calculated from metrics
        activeSessions: io.engine.clientsCount,
        errorRate: '< 1%' // Would be calculated from error tracking
      }
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
});

// Timeout middleware wrapper
function timeoutMiddleware(router) {
  return (req, res, next) => {
    // Wrap route handlers with timeout
    const originalSend = res.send;
    const originalJson = res.json;

    let timeoutId = setTimeout(() => {
      if (!res.headersSent) {
        res.status(408).json({
          error: 'Request timeout',
          message: 'Operation took too long to complete'
        });
      }
    }, 25000); // 25 second timeout for API routes

    // Clear timeout when response is sent
    res.send = function(...args) {
      clearTimeout(timeoutId);
      return originalSend.apply(this, args);
    };

    res.json = function(...args) {
      clearTimeout(timeoutId);
      return originalJson.apply(this, args);
    };

    router(req, res, next);
  };
}

// Enhanced error handling middleware
app.use((err, req, res, next) => {
  console.error('API Error:', {
    message: err.message,
    stack: err.stack,
    url: req.url,
    method: req.method,
    timestamp: new Date().toISOString()
  });

  if (res.headersSent) {
    return next(err);
  }

  // Determine error type and respond appropriately
  if (err.message && err.message.includes('timeout')) {
    res.status(408).json({
      error: 'Request Timeout',
      message: 'Operation timed out, please try again'
    });
  } else if (err.name === 'ValidationError') {
    res.status(400).json({
      error: 'Validation Error',
      message: err.message
    });
  } else {
    res.status(500).json({
      error: 'Internal Server Error',
      message: process.env.NODE_ENV === 'development' ? err.message : 'An unexpected error occurred'
    });
  }
});

// 404 handler
app.use((req, res) => {
  res.status(404).json({ error: 'API endpoint not found' });
});

// Graceful shutdown handling
process.on('SIGTERM', gracefulShutdown);
process.on('SIGINT', gracefulShutdown);

async function gracefulShutdown(signal) {
  console.log(`🛑 Received ${signal}, starting graceful shutdown...`);

  // Stop accepting new connections
  server.close(() => {
    console.log('✅ HTTP server closed');
  });

  // Shutdown scheduler and cleanup workers
  try {
    await scheduler.shutdown();
    console.log('✅ Scheduler shutdown complete');
  } catch (error) {
    console.error('❌ Scheduler shutdown error:', error);
  }

  // Close WebSocket connections
  io.close(() => {
    console.log('✅ WebSocket server closed');
  });

  // Force exit after 10 seconds if graceful shutdown fails
  setTimeout(() => {
    console.log('⚠️ Forcing exit after timeout');
    process.exit(1);
  }, 10000);

  process.exit(0);
}

// Start server
server.listen(PORT, () => {
  console.log(`🚀 vTravel API Server (Optimized) running on port ${PORT}`);
  console.log(`📡 WebSocket server ready for real-time updates`);
  console.log(`🤖 Worker-based scheduler active with ${scheduler.maxConcurrentWorkers} max workers`);

  // Performance monitoring
  setInterval(() => {
    const memUsage = process.memoryUsage();
    const heapUsedMB = Math.round(memUsage.heapUsed / 1024 / 1024);

    if (heapUsedMB > 200) {
      console.warn(`⚠️ High memory usage: ${heapUsedMB}MB`);
    }
  }, 60000); // Check every minute
});

module.exports = { app, server, scheduler };
