const express = require('express');
const cors = require('cors');
const { createServer } = require('http');
const { Server } = require('socket.io');
const path = require('path');

// Import routes
const checkinRoutes = require('./routes/checkin');
const customerRoutes = require('./routes/customer');
const ticketRoutes = require('./routes/tickets');
const flightRoutes = require('./routes/flights');

// Import services
const SchedulerService = require('./services/SchedulerService');

const app = express();
const server = createServer(app);
const io = new Server(server, {
  cors: {
    origin: ["http://localhost:3000", "http://localhost:3001"],
    methods: ["GET", "POST"]
  }
});

const PORT = process.env.PORT || 3003;

// Middleware
app.use(cors({
  origin: ["http://localhost:3000", "http://localhost:3001"],
  credentials: true
}));
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// Socket.io for real-time updates
io.on('connection', (socket) => {
  console.log('Client connected:', socket.id);
  
  socket.on('disconnect', () => {
    console.log('Client disconnected:', socket.id);
  });
});

// Initialize scheduler
const scheduler = new SchedulerService();

// Make io and scheduler available globally and to routes
global.io = io;
global.scheduler = scheduler;

app.use((req, res, next) => {
  req.io = io;
  req.scheduler = scheduler;
  next();
});

// API Routes
app.use('/api/checkin', checkinRoutes);
app.use('/api/customers', customerRoutes);
app.use('/api/tickets', ticketRoutes);
app.use('/api/flights', flightRoutes);

// Health check
app.get('/api/health', (req, res) => {
  res.json({ 
    status: 'OK', 
    message: 'vTravel API Server Running',
    timestamp: new Date().toISOString()
  });
});

// Error handling middleware
app.use((err, req, res, next) => {
  console.error(err.stack);
  res.status(500).json({ 
    error: 'Internal Server Error',
    message: err.message 
  });
});

// 404 handler
app.use((req, res) => {
  res.status(404).json({ error: 'API endpoint not found' });
});

server.listen(PORT, () => {
  console.log(`🚀 vTravel API Server running on port ${PORT}`);
  console.log(`📡 WebSocket server ready for real-time updates`);
});